/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <time.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif

#include <gtk/gtk.h>
#include <netdude/nd.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_clipboard.h>
#include <netdude/nd_capture.h>
#include <netdude/nd_globals.h>
#include <netdude/nd_packet.h>
#include <netdude/nd_trace_registry.h>
#include <netdude/nd_protocol.h>
#include <netdude/nd_protocol_inst.h>
#include <netdude/nd_prefs.h>
#include <netdude/nd_tcpdump.h>
#include <netdude/nd_misc.h>
#include <callbacks.h>
#include <interface.h>
#include <support.h>

#define STATUSBAR_TIMEOUT            5
#define GTK_CLIST_CLASS_FW(_widget_) GTK_CLIST_CLASS (((GtkObject*) (_widget_))->klass)
#define PBAR_ACTIVITY_BLOCKS         10


static GtkProgressBar *pbar;
static int             pbar_total;
static int             pbar_current;
static gboolean        activity_mode;
static int             timeout_set;
static gfloat          old_ratio = -1.0;

static GdkPixmap      *incomplete_pmap;
static GdkBitmap      *incomplete_mask;

static guint       timestamp_id;
static int         timestamp_set = FALSE;

static GdkColor bg[5];
static GdkColor red[5];
static GdkColor yellow[5];

static GtkTooltips* tt;

static GList          *monowidth_widgets = NULL;

void    
nd_gui_init(void)
{
  int         i;
  GtkWidget  *win;
  GtkWidget  *w;
  GtkStyle   *gs;

  /* No traces are loaded yet -- hide the traces notebook. */
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "traces_notebook");
  gtk_widget_hide(w);

  /* Initialize the global progress bar pointer */
  pbar = gtk_object_get_data(GTK_OBJECT(toplevel), "progressbar");
  D_ASSERT_PTR(pbar);
  gtk_progress_bar_set_activity_blocks(pbar, PBAR_ACTIVITY_BLOCKS);

  /* Pixmap and Color stuff below */
  gs = gtk_widget_get_style(toplevel);
  win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_realize(win);
  incomplete_pmap = gdk_pixmap_create_from_xpm(win->window, &incomplete_mask,
					       &gs->bg[GTK_STATE_NORMAL],
					       PACKAGE_DATA_DIR"/pixmaps/incomplete.xpm");

  /* Init red and yellow: */
  for (i = 0; i < 5; i++)
    {
      bg[i] = gs->bg[i];
      red[i] = gs->bg[i];
      red[i].red = (2 * red[i].red) > 65535 ? 65535 : 2 * red[i].red;
      if (red[i].red == 0)
	red[i].red = 32768;
      red[i].green /=  2;
      red[i].blue /= 2;
    }

  for (i = 0; i < 5; i++)
    {
      yellow[i] = gs->bg[i];
      yellow[i].green = (2 * yellow[i].green) > 65535 ? 65535 : 2 * yellow[i].green;
      yellow[i].red = (2 * yellow[i].red) > 65535 ? 65535 : 2 * yellow[i].red;

      if (yellow[i].green == 0)
	yellow[i].green = 32768;

      if (yellow[i].red == 0)
	yellow[i].red = 32768;

      yellow[i].blue /=  2;
    }

  /* Create tooltips, if necessary. */
  if (! (tt = gtk_object_get_data(GTK_OBJECT(toplevel), "tooltips")))
    {
      tt = gtk_tooltips_new();
      gtk_object_set_data (GTK_OBJECT (toplevel), "tooltips", tt);
      gtk_tooltips_enable(tt);
    }
}


static void      
gui_sync_edit_menu(GtkWidget *menu)
{
  ND_Trace  *trace;
  GtkWidget *button;

  D_ASSERT_PTR(menu);

  if (nd_clipboard_occupied())
    {
      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_paste");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
    }
  else
    {
      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_paste");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
    }

  trace = nd_trace_registry_get_current();

  if (trace)
    {
      ND_Packet *current = nd_trace_get_current_packet(trace);

      if (current)
	{
	  button = gtk_object_get_data(GTK_OBJECT(menu), "packet_delete");
	  D_ASSERT_PTR(button);
	  gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
	  button = gtk_object_get_data(GTK_OBJECT(menu), "packet_copy");
	  D_ASSERT_PTR(button);
	  gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
	  button = gtk_object_get_data(GTK_OBJECT(menu), "packet_cut");
	  D_ASSERT_PTR(button);
	  gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
	  
	  if (current->is_hidden)
	    {
	      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_hide");
	      D_ASSERT_PTR(button);
	      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
	      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_show");
	      D_ASSERT_PTR(button);
	      gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
	    }
	  else
	    {
	      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_show");
	      D_ASSERT_PTR(button);
	      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
	      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_hide");
	      D_ASSERT_PTR(button);
	      gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
	    }
	}

      if (nd_trace_sel_size(trace) > 0)
	{
	  button = gtk_object_get_data(GTK_OBJECT(toplevel), "packet_unselect_all");
	  D_ASSERT_PTR(button);
	  gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
	}

      button = gtk_object_get_data(GTK_OBJECT(toplevel), "packet_select_all");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
    }
  else
    {
      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_hide");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_show");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_delete");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_copy");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
      button = gtk_object_get_data(GTK_OBJECT(menu), "packet_cut");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

      button = gtk_object_get_data(GTK_OBJECT(toplevel), "packet_select_all");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

      button = gtk_object_get_data(GTK_OBJECT(toplevel), "packet_unselect_all");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
    }
}


static void      
gui_sync_file_menu(void)
{
  GtkWidget *menu;
  GtkWidget *button;

  menu = gtk_object_get_data(GTK_OBJECT(toplevel), "file_menu");
  D_ASSERT_PTR(menu);

  if (!nd_trace_registry_get_current())
    {
      button = gtk_object_get_data(GTK_OBJECT(toplevel), "save");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

      button = gtk_object_get_data(GTK_OBJECT(toplevel), "save_as");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
    }
  else
    {
      button = gtk_object_get_data(GTK_OBJECT(toplevel), "save");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);

      button = gtk_object_get_data(GTK_OBJECT(toplevel), "save_as");
      D_ASSERT_PTR(button);
      gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
    }

  button = gtk_object_get_data(GTK_OBJECT(toplevel), "capture");
  D_ASSERT_PTR(button);
  gtk_widget_set_sensitive(GTK_WIDGET(button), nd_capture_possible());
}


void
nd_gui_sync(void)
{
  gui_sync_edit_menu(toplevel);
  gui_sync_file_menu();
}


void         
nd_gui_show_packet_menu(GdkEventButton *event)
{
  GtkWidget *menu;

  /* Let's grab the edit menu from the main window: */
  menu = gtk_object_get_data(GTK_OBJECT(toplevel), "edit_menu");

  /* Update its settings -- everything is registered with the
     toplevel window: */
  gui_sync_edit_menu(toplevel);

  /* And show the menu :) */
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		 event->button, event->time);
}


void      
nd_gtk_clist_select_all(GtkCList *clist)
{
  int i;
  GtkCListRow *clist_row;
  GList *row_list;
  
  D_ASSERT_PTR(clist);

  if (!clist)
    return;

  row_list = clist->row_list;

  for (i = 0; row_list; i++)
    {
      clist_row = row_list->data;
      clist_row->state = GTK_STATE_SELECTED;
      if (gtk_clist_row_is_visible (clist, i) != GTK_VISIBILITY_NONE)
	GTK_CLIST_CLASS_FW(clist)->draw_row (clist, NULL, i, clist_row);
      if (!clist->selection)
	{
	  clist->selection = g_list_append (clist->selection, GINT_TO_POINTER(i));
	  clist->selection_end = clist->selection;
	}
      else
	clist->selection_end = 
	  g_list_append (clist->selection_end, GINT_TO_POINTER(i))->next;
      row_list = row_list->next;
    }

  /* XXX need to properly emit a signal here? */
  on_trace_list_select_all(clist, NULL);
}


void
nd_gtk_clist_unselect_all(GtkCList *clist)
{
  int i;
  GtkCListRow *clist_row;
  GList *row_list;
  
  D_ASSERT_PTR(clist);
  if (!clist)
    return;

  row_list = clist->row_list;

  for (i = 0; row_list; i++)
    {
      clist_row = row_list->data;
      if (clist_row->state == GTK_STATE_SELECTED)
	{
	  clist_row->state = GTK_STATE_NORMAL;
	  if (gtk_clist_row_is_visible (clist, i) != GTK_VISIBILITY_NONE)
	    GTK_CLIST_CLASS_FW(clist)->draw_row (clist, NULL, i, clist_row);
	}
      row_list = row_list->next;
    }

  g_list_free(clist->selection);
  clist->selection = NULL;
  clist->selection_end = NULL;

  /* XXX need to properly emit a signal here? */
  on_trace_list_unselect_all(clist, NULL);
}



void    
nd_gui_num_packets_set(void)
{
  ND_Trace        *trace;
  char             s[MAXPATHLEN];
  GtkWidget       *w;
  int              size;

  trace = nd_trace_registry_get_current();
  size = nd_trace_size(trace);

  if (!trace)
    s[0] = '\0';
  else if (size == 1)
    g_snprintf(s, MAXPATHLEN, _("%i packet."), size);
  else
    g_snprintf(s, MAXPATHLEN, _("%i packets."), size);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "num_packets_label");
  D_ASSERT_PTR(w);
  gtk_label_set_text(GTK_LABEL(w), s);
}


void         
nd_gui_modified_set(ND_Trace *trace)
{
  GtkWidget *button;
  GtkWidget *pixmap_red, *pixmap_green;

  if (!trace)
    return;

  button = gtk_object_get_data(GTK_OBJECT(trace->tab), "close_button");
  D_ASSERT_PTR(button);

  pixmap_green = gtk_object_get_data(GTK_OBJECT(trace->tab), "pixmap_green");
  pixmap_red = gtk_object_get_data(GTK_OBJECT(trace->tab), "pixmap_red");

  if (pixmap_green->parent)
    gtk_container_remove(GTK_CONTAINER(button), pixmap_green);

  if (pixmap_red->parent)
    gtk_container_remove(GTK_CONTAINER(button), pixmap_red);

  if (trace->dirty)
    gtk_container_add(GTK_CONTAINER(button), pixmap_red);
  else
    gtk_container_add(GTK_CONTAINER(button), pixmap_green);
}


void      
nd_gui_statusbar_set(const char *text)
{
  static guint timeout_id;
  GtkLabel *l;

  if (!text)
    return;

  l = gtk_object_get_data(GTK_OBJECT(toplevel), "statuslabel");
  D_ASSERT_PTR(l);
  gtk_label_set_text(l, text);
  while (gtk_events_pending())
    gtk_main_iteration();

  if (timeout_set)
    gtk_timeout_remove(timeout_id);

  timeout_id = gtk_timeout_add(STATUSBAR_TIMEOUT * 1000, nd_gui_statusbar_clear, NULL);
  timeout_set = TRUE;
}


gint
nd_gui_statusbar_clear(gpointer data)
{
  GtkLabel *l;

  l = gtk_object_get_data(GTK_OBJECT(toplevel), "statuslabel");
  D_ASSERT_PTR(l);
  gtk_label_set_text(l, "");

  timeout_set = 0;

  return 0;
  data = NULL;
}


void      
nd_gui_pbar_reset(int num_steps)
{
  ND_Trace *trace;
  pbar_current = 0;
  
  if (num_steps > 0)
    pbar_total = num_steps;
  else
    {
      if ( (trace = nd_trace_registry_get_current()))
	{
	  if (trace->apply_to_all)
	    pbar_total = nd_trace_size(trace);
	  else
	    pbar_total = nd_trace_sel_size(trace);
	}
    }

  gtk_progress_bar_update(pbar, 0.0);
  while (gtk_events_pending())
    gtk_main_iteration();
}


void      
nd_gui_pbar_inc(void)
{
  gfloat ratio;

  pbar_current++;

  if ( (ratio = (gfloat)pbar_current/(gfloat)pbar_total) > old_ratio + 0.01)
    {
      gtk_progress_bar_update(pbar, MIN(ratio, 1.0));
      while (gtk_events_pending())
	gtk_main_iteration();
      old_ratio = ratio;
    }
}


void      
nd_gui_pbar_clear(void)
{
  gtk_progress_bar_update(pbar, 0.0);
  old_ratio = -1.0;

  while (gtk_events_pending())
    gtk_main_iteration();
}

gint 
nd_gui_pbar_timeout(gpointer data)
{
  static guint step = 0;

  if (step > PBAR_ACTIVITY_BLOCKS)
    step = 0;

  gtk_progress_set_value(GTK_PROGRESS(pbar), step++);

  while (gtk_events_pending())
    gtk_main_iteration();

  if (activity_mode)
    return (TRUE); /* Let's get called again */

  nd_gui_pbar_clear();

  return (FALSE);
  data = NULL;
}


void      
nd_gui_pbar_start_activity(void)
{
  gtk_progress_set_activity_mode(GTK_PROGRESS(pbar), TRUE);
  activity_mode  = TRUE;
  gtk_timeout_add(100, nd_gui_pbar_timeout, NULL);
}


void      
nd_gui_pbar_stop_activity(void)
{
  gtk_progress_set_activity_mode(GTK_PROGRESS(pbar), FALSE);
  activity_mode = FALSE;
}


void      
nd_gui_list_incomplete_column_visible(GtkWidget *clist, gboolean visible)
{
  D_ASSERT_PTR(clist);

  if (!clist)
    return;

  gtk_clist_set_column_visibility(GTK_CLIST(clist), 1, visible);
}


void      
nd_gui_list_incomplete_row_set(GtkWidget *clist, int row, gboolean incomplete)
{
  D_ASSERT_PTR(clist);

  if (!clist)
    return;

  if (incomplete)
    gtk_clist_set_pixmap(GTK_CLIST(clist), row, 1, incomplete_pmap, incomplete_mask);
  else
    gtk_clist_set_text(GTK_CLIST(clist), row, 1, "");
}


void      
nd_gui_windowtitle_set(const char *filename)
{
  char  s[MAXPATHLEN];
  char *result = (char *) filename;
  int   show_full_path;

  if (!filename)
    return;

  nd_prefs_get_int_item(ND_DOM_NETDUDE, "show_full_path", &show_full_path);
  
  if (!show_full_path)
    result = g_basename(filename);
  
  g_snprintf(s, MAXPATHLEN, "Netdude: %s", result);
  gtk_window_set_title(GTK_WINDOW(toplevel), s);
}


static gint
gui_show_timestamp_timeout(gpointer data)
{
  GtkWidget *w;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "timestamp_win");
  D_ASSERT_PTR(w);
  gtk_widget_show(w);

  return (0);
  data = NULL;
}


void         
nd_gui_timestamp_schedule(GtkCList *clist,
			  GdkEventMotion *event)
{
  static GtkWidget *tooltip = NULL;

  ND_Packet *packet;
  ND_Trace  *trace;
  char       ts_text[MAXPATHLEN];
  int        row;
  int        pref_show_ts, pref_show_ts_abs;
  float      pref_ts_delay; 
  GtkWidget *ts_label;


  /* We won't show timestamps if any buttons are pressed
     while moving around ... */
  
  if ((event->state & (GDK_BUTTON1_MASK |
		       GDK_BUTTON2_MASK |
		       GDK_BUTTON3_MASK |
		       GDK_BUTTON4_MASK |
		       GDK_BUTTON5_MASK)) != 0)
    return;

  row = (event->y - clist->voffset) / (clist->row_height + 1);

  trace  = nd_trace_registry_get_current();
  D_ASSERT_PTR(trace);
  packet = nd_trace_packet_get_nth(trace, row);

  if (!packet)
    return;

  pref_show_ts = TRUE;
  pref_show_ts_abs = FALSE;
  pref_ts_delay = 0.5; 

  nd_prefs_get_int_item(ND_DOM_NETDUDE, "show_timestamps", &pref_show_ts);
  nd_prefs_get_int_item(ND_DOM_NETDUDE, "show_timestamps_absolute", &pref_show_ts_abs);
  nd_prefs_get_flt_item(ND_DOM_NETDUDE, "timestamps_delay", &pref_ts_delay);

  if (!pref_show_ts)
    return;

  if (tooltip)
    gtk_widget_hide(tooltip);

  if (timestamp_set && pref_ts_delay > 0.1)
    gtk_timeout_remove(timestamp_id);
  
  if (packet->prev && !pref_show_ts_abs)
    {
      struct timeval delta_last, delta_abs;
      
      TV_SUB(&packet->ph.ts, &packet->prev->ph.ts, &delta_last);
      TV_SUB(&packet->ph.ts, &trace->pl->ph.ts, &delta_abs);
      
      g_snprintf(ts_text, MAXPATHLEN, "%li.%06lis, %li.%06lis",
		 delta_abs.tv_sec, delta_abs.tv_usec,
		 delta_last.tv_sec, delta_last.tv_usec);
    }
  else
    {
      int offset;
      time_t tt = (time_t)packet->ph.ts.tv_sec;
      
      g_snprintf(ts_text, MAXPATHLEN, "%s", ctime(&tt));
      offset = strlen(ts_text)-1;
      g_snprintf(ts_text + offset, MAXPATHLEN, " +  %li usec",
		 packet->ph.ts.tv_usec);
    }
  
  /* If the packet is only partly captured, show
     how many bytes are missing: */
  if (packet->ph.len > packet->ph.caplen)
    {
      int offset = strlen(ts_text);
      
      g_snprintf(ts_text + offset, MAXPATHLEN - offset,
		 _(", %i bytes missing"), 
		 packet->ph.len - packet->ph.caplen);
    }

  if (!tooltip)
    {
      tooltip = create_timestamp_window();
      gtk_window_set_position(GTK_WINDOW(tooltip), GTK_WIN_POS_NONE);
      gtk_object_set_data(GTK_OBJECT(toplevel), "timestamp_win", tooltip);
    }
    
  ts_label = gtk_object_get_data(GTK_OBJECT(tooltip), "timestamp_label");
  D_ASSERT_PTR(ts_label);
  gtk_label_set_text(GTK_LABEL(ts_label), ts_text);
  
  /* HELP! What's the correct way to ensure that w->allocation.width has the correct
     size now? I haven't been able to figure it out. Since our window's content is
     simple (just the line of text), let's use the width of the text string as
     an approximation.
  */
  
  gtk_widget_realize(tooltip);
  gtk_window_reposition(GTK_WINDOW(tooltip),
			event->x_root - gdk_text_width(ts_label->style->font, ts_text, strlen(ts_text))/2,
			event->y_root + 15);      
  
  timestamp_id = gtk_timeout_add(pref_ts_delay * 1000,
				 gui_show_timestamp_timeout, NULL);
  timestamp_set = TRUE;
}


void        
nd_gui_timestamp_hide(void)
{
  GtkWidget *tooltip;

  if (timestamp_set)
    {
      gtk_timeout_remove(timestamp_id);
      timestamp_set = FALSE;
    }

  tooltip = gtk_object_get_data(GTK_OBJECT(toplevel), "timestamp_win");
  if (tooltip)
    gtk_widget_hide(tooltip);
}


static gint
gui_selection_descend_sort(gconstpointer a, gconstpointer b)
{
  if (GPOINTER_TO_UINT(a) < GPOINTER_TO_UINT(b))
    return 1;
  if (GPOINTER_TO_UINT(a) > GPOINTER_TO_UINT(b))
    return -1;

  return 0;
}


void      
nd_gui_list_remove_selected_rows(GtkWidget *clist)
{  
  GList     *l, *l2 = NULL, *l3 = NULL;

  D_ASSERT_PTR(clist);

  if (!clist)
    return;

  gtk_signal_handler_block_by_func(GTK_OBJECT(clist), on_trace_list_unselect_row, NULL);
  gtk_clist_freeze(GTK_CLIST(clist));

  if (GTK_CLIST(clist)->selection)
    {      
      for (l = GTK_CLIST(clist)->selection; l; l = l->next)
	l2 = g_list_append(l2, l->data);
      
      l2 = g_list_sort(l2, gui_selection_descend_sort);
      
      for (l3 = l2 ; l3; l3 = l3->next)
	gtk_clist_remove(GTK_CLIST(clist), GPOINTER_TO_UINT(l3->data));
    }

  gtk_clist_thaw(GTK_CLIST(clist));
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(clist), on_trace_list_unselect_row, NULL);
  
  if (l2)
    g_list_free(l2);
}


void
nd_gui_list_remove_row(GtkWidget *clist, guint index)
{
  D_ASSERT_PTR(clist);

  if (!clist)
    return;

  gtk_signal_handler_block_by_func(GTK_OBJECT(clist), on_trace_list_unselect_row, NULL);
  gtk_clist_freeze(GTK_CLIST(clist));
  gtk_clist_remove(GTK_CLIST(clist), index);
  gtk_clist_thaw(GTK_CLIST(clist));
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(clist), on_trace_list_unselect_row, NULL);
}


void    
nd_gui_list_update(ND_Trace *trace, gboolean find_context)
{
  ND_Packet *p;
  char       line[MAXPATHLEN];
  char      *s[2];
  int        i;

  D_ENTER;

  if (!trace)
    D_RETURN;;

  D_ASSERT_PTR(trace->list);

  gtk_clist_freeze(GTK_CLIST(trace->list));
  gtk_clist_clear(GTK_CLIST(trace->list));    
  gtk_clist_set_reorderable(GTK_CLIST(trace->list), 1);

  nd_gui_statusbar_set(_("Updating tcpdump output..."));
  nd_gui_pbar_reset(trace->num_packets);

  s[0] = line;
  s[1] = "";

  for (p = trace->pl, i = 0; p; p = p->next, i++)
    {
      nd_tcpdump_get_packet_line(p, line, find_context);
      
      gtk_clist_append(GTK_CLIST(trace->list), s);
      
      if (trace->incomplete)
	{
	  if (nd_packet_is_complete(p))
	    nd_gui_list_incomplete_row_set(trace->list, i, FALSE);
	  else
	    nd_gui_list_incomplete_row_set(trace->list, i, TRUE);
	}

      nd_gui_pbar_inc();
    }

  nd_gui_pbar_clear();    
  gtk_clist_thaw(GTK_CLIST(trace->list));

  D_RETURN;
}


void        
nd_gui_list_update_packet(const ND_Packet *p)
{
  int          index;
  char         line[MAXPATHLEN];

  if (!p)
    return;

  index = nd_packet_get_index(p);

  nd_tcpdump_get_packet_line(p, line, TRUE);
  gtk_clist_set_text(GTK_CLIST(p->trace->list), index, 0, line);

  if (nd_packet_is_complete(p))
    nd_gui_list_incomplete_row_set(p->trace->list, index, FALSE);
  else
    nd_gui_list_incomplete_row_set(p->trace->list, index, TRUE);

  
}


void        
nd_gui_list_update_packet_at_index(const ND_Packet *p, int index)
{
  char         line[MAXPATHLEN];

  if (!p)
    return;

  if (index < 0)
    {
      nd_gui_list_update_packet(p);
      return;
    }    

  D_ASSERT_PTR(p->trace);
  D_ASSERT_PTR(p->trace->list);

  nd_tcpdump_get_packet_line(p, line, TRUE);
  gtk_clist_set_text(GTK_CLIST(p->trace->list), index, 0, line);

  if (nd_packet_is_complete(p))
    nd_gui_list_incomplete_row_set(p->trace->list, index, FALSE);
  else
    nd_gui_list_incomplete_row_set(p->trace->list, index, TRUE);
}


static void
gui_update_line_cb(ND_Packet *packet,
		   ND_ProtoData *pd,
		   void *user_data)
{
  char *line = (char *) user_data;

  if (!packet || !pd)
    return;
  
  if (pd->inst.proto->is_stateful)
    pd->inst.proto->update_tcpdump_line(packet, line);
}


void         
nd_gui_list_update_packet_state(const ND_Packet *packet)
{
  int          index;
  char        *line;
  char         buf[MAXPATHLEN];

  if (!packet)
    return;

  D_ASSERT_PTR(packet->trace);
  D_ASSERT_PTR(packet->trace->list);

  index = nd_packet_get_index(packet);

  gtk_clist_get_text(GTK_CLIST(packet->trace->list), index, 0, &line);
  g_snprintf(buf, MAXPATHLEN, line);
  
  nd_packet_foreach_proto((ND_Packet *) packet,
			     gui_update_line_cb,
			     buf);

  gtk_clist_set_text(GTK_CLIST(packet->trace->list), index, 0, buf);
}


void         
nd_gui_list_update_packet_state_at_index(const ND_Packet *packet, int index)
{
  char        *line = NULL;
  char         buf[MAXPATHLEN];

  if (!packet)
    return;

  if (index < 0)
    {
      nd_gui_list_update_packet_state(packet);
      return;
    }    

  D_ASSERT_PTR(packet->trace);
  D_ASSERT_PTR(packet->trace->list);

  gtk_clist_get_text(GTK_CLIST(packet->trace->list), index, 0, &line);
  if (line)
    {
      g_snprintf(buf, MAXPATHLEN, line);
      
      nd_packet_foreach_proto((ND_Packet *) packet,
				 gui_update_line_cb,
				 buf);
      
      gtk_clist_set_text(GTK_CLIST(packet->trace->list), index, 0, buf);      
    }
}


static GtkWidget   *
gui_new_trace_list(void)
{
  GtkWidget *trace_list;
  gchar     *titles[2] = { N_("Tcpdump log"), N_("State") };

  trace_list = gtk_clist_new_with_titles(2, titles);

  GTK_CLIST_CLASS_FW(GTK_CLIST(trace_list))->select_all = nd_gtk_clist_select_all;
  GTK_CLIST_CLASS_FW(GTK_CLIST(trace_list))->unselect_all = nd_gtk_clist_unselect_all;
  gtk_clist_set_column_justification(GTK_CLIST(trace_list), 1, GTK_JUSTIFY_CENTER);
  gtk_clist_set_column_resizeable(GTK_CLIST(trace_list), 0, FALSE);
  gtk_clist_set_column_visibility(GTK_CLIST(trace_list), 1, FALSE);
  gtk_clist_set_reorderable(GTK_CLIST(trace_list), TRUE);
  gtk_clist_set_use_drag_icons(GTK_CLIST(trace_list), FALSE);

  gtk_widget_show (trace_list);
  gtk_widget_set_events (trace_list, GDK_POINTER_MOTION_MASK | GDK_LEAVE_NOTIFY_MASK);
  gtk_clist_set_column_width (GTK_CLIST (trace_list), 1, 80);
  gtk_clist_set_selection_mode (GTK_CLIST (trace_list), GTK_SELECTION_EXTENDED);
  gtk_clist_column_titles_show (GTK_CLIST (trace_list));
  gtk_clist_set_shadow_type (GTK_CLIST (trace_list), GTK_SHADOW_NONE);

  nd_gui_add_monowidth_widget(trace_list);

  gtk_clist_column_titles_passive(GTK_CLIST(trace_list));

  gtk_signal_connect (GTK_OBJECT (trace_list), "select_row",
                      GTK_SIGNAL_FUNC (on_trace_list_select_row),
                      NULL);
  gtk_signal_connect_after (GTK_OBJECT (trace_list), "row_move",
			    GTK_SIGNAL_FUNC (on_trace_list_row_move),
			    NULL);
  gtk_signal_connect (GTK_OBJECT (trace_list), "button_press_event",
                      GTK_SIGNAL_FUNC (on_trace_list_button_press_event),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (trace_list), "unselect_row",
                      GTK_SIGNAL_FUNC (on_trace_list_unselect_row),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (trace_list), "end_selection",
                      GTK_SIGNAL_FUNC (on_trace_list_end_selection),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (trace_list), "select_all",
                      GTK_SIGNAL_FUNC (on_trace_list_select_all),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (trace_list), "unselect_all",
                      GTK_SIGNAL_FUNC (on_trace_list_unselect_all),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (trace_list), "size_allocate",
                      GTK_SIGNAL_FUNC (on_trace_list_size_allocate),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (trace_list), "motion_notify_event",
                      GTK_SIGNAL_FUNC (on_trace_list_motion_notify_event),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (trace_list), "leave_notify_event",
                      GTK_SIGNAL_FUNC (on_trace_list_leave_notify_event),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (trace_list), "key_press_event",
                      GTK_SIGNAL_FUNC (on_trace_list_key_press_event),
                      NULL);

  return trace_list;
}


static GtkWidget   *
gui_new_trace_proto_notebook(void)
{
  GtkWidget *notebook;

  notebook = gtk_notebook_new ();
  gtk_widget_show (notebook);
  gtk_notebook_set_scrollable(GTK_NOTEBOOK(notebook), TRUE);
  gtk_widget_set_usize (notebook, -2, 180);
  gtk_container_set_border_width (GTK_CONTAINER (notebook), 5);

  return notebook;
}


void         
nd_gui_trace_new_tab(ND_Trace *trace)
{
  GtkWidget *tab_content, *tab_label_content, *scrolledwin;
  GtkWidget *trace_list, *notebook, *label, *button, *pixmap;
  const char *file;

  D_ASSERT_PTR(trace);

  if (!trace)
    return;

  /* Build tab: */

  tab_content = gtk_vpaned_new ();
  gtk_widget_ref (tab_content);
  gtk_widget_show (tab_content);
  /* gtk_paned_set_position (GTK_PANED (tab_content), 200);*/

  scrolledwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_ref (scrolledwin);
  gtk_object_set_data_full (GTK_OBJECT (tab_content), "scrolledwin", scrolledwin,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (scrolledwin);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_paned_add1 (GTK_PANED (tab_content), scrolledwin);

  trace_list = gui_new_trace_list();
  gtk_widget_ref (trace_list);
  gtk_object_set_data_full (GTK_OBJECT (tab_content), "trace_list", trace_list,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_container_add (GTK_CONTAINER (scrolledwin), trace_list);
  /* gtk_paned_add1 (GTK_PANED (tab_content), trace_list); */

  notebook = gui_new_trace_proto_notebook();
  gtk_widget_ref (notebook);
  gtk_object_set_data_full (GTK_OBJECT (tab_content), "notebook", notebook,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_paned_add2 (GTK_PANED (tab_content), notebook);

  gtk_signal_connect (GTK_OBJECT (notebook), "switch_page",
                      GTK_SIGNAL_FUNC (on_protos_notebook_switch_page),
                      notebook);


  gtk_object_set_data(GTK_OBJECT(tab_content), "trace", trace);
  gtk_object_set_data(GTK_OBJECT(notebook), "trace", trace);

  /* Build tab label: */
  if (trace->filename)
    file = g_basename(trace->filename);
  else
    file = trace->unnamed;

  tab_label_content = gtk_hbox_new (FALSE, 5);
  gtk_widget_ref (tab_label_content);
  gtk_object_set_data_full (GTK_OBJECT (tab_content), "tab_label_content",
			    tab_label_content, (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (tab_label_content);

  /* Okay -- I want to have a tooltip with the  full file name on this
     tab. Label's don't work with tooltips, so I have to wrap the
     label in an event box. But if I do that, the tabs don't get
     switched by clicking on the label, and worse, they get drawn
     differently. I'll comment this out for now.

  eb = gtk_event_box_new();
  gtk_widget_show(eb);
  gtk_box_pack_start (GTK_BOX (tab_label_content), eb, TRUE, TRUE, 0);

  tt = gtk_object_get_data(GTK_OBJECT(toplevel), "tooltips");
  gtk_tooltips_set_tip (GTK_TOOLTIPS(tt), eb, trace->filename, NULL);
  */

  label = gtk_label_new (file);
  gtk_widget_ref (label);
  gtk_object_set_data_full (GTK_OBJECT (tab_content), "tab_label", label,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label);
  gtk_container_add(GTK_CONTAINER(tab_label_content), label);
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);

  pixmap = create_pixmap (tab_label_content, "delete.xpm");
  gtk_widget_ref (pixmap);
  gtk_object_set_data_full (GTK_OBJECT (tab_content), "pixmap_red", pixmap,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (pixmap);
  gtk_misc_set_padding (GTK_MISC (pixmap), 0, 0);
  gtk_pixmap_set_build_insensitive (GTK_PIXMAP (pixmap), FALSE);

  pixmap = create_pixmap (tab_label_content, "delete_okay.xpm");
  gtk_widget_ref (pixmap);
  gtk_object_set_data_full (GTK_OBJECT (tab_content), "pixmap_green", pixmap,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (pixmap);
  gtk_misc_set_padding (GTK_MISC (pixmap), 0, 0);
  gtk_pixmap_set_build_insensitive (GTK_PIXMAP (pixmap), FALSE);

  button = gtk_button_new();
  gtk_container_add (GTK_CONTAINER (button), pixmap);
  gtk_widget_ref (button);
  gtk_object_set_data_full (GTK_OBJECT (tab_content), "close_button", button,
			    (GtkDestroyNotify) gtk_widget_unref);
  
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (on_trace_close_clicked),
                      trace);
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (tab_label_content), button, FALSE, FALSE, 0);  

  gtk_paned_set_position(GTK_PANED(tab_content), toplevel->allocation.height - 250);

  /* Hook up results: */

  trace->tab = tab_content;
  trace->tab_label = tab_label_content;
  trace->list = trace_list;
}


void         
nd_gui_trace_name_set(ND_Trace *trace)
{
  const char *sp;
  GtkLabel *label;

  if (!trace)
    return;

  label = gtk_object_get_data(GTK_OBJECT(trace->tab), "tab_label");
  D_ASSERT_PTR(label);

  if (trace->filename)
    sp = g_basename(trace->filename);
  else
    sp = trace->unnamed;
  
  gtk_label_set_text(label, sp);
}


void         
nd_gui_trace_add(ND_Trace *trace)
{
  GtkWidget *notebook;
  GtkWidget *w;

  if (!trace)
    return;

  notebook = gtk_object_get_data(GTK_OBJECT(toplevel), "traces_notebook");
  D_ASSERT_PTR(notebook);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "background_vbox");
  D_ASSERT_PTR(w);

  if (GTK_WIDGET_VISIBLE(w))
    {
      gtk_widget_hide(w);
      gtk_widget_show(notebook);
    }

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), trace->tab, trace->tab_label);

  /* Switch to the  new trace, calls nd_trace_registy_set_current()
     in the callback! */
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 
			gtk_notebook_page_num(GTK_NOTEBOOK(notebook), trace->tab));
}


void         
nd_gui_trace_remove(ND_Trace *trace)
{
  GtkWidget *notebook;
  gint       trace_num;

  if (!trace)
    return;

  notebook = gtk_object_get_data(GTK_OBJECT(toplevel), "traces_notebook");
  D_ASSERT_PTR(notebook);

  trace_num = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), trace->tab);
  gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), trace_num);

  if (gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook)) < 0)
    {
      GtkWidget *w;

      w = gtk_object_get_data(GTK_OBJECT(toplevel), "background_vbox");
      D_ASSERT_PTR(w);
      gtk_widget_show(w);

      gtk_widget_hide(notebook);      
    }

  nd_gui_num_packets_set();
}


void    
nd_gui_widget_set_color(GtkWidget *widget, ND_GuiColor color)
{
  int i;
  GtkRcStyle * rc_style = gtk_rc_style_new();

  switch (color)
    {
    case ND_COLOR_RED:
      for (i=0; i<5; i++)
	{
	  rc_style->bg[i] = red[i];
	  rc_style->color_flags[i] = GTK_RC_BG;
	}
      break;

    case ND_COLOR_YELLOW:
      for (i=0; i<5; i++)
	{
	  rc_style->bg[i] = yellow[i];
	  rc_style->color_flags[i] = GTK_RC_BG;
	}
      break;

    default:
      for (i=0; i<5; i++)
	{
	  rc_style->bg[i] = bg[i];
	  rc_style->color_flags[i] = GTK_RC_BG;
	}
    }
    
  gtk_widget_modify_style(widget, rc_style);
}


static void 
gui_menu_item_activate(GtkMenuItem *menuitem,
		       gpointer user_data)
{
  ND_Trace         *trace;
  ND_MenuEntryCB    callback;
  int value;

  return_if_no_current_trace(trace);

  callback = (ND_MenuEntryCB) gtk_object_get_data(GTK_OBJECT(menuitem), "callback");
  value = GPOINTER_TO_INT(user_data);

  D_ASSERT_PTR(callback);
  if (callback)
    callback(trace->cur_packet,
	     nd_trace_get_current_proto_header(trace),
	     value);
}


static void
gui_add_menu_item(GtkWidget *menu, ND_MenuData *data)
{
  GtkWidget *item;

  item = gtk_menu_item_new_with_label (data->label);
  gtk_widget_ref (item);
  gtk_object_set_data_full (GTK_OBJECT (menu), data->label, item,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (menu), item);

  gtk_object_set_data(GTK_OBJECT(item), "callback", data->callback);

  gtk_tooltips_set_tip(tt, item, data->tooltip, NULL);

  gtk_signal_connect (GTK_OBJECT (item), "activate",
                      GTK_SIGNAL_FUNC (gui_menu_item_activate),
                      GINT_TO_POINTER((gint) data->value));
}



GtkWidget   *
nd_gui_create_menu(ND_MenuData *data)
{
  GtkWidget *menu;
  int        i;

  if (!data)
    return NULL;

  menu = gtk_menu_new ();

  for (i = 0; data[i].label; i++)
    gui_add_menu_item(menu, &data[i]);
  
  return menu;
}


/* Callback dispatcher -- receives callbacks on the Gtk level from
   the protocol's field table, retrieves necessary data and calls
   user's callback. */
static void
gui_proto_field_cb(GtkButton       *button,
		   gpointer         user_data)
{
  ND_Packet        *current;
  ND_Trace         *trace;
  ND_ProtoInfo     *pinf;
  ND_ProtoInst     *pi;
  ND_ProtoField    *field;
  void             *header;
  guint             offset;
  
  return_if_no_current_trace(trace);

  D_ENTER;

  field = (ND_ProtoField *) user_data;
  D_ASSERT_PTR(field);

  pi = nd_trace_get_current_proto_inst(trace);
  D_ASSERT_PTR(pi);

  pinf = nd_trace_get_proto_info(trace, pi->proto, pi->nesting);
  D_ASSERT_PTR(pinf);

  if (! (current = nd_trace_get_current_packet(trace)))
    {
      D(("No currently selected packet!? -- aborting\n"));
      return;
    }

  D_ASSERT((nd_trace_get_current_proto_header(trace) ==
	    nd_packet_get_data(current, pi->proto, pi->nesting)),
	   "Field callback dispatcher uses wrong packet header!");

  header = nd_trace_get_current_proto_header(trace);
  offset = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "offset"));

  nd_gui_proto_table_block_events(trace, pinf);

  if (field->callback)
    {
      D(("Field %s callback on packet %p with offset %i\n",
	 field->label, current, offset));
      field->callback(current, header, ((guchar*) header) + offset);
    }

  nd_gui_proto_table_unblock_events(trace, pinf);
  
  D_RETURN;
  TOUCH(button);
}


/**
 * gui_table_create_button - creates a button widget for a header field
 * @parent: parent widget, for automatic destruction of new button
 * @pinf: the protocol info that provides nesting info and the hash to store the buttons
 * @field: the field for which the buttons is created
 * @label: override label if the one given in @field isn't wanted (used
 * for multi-line buttons)
 * @offset: offset of the new button, stored with the button widget as "offset"
 * @is_error: whether the button is in error state and thus painted red
 * @set_data: whether to store the button in the packet info hashtable
 */
static GtkWidget *
gui_table_create_button(GtkWidget *parent,
			ND_ProtoInfo *pinf,
			ND_ProtoField *field,
			const char *label,
			guint offset,
			gboolean is_error, gboolean set_data)
{
  GtkWidget *button = NULL;
  const char *used_label = field->label;
  
  if (label)
    used_label = label;
  
  switch (field->type)
    {
    case ND_FLG_FIELD:
      button = gtk_toggle_button_new_with_label(used_label);
      break;

    case ND_VAL_FIELD:
    default:
      button = gtk_button_new_with_label(used_label);
    }

  gtk_tooltips_set_tip (GTK_TOOLTIPS(tt), button, field->tooltip, NULL);
  
  /* Arrange for automatic cleanup of the button: */
  gtk_widget_ref(button);
  gtk_widget_show(button);
  gtk_object_set_data_full(GTK_OBJECT (parent), used_label,
			   button, (GtkDestroyNotify) gtk_widget_unref);

  /* Store the fields offset with the button */
  gtk_object_set_data(GTK_OBJECT(button), "offset", GINT_TO_POINTER((offset >> 3)));
      
  /* If wanted, hook up the button in the packet info hash table */
  if (set_data)
    nd_proto_info_set_data(pinf, nd_proto_field_to_string(field), button);
  
  /* If the field provides a callback, hook it in, otherwise set
   * the button insensitive so that it can't be pressed */
  if (field->callback)
    {
      gtk_signal_connect(GTK_OBJECT (button), "clicked",
			 GTK_SIGNAL_FUNC (gui_proto_field_cb),
			 field);
    }
  else
    {
      gtk_widget_set_sensitive(button, FALSE);
    }
  
  /* If something's wrong with it, paint it. */
  nd_gui_widget_set_color(button, (is_error ? ND_COLOR_RED : ND_COLOR_BG));

  return button;
}


GtkWidget *
nd_gui_proto_table_create(ND_Trace *trace,
			  ND_ProtoInfo *pinf)
{
  int          i, x, y, bpl, w_top, w_middle, w_bot, h_middle, largest = 0;
  GtkWidget   *proto_table, *button;
  char        *label;
  guint        offset = 0;
  ND_Protocol *proto;

  if (!trace || !pinf || !pinf->inst.proto->fields)
    return NULL;

  proto = pinf->inst.proto;
  bpl = proto->header_width;

  /* Create the GUI tab for this protocol: ---------------------------- */

  /* Create the table itself: */

  proto_table = gtk_table_new (proto->header_size / bpl, bpl, TRUE);
  gtk_widget_show (proto_table);
  gtk_container_set_border_width (GTK_CONTAINER (proto_table), 5);
  
/*   This is experimental stuff that could become a bit # indicator */
/*   at some point. Right now it sucks, so it's commented out.  */
/*
  {
    char s[MAXPATHLEN];
    GtkWidget *bitnum;
    for (i = 0; i < proto->header_width; i += 4)
      {
	g_snprintf(s, MAXPATHLEN, "%i", i);
	bitnum = gtk_label_new(s);
	gtk_widget_show(bitnum);
	
	gtk_table_attach (GTK_TABLE (proto_table), bitnum,
			  i, i + 1,
			  0, 1,
			  (GtkAttachOptions) 0,
			  (GtkAttachOptions) 0, 0, 0);	
      }
  }
*/

  /* Iterate over all fixed header fields and hook buttons into
   * the table: */
  for (i = x = 0, y = 0; proto->fields[i].label; i++)
    {
      w_middle = h_middle = w_bot = 0;      
      w_top = proto->fields[i].bits;
      
      if (x + w_top > bpl)
	w_top = bpl - x;

      if (w_top < proto->fields[i].bits)
	{
	  w_middle = proto->fields[i].bits - w_top;
	  h_middle = MAX(w_middle / bpl, 1);

	  if (w_middle > bpl)
	    {
	      w_middle = bpl;	      
	      w_bot = proto->fields[i].bits - w_top - h_middle * bpl;
	    }

	  largest = (w_middle > w_top ? 1 : 0);
	}

      
      label = (largest == 0 ? NULL : _("->"));
      button = gui_table_create_button(proto_table, pinf, &proto->fields[i],
				       label, offset, FALSE, TRUE);

      gtk_table_attach (GTK_TABLE (proto_table), button,
			x, x + w_top,
			y, y + 1,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) 0, 0, 0);

      x += w_top;

      if (w_middle)
	{
	  int options;

	  x = 0;
	  y++;
	  label = (largest == 1 ? NULL : _("<-"));
	  options = (h_middle > 1 ? (GTK_EXPAND | GTK_FILL) : 0);

	  button = gui_table_create_button(proto_table, pinf, &proto->fields[i],
					   label, offset, FALSE, TRUE);
	  gtk_table_attach (GTK_TABLE (proto_table), button,
			    0, w_middle,
			    y, y + h_middle,
			    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			    (GtkAttachOptions) options, 0, 0);

	  x = w_middle;
	  y += h_middle;
	}

      if (w_bot)
	{
	  button = gui_table_create_button(proto_table, pinf, &proto->fields[i],
					   _("<-"), offset, FALSE, FALSE);
	  gtk_table_attach (GTK_TABLE (proto_table), button,
			    0, w_bot,
			    y, y + 1,
			    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			    (GtkAttachOptions) 0, 0, 0);

	  x = w_bot;
	}
      
      if (x >= bpl)
	{
	  y++;
	  x = 0;
	}

      offset += proto->fields[i].bits;
    }

  return proto_table;
}


void
nd_gui_proto_table_add(ND_Trace *trace,
		       ND_ProtoInfo *pinf,
		       ND_ProtoField *field,
		       void *data,
		       gboolean is_error)
{
  GList         *opt_widgets;
  GtkWidget     *table, *button;
  GtkTooltips   *tt;
  int            w_top, w_middle, w_bot, h_middle, largest = 0;
  int            x, y, offset;
  char           label[MAXPATHLEN], *final_label;
  ND_Protocol   *proto;
  
  if (!trace || !pinf)
    return;

  proto = pinf->inst.proto;

  opt_widgets = nd_proto_info_get_data(pinf, nd_proto_get_opt_key(proto));

  table = pinf->proto_gui;
  D_ASSERT_PTR(table);
  tt = gtk_object_get_data(GTK_OBJECT(toplevel), "tooltips");
  D_ASSERT_PTR(tt);

  x = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(table), "options_x"));
  y = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(table), "options_y"));
  offset = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(table), "options_offset"));

  g_snprintf(label, MAXPATHLEN, field->label, data);
  w_middle = h_middle = w_bot = 0;      
  w_top = field->bits;
      
  if (x + w_top > proto->header_width)
    w_top = proto->header_width - x;

  if (w_top < field->bits)
    {
      w_middle = field->bits - w_top;
      h_middle = MAX(w_middle / proto->header_width, 1);

      if (w_middle > proto->header_width)
	{
	  w_middle = proto->header_width;
	  w_bot = field->bits - w_top - h_middle * proto->header_width;
	}

      largest = (w_middle > w_top ? 1 : 0);
    }

  final_label = (largest == 0 ? label : _("->"));

  button = gui_table_create_button(table, pinf, field,
				   final_label, offset,
				   is_error, TRUE);
  opt_widgets = g_list_append(opt_widgets, button);

  gtk_table_attach(GTK_TABLE(table), button,
		   x, x + w_top,
		   y, y + 1,
		   (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		   (GtkAttachOptions) 0, 0, 0);

  x += w_top;

  if (w_middle)
    {
      int options;

      y++;
      final_label = (largest == 1 ? label : _("<-"));
      options = (h_middle > 1 ? (GTK_EXPAND | GTK_FILL) : 0);

      button = gui_table_create_button(table, pinf, field,
				       final_label, offset,
				       FALSE, TRUE);
      opt_widgets = g_list_append(opt_widgets, button);
      gtk_table_attach (GTK_TABLE (table), button,
			0, w_middle,
			y, y + h_middle,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) options, 0, 0);

      x = w_middle;
      y += h_middle;
    }

  if (w_bot)
    {
      button = gui_table_create_button(table, pinf, field,
				       _("<-"), offset,
				       FALSE, FALSE);
      opt_widgets = g_list_append(opt_widgets, button);
      gtk_table_attach (GTK_TABLE (table), button,
			0, w_bot,
			y, y + 1,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) 0, 0, 0);

      x = w_bot;
    }

  if (x == proto->header_width)
    {
      x = 0;
      y++;
    }

  gtk_object_set_data(GTK_OBJECT(table), "options_x", GINT_TO_POINTER(x));
  gtk_object_set_data(GTK_OBJECT(table), "options_y", GINT_TO_POINTER(y));
  gtk_object_set_data(GTK_OBJECT(table), "options_offset", GINT_TO_POINTER(offset + field->bits));

  nd_proto_info_set_data(pinf, nd_proto_get_opt_key(proto), opt_widgets);
}


void    
nd_gui_proto_table_clear(ND_Trace *trace, ND_ProtoInfo *pinf)
{
  ND_Protocol *proto;
  GList      *opt_widgets;
  GList      *l;
  GtkWidget  *table;
  int         x, y;

  if (!trace || !pinf)
    return;

  proto = pinf->inst.proto;

  table = pinf->proto_gui;
  D_ASSERT_PTR(table);

  x = proto->header_size % proto->header_width;
  y = proto->header_size / proto->header_width;

  gtk_object_set_data(GTK_OBJECT(table), "options_x", GINT_TO_POINTER(x));
  gtk_object_set_data(GTK_OBJECT(table), "options_y", GINT_TO_POINTER(y));
  gtk_object_set_data(GTK_OBJECT(table), "options_offset", GINT_TO_POINTER(proto->header_size));

  if ( (opt_widgets = nd_proto_info_get_data(pinf, nd_proto_get_opt_key(proto))))
    {
      for (l = g_list_first(opt_widgets); l; l = g_list_next(l))
	{
	  gtk_container_remove(GTK_CONTAINER(table), GTK_WIDGET(l->data));
	  gtk_widget_destroy(GTK_WIDGET(l->data));
	}
      
      g_list_free(opt_widgets);
      nd_proto_info_remove_data(pinf, nd_proto_get_opt_key(proto));
    }
}


void          
nd_gui_proto_table_block_events(ND_Trace * trace, ND_ProtoInfo *pinf)
{
  ND_Protocol *proto;
  GtkWidget   *button;
  int          i;

  if (!trace || !pinf || !pinf->inst.proto->fields)
    return;

  proto = pinf->inst.proto;

  for (i = 0; proto->fields[i].label; i++)
    {
      if (proto->fields[i].callback)
	{
	  button = nd_proto_info_get_data(pinf, nd_proto_field_to_string(&proto->fields[i]));
	  D_ASSERT_PTR(button);
	  
	  gtk_signal_handler_block_by_func(GTK_OBJECT(button),
					   gui_proto_field_cb, &(proto->fields[i]));
	}
    }
}


void          
nd_gui_proto_table_unblock_events(ND_Trace * trace, ND_ProtoInfo *pinf)
{
  ND_Protocol *proto;
  GtkWidget   *button;
  int          i;

  if (!trace || !pinf || !pinf->inst.proto->fields)
    return;

  proto = pinf->inst.proto;

  for (i = 0; proto->fields[i].label; i++)
    {
      if (proto->fields[i].callback)
	{
	  button = nd_proto_info_get_data(pinf, nd_proto_field_to_string(&proto->fields[i]));
	  D_ASSERT_PTR(button);
	  
	  gtk_signal_handler_unblock_by_func(GTK_OBJECT(button),
					     gui_proto_field_cb, &(proto->fields[i]));
	}
    }
}


void      
nd_gui_proto_menu_register(ND_Protocol *proto)
{
  GtkWidget *proto_menu;
  GtkWidget *proto_menu_item;
  GList     *child;
  int        child_index;
  char      *label;

  if (!proto || !proto->menu)
    return;

  /* If the menu item already exists, we've been here before. */
  if (proto->proto_menu_item)
    return;

  D(("Hooking in menu for protocol %s\n", proto->name));

  proto_menu = gtk_object_get_data(GTK_OBJECT(toplevel), "proto_menu");
  D_ASSERT_PTR(proto_menu);
  
  for (child_index = 0, child = GTK_MENU_SHELL(proto_menu)->children; child;
       child = g_list_next(child), child_index++)
    {
      if (GTK_IS_LABEL(child))
	{
	  gtk_label_get(GTK_LABEL(GTK_BIN(child->data)->child), &label);
	  
	  if (strcmp(proto->name, label) > 0)
	    break;
	}
    }

  proto_menu_item = gtk_menu_item_new_with_label (proto->name);
  proto->proto_menu_item = proto_menu_item;

  gtk_widget_ref (proto_menu_item);
  gtk_object_set_data_full (GTK_OBJECT (toplevel), proto->name, proto_menu_item,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(proto_menu_item);
  gtk_widget_set_sensitive(proto_menu_item, FALSE);
  gtk_menu_insert(GTK_MENU(proto_menu), proto_menu_item, child_index);
  gtk_widget_show(proto->menu);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (proto_menu_item), proto->menu);  
}


void          
nd_gui_add_monowidth_widget(GtkWidget *widget)
{
  char         *fontname;
  GtkStyle     *style;

  if (!widget)
    return;

  if (!nd_prefs_get_str_item(ND_DOM_NETDUDE, "font_mono", &fontname))
    fontname = "-*-courier-medium-r-normal-*-*-100-*-*-m-*-*";

  style = gtk_style_copy(gtk_widget_get_style(widget));
  gdk_font_unref(style->font);
  style->font = gdk_font_load (fontname);
  gtk_widget_set_style(widget, style);

  if (GTK_IS_CLIST(widget))
    {
      gtk_clist_set_row_height(GTK_CLIST(widget),
      			       widget->style->font->ascent +
			       widget->style->font->descent + 2);

      /* This is a workaround for what appears to be a bug in
	 the clist code -- the vertical font alignment isn't
	 set correctly sometimes: */
      GTK_CLIST(widget)->row_center_offset = widget->style->font->ascent + 1.5;
    }

  gtk_widget_queue_draw(widget);
  while (gtk_events_pending())
    gtk_main_iteration();

  if (g_list_index(monowidth_widgets, widget) < 0)
    monowidth_widgets = g_list_prepend(monowidth_widgets, widget);
}


void
nd_gui_del_monowidth_widget(GtkWidget *widget)
{
  GList *l;

  if (!widget)
    return;

  if ( (l = g_list_find(monowidth_widgets, widget)))
    monowidth_widgets = g_list_remove_link(monowidth_widgets, l);
}


void          
nd_gui_update_monowidth_widgets(void)
{
  GList *l;

  for (l = monowidth_widgets; l; l = g_list_next(l))
    nd_gui_add_monowidth_widget(l->data);
}
