/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <netdude/nd_debug.h>
#include <netdude/nd_gettext.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_raw_protocol.h>

#include <nd_sll.h>
#include <nd_sll_callbacks.h>


static ND_ProtoField sll_fields[] = {
  { ND_VAL_FIELD, N_("Packet type (%u)"), N_("SLL packet type"),          16, nd_sll_pkttype_cb },
  { ND_VAL_FIELD, N_("HW type (%u)"),     N_("SLL hardware type"),        16, nd_sll_hatype_cb  },
  { ND_VAL_FIELD, N_("HW len (%u)"),      N_("SLL hardware address len"), 16, nd_sll_halen_cb   },
  { ND_VAL_FIELD, N_("HW addr (%s)"),     N_("SLL hardware address"),     64, nd_sll_addr_cb    },
  { ND_VAL_FIELD, N_("Proto (0x%.4x)"),   N_("SLL protocol"),             16, nd_sll_proto_cb   },
  { 0, NULL, NULL, 0, NULL }
};


static ND_Protocol *sll;

static gboolean
sll_header_complete(const ND_Packet *packet, guchar *data)
{
  return (data + SLL_HEADER_LEN <= nd_packet_get_end(packet));
}


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return _("Linux SLL Plugin");
}


const char *
description(void)
{
  return _("A plugin providing support for the Linux sll "
	   "protocol for cooked packets.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return ("0.1.0");
}


ND_Protocol *
init(void)
{
  sll = nd_proto_new("SLL", ND_PROTO_LAYER_LINK, DLT_LINUX_SLL);
			     		
  sll->create_gui      = nd_sll_create_gui;
  sll->set_gui         = nd_sll_set_gui;
  sll->init_packet     = nd_sll_init_packet;
  sll->header_complete = nd_sll_header_complete;

  /* We're using a button table to display the protocol content,
     so we need to hook them in here: */
  sll->fields = sll_fields;

  sll->header_width = 32;
	     
  return sll;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_sll_create_gui(ND_Trace *trace, ND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);

  return table;
}


void       
nd_sll_set_gui(const ND_Packet *packet, ND_ProtoInfo *pinf)
{
  struct sll_header *sllhdr;
  
  sllhdr = (struct sll_header*) nd_packet_get_data(packet, sll, 0);

  nd_sll_set_gui_pkttype(pinf, sllhdr);
  nd_sll_set_gui_hatype(pinf, sllhdr);
  nd_sll_set_gui_halen(pinf, sllhdr);
  nd_sll_set_gui_addr(pinf, sllhdr);
  nd_sll_set_gui_protocol(pinf, sllhdr);
}


void       
nd_sll_init_packet(ND_Packet *packet, guchar *data, guchar *data_end)
{
  struct sll_header *sllhdr;
  ND_Protocol *payload_proto;

  sllhdr = (struct sll_header *) data;

  if (!sll_header_complete(packet, data))
    {
      nd_raw_proto_get()->init_packet(packet, data, data_end);
      return;
    }

  nd_packet_add_proto_data(packet, sll, data, data_end);  
  
  /* Check the appriopriate header field value to demultiplex
     packet initialization up to the next correct protocol: */

  payload_proto = nd_proto_registry_find(ND_PROTO_LAYER_LINK | ND_PROTO_LAYER_NET,
					 ntohs(sllhdr->sll_protocol));
  payload_proto->init_packet(packet,
			     data + SLL_HEADER_LEN,
			     data_end);
}


gboolean
nd_sll_header_complete(const ND_Packet *packet, guint nesting)
{
  guchar      *data;

  if (!packet)
    return FALSE;

  data = nd_packet_get_data(packet, sll, 0);
  
  return sll_header_complete(packet, data);
  TOUCH(nesting);
}


/* Misc helper stuff below --------------------------------------------- */

void
nd_sll_set_gui_pkttype(ND_ProtoInfo *pinf, struct sll_header *sllhdr)
{
  nd_proto_field_set(pinf, &sll_fields[0], DATA_TO_PTR(ntohs(sllhdr->sll_pkttype)));
}


void       
nd_sll_set_gui_hatype(ND_ProtoInfo *pinf, struct sll_header *sllhdr)
{
  nd_proto_field_set(pinf, &sll_fields[1], DATA_TO_PTR(ntohs(sllhdr->sll_hatype)));
}


void       
nd_sll_set_gui_halen(ND_ProtoInfo *pinf, struct sll_header *sllhdr)
{
  nd_proto_field_set(pinf, &sll_fields[2], DATA_TO_PTR(ntohs(sllhdr->sll_halen)));
}


void       
nd_sll_set_gui_addr(ND_ProtoInfo *pinf, struct sll_header *sllhdr)
{
  char label[MAXPATHLEN];

  g_snprintf(label, MAXPATHLEN, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x:%.2x:%.2x",
	     sllhdr->sll_addr[0], sllhdr->sll_addr[1], sllhdr->sll_addr[2], 
	     sllhdr->sll_addr[3], sllhdr->sll_addr[4], sllhdr->sll_addr[5], 
	     sllhdr->sll_addr[6], sllhdr->sll_addr[7]);
  nd_proto_field_set(pinf, &sll_fields[3], label);
}


void       
nd_sll_set_gui_protocol(ND_ProtoInfo *pinf, struct sll_header *sllhdr)
{
  nd_proto_field_set(pinf, &sll_fields[4], DATA_TO_PTR(ntohs(sllhdr->sll_protocol)));
}


ND_Protocol *
nd_sll_get(void)
{
  return sll;
}

