/**
 * \file main.c
 * \brief This file has the main function. Execution starts here
 *
 * \author Fernando J. Pereda <ferdy@ferdyx.org>
 * \author Ricardo Cervera Navarro <ricardo@zonasiete.org>
 *
 * This is part of nbsmtp. nbsmtp is free software;
 * you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * nbsmtp is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with nbsmtp; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See COPYING for details.
 */

#include <syslog.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "original.h"
#include "util.h"
#include "servinfo.h"
#include "nbsmtp.h"

#ifdef HAVE_GROWLNOTIFY
#include "osx.h"
#endif

/**
 * \brief nbSMTP starts its execution here
 *
 * \param[in] argc Number of arguments
 * \param[in] argv Array with the arguments
 * \return 0 if everything goes ok, 1 if something goes wrong
 */
int main(int argc, char *argv[])
{
	servinfo_t serverinfo;
	string_t msg_buffer;
	string_t *rcpts = NULL;

	str_init(&msg_buffer, MAX_MSG_LEN);

	serverinfo.host = (char) NULL;
	serverinfo.fromaddr = (char) NULL;
	serverinfo.domain = (char) NULL;
	serverinfo.auth_user = (char) NULL;
	serverinfo.auth_pass = (char) NULL;
	serverinfo.auth_mech = SASL_DEFAULT;
	serverinfo.port = 0;
	serverinfo.num_rcpts = 0;
#ifdef HAVE_SSL
	serverinfo.use_tls = False;
	serverinfo.use_starttls = False;
	serverinfo.using_tls = False;
#endif

	/**
	 * HACK
	 * This is a quick and dirty hack to fix the silly options in nbSMTP
	 * I want to change them but it is better to provide backwards compatibility.
	 * This calls print_help if the call was something like 'nbmtp -h' with no aditional args
	 * Also add --help and --version
	 *
	 * \todo Fix the command line options... maybe breaking 0.8 compatibility in 1.0
	 */
	if (argc==2)
	{
		if ( strncmp(argv[1],"-h",2)==0 || strncmp(argv[1],"--help",6)==0 )
		{
			print_help(argv[0]);
			return 0;
		}
		else if ( strncmp(argv[1],"--version",9)==0 )
		{
			printf("nbSMTP version %s\n",PACKAGE_VERSION);
			return 0;
		}
	}

	switch (parse_options(argc,argv, &serverinfo))
	{
		case 1:
			return 1;
			break;
		case 2:
			return 0;
			break;
	}

	if((rcpts = parse_mail(&msg_buffer, &(serverinfo.num_rcpts))) == NULL)
	{
		log_msg(LOG_DEBUG, "Error in parse_mail");
		if (debug > 0)
		{
			log_msg(LOG_ERR, "Mail NOT sent.");
		}
		else
		{
			log_msg(LOG_ERR, "Something went wrong. Mail NOT sent. Activate debug for more info");
		}
		return 1;

#ifdef HAVE_GROWLNOTIFY
		osx_notify("nbSMTP","Mail NOT sent, see log for more details");
#endif
	}

	if(send_mail(&msg_buffer, &serverinfo, rcpts))
	{
		log_msg(LOG_DEBUG, "Error in send_mail");
		if (debug > 0)
		{
			log_msg(LOG_ERR, "Mail NOT sent.");
		}
		else
		{
			log_msg(LOG_ERR, "Something went wrong. Mail NOT sent. Activate debug for more info");
		}

#ifdef HAVE_GROWLNOTIFY
		osx_notify("nbSMTP","Mail NOT sent, see log for more details");
#endif
		return 1;
	}

	log_msg(LOG_INFO,"Mail sent for %s. Closing connection",serverinfo.fromaddr);

#ifdef HAVE_GROWLNOTIFY
	osx_notify("nbSMTP","Mail sent for %s",serverinfo.fromaddr);
#endif

	str_free(rcpts);
	free(rcpts);
	str_free(&msg_buffer);

	SERVINFO_RELEASE_OPTION(serverinfo.host);
	SERVINFO_RELEASE_OPTION(serverinfo.fromaddr);
	SERVINFO_RELEASE_OPTION(serverinfo.domain);
	SERVINFO_RELEASE_OPTION(serverinfo.auth_user);
	SERVINFO_RELEASE_OPTION(serverinfo.auth_pass);

	return 0;
}
