--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Strings.Fixed;
with Ada.Strings.Unbounded;
with GNAT.OS_Lib;

with Narval.Configurator.Actors_Description;
with Narval.Coordination_Local_Hook;
with Narval.Environment;

with Log4ada.Loggers;

use Narval.Coordination_Local_Hook;

package body Narval.Configurator.Compilation is

   use Log4ada.Loggers;

   ---------------
   -- Configure --
   ---------------

   procedure Configure is
      use Ada.Strings.Fixed;
      use Ada.Strings.Unbounded;
      File_Name : constant String := To_String
        (Configuration_Hook.Configuration_File);
      Configuration_Directory : constant GNAT.OS_Lib.String_Access :=
        Environment.Configuration_Directory;
      Xml_Configuration_File : Boolean := False;
   begin
      if File_Name = "" then
         raise No_Configuration_File;
      end if;
      Xml_Configuration_File := Index (File_Name, ".xml") /= 0;
      if Xml_Configuration_File then
         declare
            Full_File_Name : constant String :=
              Configuration_Directory.all & "/" & File_Name;
         begin
            Actors_Description.Fill (Configuration_Hook.Actors,
                                     Full_File_Name);
            Actors_Description.Rank_Calcul (Configuration_Hook.Actors);
         exception
            when E : others =>
               Fatal_Out (Configuration_Hook.Logger'Access,
                          "problem in configure procedure", E);
               raise;
         end;
      else
         Fatal_Out (Configuration_Hook.Logger'Access,
                    "file " & File_Name &
                    " doesn't seem to be a configuration file");
         raise Configuration_Failed;
      end if;
      Actors_Description.Actors_Sorting.Sort (Configuration_Hook.Actors);
      declare
         use Actors_Description;
         Cursor : Actors_Description_Vector.Cursor;
         use type Actors_Description_Vector.Cursor;
      begin
         Cursor := Actors_Description_Vector.First (Configuration_Hook.Actors);
         loop
            exit when Cursor = Actors_Description_Vector.No_Element;
            Info_Out
              (Configuration_Hook.Logger'Access,
               To_String (Actors_Description_Vector.Element (Cursor).Name) &
                 Actors_Description_Vector.Element (Cursor).Rank_Number'Img);
            Cursor := Actors_Description_Vector.Next (Cursor);
         end loop;
      end;
   exception
      when No_Configuration_File =>
         raise;
      when Configuration_Failed =>
         raise;
      when E : others =>
         Error_Out (Configuration_Hook.Logger'Access,
                    "configuration ratee configure line 113", E);
         Reset;
         raise Configuration_Failed;
   end Configure;

   -------------------
   -- Remise_A_Zero --
   -------------------

   procedure Reset is
   begin
      Actors_Description.Free (Configuration_Hook.Actors);
   end Reset;

   procedure Pointers_Reset is
      use Actors_Description;
      Cursor : Actors_Description_Vector.Cursor;
      use type Actors_Description_Vector.Cursor;
   begin
      Cursor := Actors_Description_Vector.First (Configuration_Hook.Actors);
      loop
         exit when Cursor = Actors_Description_Vector.No_Element;
         select
            Actors_Description_Vector.Element (Cursor).Task_Register.Clear;
         or
            delay 0.1;
            Warn_Out (Configuration_Hook.Logger'Access,
                      "cannot clear actor pointer");
         end select;
         Cursor := Actors_Description_Vector.Next (Cursor);
      end loop;
   end Pointers_Reset;

end Narval.Configurator.Compilation;
