/* 
 * Copyright (c) 2008, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _MFORMS_APP_H_
#define _MFORMS_APP_H_

#include <map>

#include "mforms/base.h"
#include "base/geometry.h"
#include "base/drawing.h"

namespace mforms {

  // Known system colors.
  enum SystemColor
  {
    SystemHighlight,         // The system-defined color of the background of selected items.
                             // This includes selected menu items as well as selected text.
  };

  class App;
  class AppView;
  class View;

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#ifndef SWIG
  struct AppImplPtrs
  {
    void (__stdcall *dock_view)(App *app, AppView *view, const std::string &position);
    bool (__stdcall *select_view)(App *app, const std::string &identifier);
    void (__stdcall *undock_view)(App *app, AppView *view);

    void (__stdcall *set_status_text)(App *app, const std::string &title);
    
    void (__stdcall *set_view_title)(App *app, AppView *view, const std::string &title);
    std::string (__stdcall *get_resource_path)(App *app, const std::string &file);
    std::string (__stdcall *get_bounds)(App *app);
    
    int (__stdcall *enter_event_loop)(App *app, float max_wait_time);
    void (__stdcall *exit_event_loop)(App *app, int result);
    
    void (__stdcall *begin_thread_loop)();
    void (__stdcall *end_thread_loop)();

    MySQL::Drawing::Color (__stdcall *get_system_color)(SystemColor type);
  };
#endif
#endif

  /** Proxy class for interfacing with the host application window.
   
   Provides some utility functions to perform certain actions in the main window
   of the host application. This class uses a singleton.
   */
  class MFORMS_EXPORT App : public Object
  {
  private:
    App() {}

  public:
    
    /** Gets the instance of the App class.
     */
    static App *get();

    /** Docks an AppView into the main window, at the requested position.
     
     @param view - an AppView object to be docked in the main window
     @param position - position in the main window to dock
     
     In Workbench, view will be added into one of the tabs in the application window.
     Currently, only a position of "maintab" is recognized.
     */
    void dock_view(AppView *view, const std::string &position);
    
    /** Selects a docked view by its identifier.
     
     @return false if a view with the identifier could not be found. identifier
     must be set in the AppView before its docked.
     */
    bool select_view(const std::string &identifier);
    
    /** Undocks an AppView from the main window.
     */
    void undock_view(AppView *view);
    
    /** Sets the title of a previously docked AppView. 
     
     In Workbench this will set the label of the Tab where the AppView is docked.
     */
    void set_view_title(AppView *view, const std::string &title);
    
    /** Sets the status text at the bottom of the window.
     */
    void set_status_text(const std::string &text);
    
    /** Gets the path to a resource file bundled in the application. 
     
     Passing an empty string will return the path to the resource directory. */
    std::string get_resource_path(const std::string &file);

    std::string get_user_data_folder() { return _user_data_folder; }
    
    /** Gets the bounds of the main window. */
    void get_bounds(int* x, int* y, int* w, int* h);
    
    /** Enters the event loop and exit only once exit_event_loop() is called.
     @param timeout Given in seconds. If > 0.0, the function will return -1 after that time.
       */
    int enter_event_loop(float timeout = 0.0);
    
    /** Exits from enter_event_loop() */
    void exit_event_loop(int retcode);
    
    void set_view_for_identifier(View *view, const std::string &identifier);
    View *get_view_for_identifier(const std::string &identifier);
    
    // use in threads where mforms is called
    static void begin_thread_loop();
    static void end_thread_loop();
    
    // Retrieve system information like predefined colors, screen size, monitor count etc.
    MySQL::Drawing::Color get_system_color(SystemColor type);
  public:
#ifndef DOXYGEN_SHOULD_SKIP_THIS
#ifndef SWIG
    void set_user_data_folder_path(const std::string &path) { _user_data_folder = path; }
#endif
#endif
  protected:
    AppImplPtrs *_app_impl;
    std::map<std::string, View*> _view_by_identifier;
    
    std::string _user_data_folder;
    
    friend class View;
    void view_destroyed(View *view);
  };
};

#endif
