#! /usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c)2003, Matthias A. Benkard.

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


from core.common import _, path
import core
from core import common, parser, maxima

import gtk
import pango
from gtk import gdk
import logging

log = logging.getLogger("gui.algebra")


class AlgebraSheet (gtk.VBox):
	"""A composite widget that gets input from the user and displays MAXIMA output."""
	
	def __init__(self, maxima):
		gtk.VBox.__init__(self, 2)
		self.set_homogeneous(False)
		
		self.viewer = AlgebraViewer()
		self.input = gtk.Entry()
		
		scrollwin = gtk.ScrolledWindow()
		scrollwin.add(self.viewer)
		
		self.pack_start(scrollwin, expand = gtk.TRUE, fill = gtk.TRUE, padding = 0)
		self.pack_start(self.input, expand = gtk.FALSE, fill = gtk.FALSE, padding = 0)
		
		self.show_all()
		
		self.input.connect("activate", self.on_input)
		
		self.maxima = maxima
	
	
	def on_input(self, widget):
		if widget.get_text():
			id, type, output, error = self.maxima.maxima_send_command(widget.get_text())
			if not error:
				self.viewer.show((id, maxima.INPUT, widget.get_text()))
				self.viewer.show((id, maxima.RESULT, output))
			widget.set_text("")


# Now we're getting low-level.
# The GTK tutorial is really helpful :)
# http://www.gtk.org/tutorial1.0/gtk_tut-25.html
class AlgebraViewer (gtk.Layout):
	def __init__(self):
		gtk.Layout.__init__(self)
		self.set_events(gdk.BUTTON_PRESS_MASK | gdk.BUTTON_RELEASE_MASK |
				gdk.KEY_PRESS_MASK            | gdk.KEY_RELEASE_MASK    |
				gdk.POINTER_MOTION_HINT_MASK  | gdk.POINTER_MOTION_MASK |
				gdk.EXPOSURE_MASK             | gdk.STRUCTURE_MASK      |
				gdk.BUTTON_MOTION_MASK)
		
		# Interesting litte detail:
		# The first argument to an event handler function is the
		# event-emitting widget, that is, "self" :)
		self.connect("motion_notify_event", AlgebraViewer.on_motion_notify)
		self.connect("button_press_event", AlgebraViewer.on_button_press)
		
		self.get_hadjustment().step_increment = 10.0
		self.get_vadjustment().step_increment = 10.0
		
		self.maxima_data = []
		self.h = 0
		self.w = 0
	
	
	def on_motion_notify(self, event):
		pass
	
	
	def on_button_press(self, event):
		pass
	
	
	def show(self, record):
		id, type, text = record
		
		obj = parser.parse(text)
		
		self.maxima_data += id, type, text, obj
		
		pm = obj.draw(self, self.style)
		pw, ph = pm.get_size()
		
		font_desc = self.style.font_desc.copy()
		font_desc.set_weight(pango.WEIGHT_BOLD)
		font = gdk.font_from_description(font_desc)
		
		# The ID.
		if type == core.maxima.RESULT:
			a = "(D%i) " % id
		else:
			a = "(C%i) " % id
		
		w = font.string_width(a)
		h = font.ascent + font.descent
		
		idpm = gdk.Pixmap(self.window, w, h)
		idpm.draw_rectangle(
				self.style.bg_gc[gtk.STATE_NORMAL], True,
				0, 0,
				w, h)
		
		idpm.draw_text(
				font, self.style.fg_gc[gtk.STATE_NORMAL],
				0, font.ascent,
				a)
		
		img = gtk.Image()
		img.set_from_pixmap(pm, None)
		idimg = gtk.Image()
		idimg.set_from_pixmap(idpm, None)
		
		self.put(idimg, 0, self.h)
		
		if type == core.maxima.RESULT:
			w += 40
		
		self.put(img, w, self.h)
		
		self.h += max(h, ph)
		self.w = max(self.w, w + pw)
		self.set_size(self.w, self.h)
		self.show_all()

