# -*- coding: utf-8 -*-
#
# Copyright 2010-2012 Chicharreros
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Magicicada GTK UI."""

import sys
import gettext
import logging

# pylint: disable=E0611
from gi.repository import GdkPixbuf
# pylint: enable=E0611

# optional Launchpad integration, pylint: disable=F0401
# this shouldn't crash if not found as it is simply used for bug reporting
try:
    import LaunchpadIntegration
    LAUNCHPAD_AVAILABLE = True
except ImportError:
    LAUNCHPAD_AVAILABLE = False
# pylint: enable=F0401

from magicicada import syncdaemon
from magicicada.helpers import log, NO_OP
from magicicada.gui.gtk.helpers import (
    Buildable,
    build_icon_dict,
    get_data_file,
)
from magicicada.gui.gtk.operations import Operations
from magicicada.gui.gtk.status import Status


INSTALL_KWARGS = {}
if sys.version_info < (3,):
    INSTALL_KWARGS["unicode"] = True

gettext.install('ubuntu-sso-client', **INSTALL_KWARGS)
logger = logging.getLogger('magicicada.gui.gtk')


# Instance of 'A' has no 'y' member
# pylint: disable=E1101

# Unused argument, we need them for GTK callbacks
# pylint: disable=W0613


class MagicicadaUI(Buildable):
    """Magicicada GUI main class."""

    CURRENT_ROW = '<b><span foreground="#000099">%s</span></b>'
    filename = 'main.ui'
    logger = logger

    def __init__(self, on_destroy=NO_OP):
        """Init."""
        super(MagicicadaUI, self).__init__()
        self.sd = syncdaemon.SyncDaemon()

        if LAUNCHPAD_AVAILABLE:
            # for more information about LaunchpadIntegration:
            # wiki.ubuntu.com/UbuntuDevelopment/Internationalisation/Coding
            helpmenu = self.builder.get_object('helpMenu')
            if helpmenu:
                LaunchpadIntegration.set_sourcepackagename('magicicada')
                LaunchpadIntegration.add_items(helpmenu, 0, False, True)

        self._on_destroy = on_destroy

        active_filename = get_data_file('media', 'active-016.png')
        self.active_indicator = GdkPixbuf.Pixbuf.new_from_file(active_filename)

        self.status = Status(syncdaemon_instance=self.sd, xscale=1, yscale=1)
        self.main_box.pack_start(self.status, expand=False, fill=True,
                                 padding=6)

        self._icons = {}
        for size in (16, 32, 48, 64, 128):
            icon_filename = get_data_file('media', 'logo-%.3i.png' % size)
            self._icons[size] = GdkPixbuf.Pixbuf.new_from_file(icon_filename)
        self.main_window.set_default_icon_list(self._icons.values())
        self.main_window.set_icon_list(self._icons.values())

        self._status_icons = build_icon_dict(16)
        self.status_icon = self.builder.get_object('status_icon')
        self.status_icon.set_from_pixbuf(self._status_icons['idle'])

        about_fname = get_data_file('media', 'logo-128.png')
        self.about_dialog.set_logo(GdkPixbuf.Pixbuf.new_from_file(about_fname))

        self.operations = Operations(syncdaemon_instance=self.sd)
        self.main_box.pack_start(self.operations, expand=True,
                                 fill=True, padding=0)

        self.sd.status_changed_callback = self.on_status_changed
        self.sd.on_initial_data_ready_callback = self.on_initial_data_ready
        self.sd.on_initial_online_data_ready_callback = \
            self.on_initial_online_data_ready

    def destroy(self, *a, **kw):
        """Destroy all widgets."""
        self.main_window.destroy()

    def on_destroy(self, widget=None, data=None):
        """Called when this widget is destroyed."""
        self.sd.shutdown()
        self._on_destroy()

    on_main_window_destroy = on_destroy

    def on_quit_activate(self, widget, data=None):
        """Signal handler for closing the program."""
        self.on_main_window_destroy(self.main_window)

    def on_about_activate(self, widget, data=None):
        """Display the about box."""
        self.about_dialog.run()
        self.about_dialog.hide()

    def on_status_icon_activate(self, widget, data=None):
        """Systray icon was clicked."""
        if self.main_window.get_visible():
            self.main_window.hide()
        else:
            self.main_window.show()

    @log(logger)
    def on_status_changed(self, *args, **kwargs):
        """Callback'ed when the SD status changed."""
        self.status.update(*args, **kwargs)

        # change status icon
        state = kwargs.get('state')
        if state == syncdaemon.STATE_IDLE:
            self.status_icon.set_from_pixbuf(self._status_icons['idle'])
        elif state == syncdaemon.STATE_WORKING:
            self.status_icon.set_from_pixbuf(self._status_icons['working'])
        else:
            self.status_icon.set_from_pixbuf(self._status_icons['alert'])

    @log(logger, level=logging.INFO)
    def on_initial_data_ready(self):
        """Initial data is now available in syncdaemon."""
        self.status.on_initial_data_ready()
        self.operations.load()

    @log(logger, level=logging.INFO)
    def on_initial_online_data_ready(self):
        """Online initial data is now available in syncdaemon."""
        self.status.on_initial_online_data_ready()
