/* SimpleAction.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */
package be.ugent.caagt.swirl.actions;

import java.awt.event.ActionEvent;
import java.util.ResourceBundle;
import javax.swing.AbstractAction;
import javax.swing.Icon;

/**
 * Abstract action (controller) based on an internationalized description.
 * As with {@link AbstractAction} you need only  define the
 * <code>actionPerformed</code> method when you extend this class.<p>
 * Also defines two new action properties for use with tool bar buttons.
 *
 * @see Description
 * @see ToolBarButton
 * @see ToolBarToggleButton
 */
public abstract class SimpleAction extends AbstractAction {

    /**
     * The action key used for storing a medium size <code>Icon</code>
     * for use with tool bar buttons. Icons registered
     * with this key will be used for tool bar buttons
     * of type {@link ToolBarToggleButton} or {@link ToolBarButton}, but not
     * for standard Swing buttons.
     */
    public static final String MEDIUM_ICON = "MediumIcon";
    
    /**
     * The action key used for storing a medium size <i>disabled</i>
     * <code>Icon</code> for use with tool bar buttons. Icons registered
     * with this key will be used for tool bar buttons
     * of type {@link ToolBarToggleButton} or {@link ToolBarButton}, but not
     * for standard Swing buttons.
     */
    public static final String MEDIUM_DISABLED_ICON = "MediumDisabledIcon";
    
    /**
     * Construct an action with an internationalized description
     * string and given icons.
     *
     * @param bundle Resource bundle which contains the description for
     * this action.
     * @param key Key used for the description of this action in the bundle.
     * @param icon Small icon to be used for menu and standard button decoration (or null)
     * @param mediumIcon Medium size icon to be used for tool bar buttons (or null)
     * @param mediumDisabledIcon Medium size icon to be used for a disabled tool bar buttons (or null)
     */
    protected SimpleAction 
        (ResourceBundle bundle, String key, Icon icon, Icon mediumIcon, Icon mediumDisabledIcon) {
	this (bundle, key, icon);
        if (mediumIcon != null)
            putValue (MEDIUM_ICON, mediumIcon);
        if (mediumDisabledIcon != null)
            putValue (MEDIUM_DISABLED_ICON, mediumDisabledIcon);
    }
    
    /**
     * Construct an action with an internationalized description
     * string and given icon.
     *
     * @param bundle Resource bundle which contains the description for
     * this action.
     * @param key Key used for the description of this action in the bundle.
     * @param icon Small icon to be used for menu and standard button decoration (or null).
     */
    protected SimpleAction (ResourceBundle bundle, String key, Icon icon) {
	super (null, icon);
        new Description(bundle.getString(key)).init (this);
    }
    
    // avoids buggy PMD warning
    public abstract void actionPerformed(ActionEvent e);

}
