/* SelectionGroup.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;
import javax.swing.AbstractButton;
import javax.swing.DefaultSingleSelectionModel;
import javax.swing.SingleSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * Acts like a button group with associated single selection model. Of all toggle buttons
 * added to a group of this kind, at most one can be selected at the same time.
 * Every time a new selection is made, this is reported to the selection model.
 * Buttons are given an internal index as they are added to the group (starting at 0).
 * When the selection model selects a new index, the corresponding button will
 * be selected and the old button will be deselected.
 * <p>Provides convenience methods to add listeners to the associated selection model
 * and to determine what is the currently selected button,
 * either as an index, an action command or a button object.
 * <p>A group like this is especially useful when the same set of choices must be 
 * available at the same time in different forms: as a set of radio button menu
 * items and a set of toggle buttons in a tool bar, say. Implementing both groups as 
 * selection groups with the same model makes both groups automatically synchronized.
 * <p>In the above example, the group of radio menu items 
 * would for instance be created as follows:
 * <pre>
 *    SelectionGroup menuGroup = new SelectionGroup ();
 *    JRadioButtonMenuItem item0 = new JRadioButtonMenuItem (...);
 *    JRadioButtonMenuItem item1 = new JRadioButtonMenuItem (...);
 *    JRadioButtonMenuItem item2 = new JRadioButtonMenuItem (...);
 *    menuGroup.add (item0);
 *    menuGroup.add (item1);
 *    menuGroup.add (item2);
 * </pre>
 * and the group of toggle buttons
 * <pre>
 *    SelectionGroup buttonGroup = new SelectionGroup (menuGroup.getModel(), true);
 *    JToggleButton button0 = new JToggleButton (...);
 *    JToggleButton button1 = new JToggleButton (...);
 *    JToggleButton button2 = new JToggleButton (...);
 *    buttonGroup.add(item0);
 *    buttonGroup.add(item1);
 *    buttonGroup.add(item2);
 * </pre>
 * To query which of the three buttons in each group is currently selected,
 * use {@code menuGroup.getSelectedIndex()} or 
 * {@code buttonGroup.getSelectedIndex()}, which will both yield the same answer.
 * Alternatively, the selection model can be queried directly, or a change listener
 * may be registered with the selection model.
 */
public class SelectionGroup  {
    
    //
    private SingleSelectionModel model;
    
    /**
     * The single selection model used by this group.
     */
    public SingleSelectionModel getModel() {
        return model;
    }
    
    //
    private boolean clearable;
    
    /**
     * Create a selection group with a newly created single selection model and no
     * buttons. Short for {@code Selectiongroup (true)}.
     */
    public SelectionGroup() {
        this(new DefaultSingleSelectionModel(), true);
    }
    
    /**
     * Create a selection group with a newly created single selection model and no
     * buttons.
     * @param clearable indicates whether buttons can be cleared by clicking
     * on them. If false the group behaves like a button group.
     */
    public SelectionGroup(boolean clearable) {
        this(new DefaultSingleSelectionModel(), clearable);
    }
    
    /**
     * Create a selection group with the given model.
     * @param clearable indicates whether buttons can be cleared by clicking
     * on them. If false the group behaves like a button group.
     */
    public SelectionGroup(SingleSelectionModel model, boolean clearable) {
        super();
        this.model = model;
        this.clearable = clearable;
        buttons = new ArrayList<AbstractButton> ();
        lastButton = null;
        model.addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                selectButton(getSelectedButton());
            }
        });
    }
    
    /**
     * Register the listener with the model.
     */
    public void addChangeListener(ChangeListener listener) {
        if (model != null)
            model.addChangeListener(listener);
    }
    
    /**
     * Unregister the listener form the model.
     */
    public void removeChangeListener(ChangeListener listener) {
        if (model != null)
            model.removeChangeListener(listener);
    }
    
    // Button which was selected last
    private AbstractButton lastButton;
    
    private List<AbstractButton> buttons;
    
    /**
     * Select the given button.
     */
    private void selectButton(AbstractButton button) {
        if (lastButton != null)
            lastButton.setSelected(false);
        lastButton = button;
        if (button != null)
            button.setSelected(true);
    }
    
    /**
     * Add the given button to the group.
     */
    public void add(AbstractButton button) {
        int index = buttons.size();
        buttons.add(button);
        if (model != null && model.getSelectedIndex() == index)
            selectButton(button);
        button.addActionListener(new ToggleAction(index));
    }
    
    /**
     * Return the button which is currently selected.
     * @return the currently selected button or null if none is selected
     */
    public AbstractButton getSelectedButton() {
        int index = model.getSelectedIndex();
        if (index >=0 && index < buttons.size())
            return buttons.get(index);
        else
            return null;
    }
    
    /**
     * Return the current selection.
     * @return the current selection index or -1 if none.
     */
    public int getSelectedIndex() {
        return model.getSelectedIndex();
    }
    
    /**
     * Return the action command of the currently selected button,
     * or null if none is selected.
     */
    public String getActionCommand() {
        int index = model.getSelectedIndex();
        if (index >=0 && index < buttons.size())
            return buttons.get(index).getActionCommand();
        else
            return null;
    }
    
    /**
     * Set the currently selected index.
     * @param index Index to be selected, or -1 to clear the selection.
     */
    public void setSelectedIndex(int index) {
        model.setSelectedIndex(index);
    }
    
    
    // listener for buttons
    private class ToggleAction implements ActionListener {
        
        private final int index;
        
        public ToggleAction(int index) {
            this.index = index;
        }
        
        // implements ActionListener
        public void actionPerformed(ActionEvent e) {
            AbstractButton button = (AbstractButton)e.getSource();
            if (button.isSelected())
                model.setSelectedIndex(index);
            else if (model.getSelectedIndex()==index) {
                if  (clearable)
                    model.clearSelection(); // clear if deselected
                else
                    button.setSelected(true); // override deselection
            }
        }
        
    }
    
}
