/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDVNCLOGIMPORTER_H
#define ISDVNCLOGIMPORTER_H

/**
 * Importer implementation class that imports isdviewer (vnc logs) files to the instrudeo format.
 *
 * @author Bram Biesbrouck <b@beligum.org>
*/

#include <string>
#include <sys/stat.h>

#include <libinstrudeo/isdimporter.h>
#include <libinstrudeo/isddatafile.h>
#include <libinstrudeo/isdobject.h>

using namespace std;

#if defined (_LP64) || \
    defined(__alpha) || defined(__alpha__) || \
    defined(__ia64__) || defined(ia64) || \
    defined(__sparc64__) || \
    defined(__s390x__) || \
    (defined(__hppa__) && defined(__LP64__)) || \
    defined(__amd64__) || defined(amd64) || \
    defined(__powerpc64__) || \
    (defined(sgi) && (_MIPS_SZLONG == 64))
#define _64_BIT_ARCH /* 32/64-bit architecture */
#endif

#define ISDVIEWER_MAGIC_TEMPLATE "isdviewer*.*"
#define MAX_MAJOR_SUPPORTED_VERSION 0
#define MAX_MINOR_SUPPORTED_VERSION 1

#define RFB_ENCODING_RAW 0

class ISDVncLogImporter : public ISDImporter
{

 public:
    /**
     * Loads a recording from a vnclog file, produced by the isdviewer program.
     * The object is merely contructed with all necessary field filled in. To actually 
     * parse the file and convert it to an instrudeo file, use the "convertTo" function. 
     *
     * @param vncLogFileName The filename of the file that was produced by the isdviewer program.
     */
    ISDVncLogImporter(const string vncLogFileName);
    
    virtual ~ISDVncLogImporter();

    /**
     * @see ISDImporter::convertTo()
     */
    ISDErrorCode convertTo(string& outputFileName, ISDProgressCallback* callbackClass = NULL, bool createNew=false);
    
 protected:
    //-----TYPEDEFS-----
    /**
     * The following structures are used to format some of the meta data in the binary file
     */
    typedef unsigned int U32;
    typedef unsigned short U16;
    typedef unsigned char U8;
    
    typedef struct {
	U8 bitsPerPixel;            /* 8,16,32 only */
	U8 depth;                   /* 8 to 32 */
      	U8 bigEndian;		/* True if multi-byte pixels are interpreted
				   as big endian, or if single-bit-per-pixel
				   has most significant bit of the byte
				   corresponding to first (leftmost) pixel. Of
				   course this is meaningless for 8 bits/pix */
	U8 trueColour;		/* If false then we need a "colour map" to
				   convert pixels to RGB.  If true, xxxMax and
				   xxxShift specify bits used for red, green
				   and blue */
	/* the following fields are only meaningful if trueColour is true */
	U16 redMax;	        	/* maximum red value (= 2^n - 1 where n is the
					   number of bits used for red). Note this
					   value is always in big endian order. */
	U16 greenMax;		/* similar for green */
	U16 blueMax;		/* and blue */
	U8 redShift;		/* number of shifts needed to get the red
				   value in a pixel to the least significant
				   bit. To find the red value from a given
				   pixel, do the following:
				   1) Swap pixel value according to bigEndian
				   (e.g. if bigEndian is false and host byte
				   order is big endian, then swap).
				   2) Shift right by redShift.
				   3) AND with redMax (in host byte order).
				   4) You now have the red value between 0 and
				   redMax. */
	U8 greenShift;		/* similar for green */
	U8 blueShift;		/* and blue */
	U8 pad1;
	U16 pad2;
    } RFBPixelFormat;

#define RFB_PIXEL_FORMAT_SIZE 16

    typedef struct {
	U16 framebufferWidth;
	U16 framebufferHeight;
	RFBPixelFormat format;
	U32 nameLength;
	/* followed by char name[nameLength] */
    } RFBInitMsg;

#define RFB_INIT_MSG_SIZE 24

    typedef struct {
	//this struct contains two U32 or two U64 values (depending on 64bit or not)
	struct timeval startTime;
	U16 numRects;
	// followed by the rect headers
    } RFBFbUpdate;

#if defined (_64_BIT_ARCH)
#define RFB_FB_UPDATE_SIZE 18
#else
#define RFB_FB_UPDATE_SIZE 10
#endif
    
    typedef struct {
	U16 x;
	U16 y;
	U16 w;
	U16 h;
	U32 encType;
	// followed by the rect data
    } RFBRectHeader;

#define RFB_RECT_HEADER_SIZE 12

    //-----METHODS-----
    /**
     * Is called when the import was cancelled.
     */
    void abortCleanup();
    
    /**
     * Reads the magic code string from the input file and stores it in the supplied string.
     * The string must contain a template of the magic code, so it can be used to determine how much data must be read
     * eg. instrudeo*.* (* will be filled with major and minor version numbers)
     * 
     * @param magic Is used to return the magic string.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode readMagic(string& magic);
    
    /**
     * Reads in the rfbInit variable with the data from the input file.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode readRfbInit();
    
    /**
     * Writes the header data to the output file.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode writeOutputHeader();

    /**
     * Processes a new framebufferupdate (extracting data and writing to the output file)
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode processFramebufferUpdate();
    
    /**
     * Encodes and writes rectangle-pixer-data to the output file
     *
     * @param rec The header of the rect to record.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode recordRectangle(ISDRectangle* rect);

    /**
     * Processes the pixels in the buffer, so that they are in the default ISD pixel format.
     *
     * @param rec The header of the rect to record.
     * @param buffer The pixel buffer.
     * @param bufLen The size of the pixel buffer.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode normalizePixels(ISDRectangle* rect, char*& buffer, int& bufLen);

    /**
     * Closes the two streams and does necessary cleanup.
     */
    void cleanup();
    
    //-----VARIABLES-----
    RFBInitMsg isdViewerHeader;
    string vncServerName;
    struct timeval startTime;
    bool firstFrame;
    bool needsNormalization;
};

#endif
