/**
 * @file libgalago-gtk/galago-gtk-service-combo-box.c Service ComboBox widget
 *
 * @Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-service-combo-box.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <libgalago/galago.h>
#include <string.h>

struct _GalagoGtkServiceComboBoxPriv
{
	gboolean rebuilding_list;

	gulong service_added_id;
	gulong service_removed_id;
};

enum
{
	SERVICE_CHANGED,
	LAST_SIGNAL
};

enum
{
	COLUMN_DATA,
	COLUMN_ICON,
	COLUMN_NAME,
	NUM_COLUMNS
};

static void galago_gtk_service_combo_box_class_init(GalagoGtkServiceComboBoxClass *klass);
static void galago_gtk_service_combo_box_init(GalagoGtkServiceComboBox *combobox);
static void galago_gtk_service_combo_box_finalize(GObject *obj);
static void galago_gtk_service_combo_box_destroy(GtkObject *obj);
static void service_added_cb(GalagoObject *core, GalagoService *service,
							 GalagoGtkServiceComboBox *combobox);
static void service_removed_cb(GalagoObject *core, GalagoService *service,
							   GalagoGtkServiceComboBox *combobox);

static GtkComboBoxClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

GType
galago_gtk_service_combo_box_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkServiceComboBoxClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_service_combo_box_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkServiceComboBox),
			0,
			(GInstanceInitFunc)galago_gtk_service_combo_box_init
		};

		type = g_type_register_static(GTK_TYPE_COMBO_BOX,
									  "GalagoGtkServiceComboBox", &info, 0);
	}

	return type;
}

static void
galago_gtk_service_combo_box_class_init(GalagoGtkServiceComboBoxClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GTK_OBJECT_CLASS(klass);

	gobject_class->finalize = galago_gtk_service_combo_box_finalize;

	object_class->destroy = galago_gtk_service_combo_box_destroy;

	signals[SERVICE_CHANGED] =
		g_signal_new("service_changed",
					 G_TYPE_FROM_CLASS(gobject_class),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoGtkServiceComboBoxClass,
									 service_changed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);
}

static void
combo_box_changed(GalagoGtkServiceComboBox *combobox, gpointer user_data)
{
	GalagoService *service;

	service = galago_gtk_service_combo_box_get_service(combobox);

	g_signal_emit(combobox, signals[SERVICE_CHANGED], 0, service);
}

static void
galago_gtk_service_combo_box_init(GalagoGtkServiceComboBox *combobox)
{
	GtkListStore *store;
	GtkCellRenderer *cell;

	combobox->priv = g_new0(GalagoGtkServiceComboBoxPriv, 1);

	store = gtk_list_store_new(NUM_COLUMNS, G_TYPE_POINTER,
							   GDK_TYPE_PIXBUF, G_TYPE_STRING);

	gtk_combo_box_set_model(GTK_COMBO_BOX(combobox),
							GTK_TREE_MODEL(store));

	g_object_unref(G_OBJECT(store));

	cell = gtk_cell_renderer_pixbuf_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combobox), cell, FALSE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combobox), cell,
								   "pixbuf", COLUMN_ICON,
								   NULL);

	cell = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combobox), cell, TRUE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combobox), cell,
								   "text", COLUMN_NAME,
								   NULL);

	g_signal_connect(G_OBJECT(combobox), "changed",
					 G_CALLBACK(combo_box_changed), NULL);

	combobox->priv->service_added_id =
		g_signal_connect(G_OBJECT(galago_get_core()), "service-added",
						 G_CALLBACK(service_added_cb), combobox);
	combobox->priv->service_removed_id =
		g_signal_connect(G_OBJECT(galago_get_core()), "service-removed",
						 G_CALLBACK(service_removed_cb), combobox);
}

static void
galago_gtk_service_combo_box_finalize(GObject *obj)
{
	GalagoGtkServiceComboBox *combobox;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_COMBO_BOX(obj));

	combobox = GALAGO_GTK_SERVICE_COMBO_BOX(obj);

	g_free(combobox->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_service_combo_box_destroy(GtkObject *obj)
{
	GalagoGtkServiceComboBox *combobox;
	GalagoObject *core;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_COMBO_BOX(obj));

	combobox = GALAGO_GTK_SERVICE_COMBO_BOX(obj);

	core = galago_get_core();

	GALAGO_GTK_DISCONNECT_HANDLER(core, combobox->priv->service_added_id);
	GALAGO_GTK_DISCONNECT_HANDLER(core, combobox->priv->service_removed_id);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
rebuild_list(GalagoGtkServiceComboBox *combobox)
{
	GtkListStore *store;
	GList *l;

	if (combobox->priv->rebuilding_list)
		return;

	combobox->priv->rebuilding_list = TRUE;

	store = GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(combobox)));

	gtk_list_store_clear(store);

	for (l = galago_get_services(GALAGO_REMOTE, TRUE);
		 l != NULL;
		 l = l->next)
	{
		GalagoService *service = (GalagoService *)l->data;
		GdkPixbuf *pixbuf;
		GtkTreeIter iter;

		pixbuf = galago_gdk_pixbuf_new_from_service_with_size(service,
			GTK_ICON_SIZE_MENU);

		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,
						   COLUMN_DATA, service,
						   COLUMN_ICON, pixbuf,
						   COLUMN_NAME, galago_service_get_name(service),
						   -1);

		if (pixbuf != NULL)
			g_object_unref(G_OBJECT(pixbuf));
	}

	combobox->priv->rebuilding_list = FALSE;
}

static void
service_added_cb(GalagoObject *core, GalagoService *service,
				 GalagoGtkServiceComboBox *combobox)
{
	GalagoService *cur_service;

	cur_service = galago_gtk_service_combo_box_get_service(combobox);

	rebuild_list(combobox);

	if (cur_service != NULL)
	{
		if (galago_gtk_service_combo_box_get_service(combobox) != cur_service)
			galago_gtk_service_combo_box_set_service(combobox, cur_service);
	}
	else
		gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), 0);
}

static void
service_removed_cb(GalagoObject *core, GalagoService *service,
				   GalagoGtkServiceComboBox *combobox)
{
	GalagoService *cur_service;

	cur_service = galago_gtk_service_combo_box_get_service(combobox);

	rebuild_list(combobox);

	if (cur_service == service)
	{
		if (galago_gtk_service_combo_box_get_service(combobox) != cur_service)
			galago_gtk_service_combo_box_set_service(combobox, cur_service);
	}
	else
		gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), 0);
}

GtkWidget *
galago_gtk_service_combo_box_new(void)
{
	GalagoGtkServiceComboBox *combobox;

	combobox = g_object_new(GALAGO_GTK_TYPE_SERVICE_COMBO_BOX, NULL);

	rebuild_list(combobox);

	gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), 0);

	return GTK_WIDGET(combobox);
}

void
galago_gtk_service_combo_box_set_service(GalagoGtkServiceComboBox *combobox,
										 const GalagoService *service)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean cont;

	g_return_if_fail(combobox != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_COMBO_BOX(combobox));
	g_return_if_fail(service != NULL);

	model = gtk_combo_box_get_model(GTK_COMBO_BOX(combobox));

	for (cont = gtk_tree_model_get_iter_first(model, &iter);
		 cont;
		 cont = gtk_tree_model_iter_next(model, &iter))
	{
		GalagoService *temp_service;

		gtk_tree_model_get(model, &iter,
						   COLUMN_DATA, &temp_service,
						   -1);

		if (temp_service == service)
			gtk_combo_box_set_active_iter(GTK_COMBO_BOX(combobox), &iter);
	}
}

GalagoService *
galago_gtk_service_combo_box_get_service(
	const GalagoGtkServiceComboBox *combobox)
{
	GalagoService *service = NULL;
	GtkTreeModel *model;
	GtkTreeIter iter;

	g_return_val_if_fail(combobox != NULL,                          NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_SERVICE_COMBO_BOX(combobox), NULL);

	model = gtk_combo_box_get_model(GTK_COMBO_BOX(combobox));

	if (gtk_combo_box_get_active_iter(GTK_COMBO_BOX(combobox), &iter))
	{
		gtk_tree_model_get(model, &iter,
						   COLUMN_DATA, &service,
						   -1);
	}

	return service;
}
