/**
 * @file libgalago-gtk/galago-gtk-account-menu-item.c Account menu item widget
 *
 * @Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-account-menu-item.h>
#include <libgalago-gtk/galago-gtk-account-label.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <gtk/gtk.h>

struct _GalagoGtkAccountMenuItemPriv
{
	GalagoAccount *account;

	GtkWidget *label;

	gulong account_destroy_id;
};

static void galago_gtk_account_menu_item_class_init(GalagoGtkAccountMenuItemClass *klass);
static void galago_gtk_account_menu_item_init(GalagoGtkAccountMenuItem *menu_item);
static void galago_gtk_account_menu_item_finalize(GObject *obj);
static void galago_gtk_account_menu_item_destroy(GtkObject *obj);

static GtkMenuItemClass *parent_class = NULL;

GType
galago_gtk_account_menu_item_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkAccountMenuItemClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_account_menu_item_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkAccountMenuItem),
			0,
			(GInstanceInitFunc)galago_gtk_account_menu_item_init
		};

		type = g_type_register_static(GTK_TYPE_IMAGE_MENU_ITEM,
									  "GalagoGtkAccountMenuItem", &info, 0);
	}

	return type;
}

static void
galago_gtk_account_menu_item_class_init(GalagoGtkAccountMenuItemClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GTK_OBJECT_CLASS(klass);

	gobject_class->finalize = galago_gtk_account_menu_item_finalize;

	object_class->destroy = galago_gtk_account_menu_item_destroy;
}

static void
galago_gtk_account_menu_item_init(GalagoGtkAccountMenuItem *menu_item)
{
	menu_item->priv = g_new0(GalagoGtkAccountMenuItemPriv, 1);
}

static void
galago_gtk_account_menu_item_finalize(GObject *obj)
{
	GalagoGtkAccountMenuItem *menu_item;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_MENU_ITEM(obj));

	menu_item = GALAGO_GTK_ACCOUNT_MENU_ITEM(obj);

	g_free(menu_item->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_account_menu_item_destroy(GtkObject *obj)
{
	GalagoGtkAccountMenuItem *menu_item;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_MENU_ITEM(obj));

	menu_item = GALAGO_GTK_ACCOUNT_MENU_ITEM(obj);

	GALAGO_GTK_DISCONNECT_HANDLER(menu_item->priv->account,
								  menu_item->priv->account_destroy_id);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
account_destroy_cb(GalagoAccount *account, GalagoGtkAccountMenuItem *menu_item)
{
	galago_gtk_account_menu_item_set_account(menu_item, NULL);
}

GtkWidget *
galago_gtk_account_menu_item_new(GalagoAccount *account)
{
	GalagoGtkAccountMenuItem *menu_item;
	GtkWidget *label;

	menu_item = g_object_new(GALAGO_GTK_TYPE_ACCOUNT_MENU_ITEM, NULL);

	label = galago_gtk_account_label_new();
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_container_add(GTK_CONTAINER(menu_item), label);
	gtk_widget_show(label);

	menu_item->priv->label = label;

	galago_gtk_account_menu_item_set_account(menu_item, account);

	return GTK_WIDGET(menu_item);
}

void
galago_gtk_account_menu_item_set_account(GalagoGtkAccountMenuItem *menu_item,
										 GalagoAccount *account)
{
	g_return_if_fail(menu_item != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_MENU_ITEM(menu_item));

	if (menu_item->priv->account == account)
		return;

	GALAGO_GTK_DISCONNECT_HANDLER(menu_item->priv->account,
								  menu_item->priv->account_destroy_id);

	menu_item->priv->account = account;

	if (account != NULL)
	{
		menu_item->priv->account_destroy_id =
			g_signal_connect(G_OBJECT(account), "destroy",
							 G_CALLBACK(account_destroy_cb), menu_item);
	}

	if (account != NULL)
	{
		GdkPixbuf *pixbuf;
		GalagoService *service;

		service = galago_account_get_service(account);

		pixbuf = galago_gdk_pixbuf_new_from_service_with_size(service,
			GTK_ICON_SIZE_MENU);

		if (pixbuf != NULL)
		{
			GtkWidget *image = gtk_image_new_from_pixbuf(pixbuf);

			g_object_unref(pixbuf);

			gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menu_item),
										  image);
		}
	}

	galago_gtk_account_label_set_account(
		GALAGO_GTK_ACCOUNT_LABEL(menu_item->priv->label), account);
}

GalagoAccount *
galago_gtk_account_menu_item_get_account(
	const GalagoGtkAccountMenuItem *menu_item)
{
	g_return_val_if_fail(menu_item != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_ACCOUNT_MENU_ITEM(menu_item), NULL);

	return menu_item->priv->account;
}
