// 
// Copyright (c) 2006-2008 Ben Motmans
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// Author(s):
//    Ben Motmans <ben.motmans@gmail.com>
//

using System;
using System.Collections.Generic;

namespace Anculus.Core
{
	public abstract class AbstractInserter : IInserter
	{
		public virtual int Insert<T> (List<T> list, T item)
		{
			return Insert<T> (list, Comparer<T>.Default, item);
		}

		public virtual int Insert<T> (List<T> list, SortOrder sortOrder, T item)
		{
			if (sortOrder == SortOrder.Ascending)
				return Insert<T> (list, Comparer<T>.Default, item);
			else
				return Insert<T> (list, new ReverseComparer<T> (Comparer<T>.Default), item);
		}

		public virtual int Insert<T> (List<T> list, IComparer<T> comparer, SortOrder sortOrder, T item)
		{
			if (comparer == null)
				throw new ArgumentNullException ("comparer");

			if (sortOrder == SortOrder.Ascending)
				return Insert<T> (list, comparer, item);
			else
				return Insert<T> (list, new ReverseComparer<T> (comparer), item);
		}

		public abstract int Insert<T> (List<T> list, IComparer<T> comparer, T item);

		public virtual int Insert<T> (ref T[] array, T item)
		{
			return Insert<T> (ref array, Comparer<T>.Default, item);
		}

		public virtual int Insert<T> (ref T[] array, SortOrder sortOrder, T item)
		{
			if (sortOrder == SortOrder.Ascending)
				return Insert<T> (ref array, Comparer<T>.Default, item);
			else
				return Insert<T> (ref array, new ReverseComparer<T> (Comparer<T>.Default), item);
		}

		public virtual int Insert<T> (ref T[] array, IComparer<T> comparer, SortOrder sortOrder, T item)
		{
			if (comparer == null)
				throw new ArgumentNullException ("comparer");

			if (sortOrder == SortOrder.Ascending)
				return Insert<T> (ref array, comparer, item);
			else
				return Insert<T> (ref array, new ReverseComparer<T> (comparer), item);
		}

		public abstract int Insert<T> (ref T[] array, IComparer<T> comparer, T item);

		protected static T[] ResizeArray<T> (T[] array, int leaveBlank)
		{
			int len = array.Length;
			T[] narr = new T[len + 1];
			if (leaveBlank < 0 || leaveBlank > len) {
				Array.Copy (array, narr, array.Length);
			} else {
				if (leaveBlank == 0) {
					Array.Copy (array, 0, narr, 1, len);
				} else {
					Array.Copy (array, 0, narr, 0, leaveBlank);
					Array.Copy (array, leaveBlank, narr, leaveBlank + 1, len - leaveBlank);
				}
			}
			return narr;
		}
		
		protected static void Insert<T> (ref T[] array, int index, T item)
		{
			int len = array.Length;
			T[] narr = new T[len + 1];
			
			if (len == 0) {
				narr[0] = item;
			} else if (index == 0) { //first item in the array
				narr[0] = item;
				Array.Copy (array, 0, narr, 1, len);
			} else if (index == len) { //last item in the array
				Array.Copy (array, 0, narr, 0, len);
				narr[len] = item;
			} else { //the new item is in the middle of the array
				Array.Copy (array, 0, narr, 0, index);
				narr[index] = item;
				Array.Copy (array, index, narr, index + 1, len - index);
			}
			
			array = narr;
		}
	}
}
