// 
// Copyright (c) 2006-2008 Ben Motmans
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// Author(s):
//    Ben Motmans <ben.motmans@gmail.com>
//

using System;
using System.Collections.Generic;

namespace Anculus.Core
{
	public abstract class AbstractSetSearchAlgorithm : ISetSearchAlgorithm
	{
		public virtual SearchResult[] SearchAll (string text, params string[] keywords)
		{
			return SearchAll (text, 0, int.MaxValue, keywords);
		}

		public virtual SearchResult[] SearchAll (string text, int start, params string[] keywords)
		{
			return SearchAll (text, start, int.MaxValue, keywords);
		}

		public abstract SearchResult[] SearchAll (string text, int start, int count, params string[] keywords);

		public virtual SearchResult SearchFirst (string text, params string[] keywords)
		{
			return SearchFirst (text, 0, keywords);
		}

		public abstract SearchResult SearchFirst (string text, int start, params string[] keywords);

		public virtual bool Contains (string text, params string[] keywords)
		{
			return Contains (text, 0, keywords);
		}

		public virtual bool Contains (string text, int start, params string[] keywords)
		{
			return SearchFirst (text, start, keywords) != SearchResult.Empty;
		}

		protected static void CheckKeywords (params string[] keywords)
		{
			if (keywords == null)
				throw new ArgumentNullException ("keywords");
			if (keywords.Length == 0)
				throw new ArgumentException ("keywords");
			
			foreach (string keyword in keywords) {
				if (string.IsNullOrEmpty (keyword))
					throw new ArgumentException ("The keyword set cannot contain null references or empty strings.");
			}
		}
		
		[System.Diagnostics.DebuggerHidden]
		protected static void CheckArguments (string text, int start, int count)
		{
			if (text == null)
				throw new ArgumentNullException ("text");
			if (text.Length == 0)
				throw new ArgumentException ("text");

			if (start < 0)
				throw new ArgumentOutOfRangeException ("start");
			if (start >= text.Length)
				throw new ArgumentOutOfRangeException ("start");
			
			if (count <= 0)
				throw new ArgumentOutOfRangeException ("count");
		}
	}
}