// =============================================================================
//
//      --- kvi_mdi_child.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Opaque move & resize code by Falk Brettschneider (gigafalk@geocities.com)
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviMDIChild"

#define _KVI_MDI_CHILD_CPP_

#include <qcursor.h>
#include <qpainter.h>
#include <qtooltip.h>

#include "kvi_application.h"
#include "kvi_cursor.h"
#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_locale.h"
#include "kvi_mdi_button.h"
#include "kvi_mdi_caption.h"
#include "kvi_mdi_child.h"
#include "kvi_mdi_manager.h"
#include "kvi_xutils.h"

#include <X11/Xlib.h> // For XGrabPointer, GrabModeAsync, etc.
#if defined(FocusIn)
	#undef FocusIn // Defined in X.h
#endif

#define KVI_MDI_NORESIZE           0
#define KVI_MDI_RESIZE_TOP         1
#define KVI_MDI_RESIZE_LEFT        2
#define KVI_MDI_RESIZE_RIGHT       4
#define KVI_MDI_RESIZE_BOTTOM      8
#define KVI_MDI_RESIZE_TOPLEFT     (1|2)
#define KVI_MDI_RESIZE_TOPRIGHT    (1|4)
#define KVI_MDI_RESIZE_BOTTOMLEFT  (8|2)
#define KVI_MDI_RESIZE_BOTTOMRIGHT (8|4)

KviMdiChild::KviMdiChild(KviMdiManager *parent, const char *name)
	: QFrame(parent, name)
{
	m_pClient   = 0;
	m_pCaption  = new KviMdiCaption(this, "mdi_caption");
	m_pManager  = parent;

	m_pMinimize = new KviMdiButton(m_pCaption, KviMdiButton::Minimize, "minimize_button");
	connect(m_pMinimize, SIGNAL(buttonPressed()), this, SLOT(minimizePressed()));
	QToolTip::add(m_pMinimize, _CHAR_2_QSTRING(_i18n_("Minimize")));

	m_pMaximize = new KviMdiButton(m_pCaption, KviMdiButton::Maximize, "maximize_button");
	connect(m_pMaximize, SIGNAL(buttonPressed()), this, SLOT(maximizePressed()));
	QToolTip::add(m_pMaximize, _CHAR_2_QSTRING(_i18n_("Maximize")));

	m_pClose    = new KviMdiButton(m_pCaption, KviMdiButton::Close, "close_button");
	connect(m_pClose, SIGNAL(buttonPressed()), this, SLOT(closePressed()));
	QToolTip::add(m_pClose, _CHAR_2_QSTRING(_i18n_("Close")));

	setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
	setFocusPolicy(ClickFocus);

	m_state             = Normal;
	m_iResizeCorner     = KVI_MDI_NORESIZE;
	m_iLastCursorCorner = KVI_MDI_NORESIZE;

	setMouseTracking(true);
	setMinimumSize(QSize(KVI_MDI_CHILD_MIN_WIDTH, KVI_MDI_CHILD_MIN_HEIGHT));
	m_bResizeMode = false;
}

// ============ ~KviMdiChild ============
KviMdiChild::~KviMdiChild()
{
	// Nothing here
}

// ============ mousePressEvent =============
void KviMdiChild::mousePressEvent(QMouseEvent *e)
{
	m_iResizeCorner = getResizeCorner(e->pos().x(), e->pos().y());
	if( m_iResizeCorner != KVI_MDI_NORESIZE ) {
		if( m_pManager->m_bOpaqueResize ) {
			grabMouse(getResizeCursor(m_iResizeCorner));
			m_bResizeMode = true;
		} else resizeWindow(m_iResizeCorner);
	}
}

// ============ mouseReleaseEvent ==============
void KviMdiChild::mouseReleaseEvent(QMouseEvent *)
{
	m_iResizeCorner     = KVI_MDI_NORESIZE;
	m_iLastCursorCorner = KVI_MDI_NORESIZE;
	if( m_bResizeMode ) {
		m_bResizeMode = false;
		releaseMouse();
	}
	if( KviApplication::overrideCursor() ) KviApplication::restoreOverrideCursor();
}

// ============= leaveEvent ===============
void KviMdiChild::leaveEvent(QEvent *)
{
	if( !m_bResizeMode ) {
		m_iResizeCorner     = KVI_MDI_NORESIZE;
		m_iLastCursorCorner = KVI_MDI_NORESIZE;
		if( KviApplication::overrideCursor() ) KviApplication::restoreOverrideCursor();
	} else {
		if( m_iResizeCorner != KVI_MDI_NORESIZE ) resizeWindowOpaque(m_iResizeCorner);
	}
}

QCursor KviMdiChild::getResizeCursor(int resizeCorner)
{
	switch( resizeCorner ) {
		case KVI_MDI_RESIZE_LEFT:
		case KVI_MDI_RESIZE_RIGHT:
			return KviCursor::sizeHorCursor();
			break;
		case KVI_MDI_RESIZE_TOP:
		case KVI_MDI_RESIZE_BOTTOM:
			return KviCursor::sizeVerCursor();
			break;
		case KVI_MDI_RESIZE_TOPLEFT:
		case KVI_MDI_RESIZE_BOTTOMRIGHT:
			return KviCursor::sizeFDiagCursor();
			break;
		case KVI_MDI_RESIZE_BOTTOMLEFT:
		case KVI_MDI_RESIZE_TOPRIGHT:
			return KviCursor::sizeBDiagCursor();
			break;
		default:
			return KviCursor::arrowCursor();
			break;
	}
}

// ============= setResizeCursor ===============
void KviMdiChild::setResizeCursor(int resizeCorner)
{
	if( resizeCorner == m_iLastCursorCorner ) return; // Do not do it twice
	m_iLastCursorCorner = resizeCorner;
	if( resizeCorner == KVI_MDI_NORESIZE ) {
		if( KviApplication::overrideCursor() ) KviApplication::restoreOverrideCursor();
	} else KviApplication::setOverrideCursor(getResizeCursor(resizeCorner), true);
}

// ============= mouseMoveEvent ===============
void KviMdiChild::mouseMoveEvent(QMouseEvent *e)
{
	if( e->state() & LeftButton ) {
		if( m_iResizeCorner ) {
			if( m_bResizeMode )
				resizeWindowOpaque(m_iResizeCorner);
			else
				resizeWindow(m_iResizeCorner);
		}
	} else setResizeCursor(getResizeCorner(e->pos().x(), e->pos().y()));
}

void KviMdiChild::calculateResizeRect(int resizeCorner, QPoint mousePos, QRect &resizeRect, int minWidth, int minHeight)
{
	switch( resizeCorner ) {
		case KVI_MDI_RESIZE_LEFT:
			resizeRect.setLeft(mousePos.x() - 1);
			if( resizeRect.width() < minWidth )
				resizeRect.setLeft(resizeRect.right() - minWidth);
			break;
		case KVI_MDI_RESIZE_RIGHT:
			resizeRect.setRight(mousePos.x() + 1);
			if( resizeRect.width() < minWidth )
				resizeRect.setRight(resizeRect.left() + minWidth);
			break;
		case KVI_MDI_RESIZE_TOP:
			resizeRect.setTop(mousePos.y() - 1);
			if( resizeRect.height() < minHeight )
				resizeRect.setTop(resizeRect.bottom() - minHeight);
			break;
		case KVI_MDI_RESIZE_BOTTOM:
			resizeRect.setBottom(mousePos.y() + 1);
			if( resizeRect.height() < minHeight )
				resizeRect.setBottom(resizeRect.top() + minHeight);
			break;
		case KVI_MDI_RESIZE_BOTTOMRIGHT:
			resizeRect.setBottom(mousePos.y() + 1);
			if( resizeRect.height() < minHeight )
				resizeRect.setBottom(resizeRect.top() + minHeight);
			resizeRect.setRight(mousePos.x() + 1);
			if( resizeRect.width() < minWidth )
				resizeRect.setRight(resizeRect.left() + minWidth);
			break;
		case KVI_MDI_RESIZE_TOPRIGHT:
			resizeRect.setTop(mousePos.y() - 1);
			if( resizeRect.height() < minHeight )
				resizeRect.setTop(resizeRect.bottom() - minHeight);
			resizeRect.setRight(mousePos.x() + 1);
			if( resizeRect.width() < minWidth )
				resizeRect.setRight(resizeRect.left() + minWidth);
			break;
		case KVI_MDI_RESIZE_BOTTOMLEFT:
			resizeRect.setBottom(mousePos.y() + 1);
			if( resizeRect.height() < minHeight )
				resizeRect.setBottom(resizeRect.top() + minHeight);
			resizeRect.setLeft(mousePos.x() - 1);
			if( resizeRect.width() < minWidth )
				resizeRect.setLeft(resizeRect.right() - minWidth);
			break;
		case KVI_MDI_RESIZE_TOPLEFT:
			resizeRect.setTop(mousePos.y() - 1);
			if( resizeRect.height() < minHeight )
				resizeRect.setTop(resizeRect.bottom() - minHeight);
			resizeRect.setLeft(mousePos.x() - 1);
			if( resizeRect.width() < minWidth )
				resizeRect.setLeft(resizeRect.right() - minWidth);
			break;
	}
}

void KviMdiChild::calculateMinimumSize(int &minWidth, int &minHeight)
{
	if( m_pClient ) {
		minWidth  = m_pClient->minimumSize().width() + KVI_MDI_CHILD_DOUBLE_BORDER;
		minHeight = m_pClient->minimumSize().height()+ KVI_MDI_CHILD_DOUBLE_BORDER + m_pCaption->heightHint() + KVI_MDI_CHILD_SEPARATOR;
	}
	if( minWidth  < KVI_MDI_CHILD_MIN_WIDTH ) minWidth  = KVI_MDI_CHILD_MIN_WIDTH;
	if( minHeight < KVI_MDI_CHILD_MIN_WIDTH ) minHeight = KVI_MDI_CHILD_MIN_HEIGHT;
}

void KviMdiChild::resizeWindowOpaque(int resizeCorner)
{
	int minWidth  = 0;
	int minHeight = 0;
	QRect resizeRect(x(), y(), width(), height());
	calculateMinimumSize(minWidth, minHeight);
	QPoint mousePos = m_pManager->mapFromGlobal(QCursor::pos());
	calculateResizeRect(resizeCorner, mousePos, resizeRect, minWidth, minHeight);
	setGeometry(resizeRect.x(), resizeRect.y(), resizeRect.width(), resizeRect.height());
	if( m_state == Maximized ) {
		m_state = Normal;
		m_pMaximize->setType(KviMdiButton::Maximize);
		m_pManager->childRestored(this, true);
	}
}

void KviMdiChild::resizeWindow(int resizeCorner)
{
	m_iResizeCorner = KVI_MDI_NORESIZE;
	QRect resizeRect(x(), y(), width(), height());

	int grab = XGrabPointer(qt_xdisplay(), m_pManager->winId(), true, (ButtonReleaseMask), GrabModeAsync, GrabModeAsync, m_pManager->winId(), None, CurrentTime);

	QPainter pa(m_pManager->parentWidget());
	kvi_drawDragRectangle(&pa, resizeRect);

	int minWidth  = 0;
	int minHeight = 0;

	calculateMinimumSize(minWidth, minHeight);

	XEvent ev;
	QPoint oldMousePos = m_pManager->mapFromGlobal(QCursor::pos());

	while( !XCheckMaskEvent(qt_xdisplay(), ButtonReleaseMask, &ev) ) {
		Window dw1, dw2;
		int t1, t2, newX, newY;
		unsigned int t3;
		XQueryPointer(qt_xdisplay(), qt_xrootwin(), &dw1, &dw2, &newX, &newY, &t1, &t2, &t3);
		QPoint mousePos = m_pManager->mapFromGlobal(QPoint(newX, newY));
		if( oldMousePos != mousePos ) {
			kvi_drawDragRectangle(&pa, resizeRect);
			calculateResizeRect(resizeCorner, mousePos, resizeRect, minWidth, minHeight);
			kvi_drawDragRectangle(&pa, resizeRect);
			oldMousePos = mousePos;
		}
	}
	kvi_drawDragRectangle(&pa, resizeRect);

	if( grab == GrabSuccess ) XUngrabPointer(qt_xdisplay(), CurrentTime);

	setGeometry(resizeRect.x(), resizeRect.y(), resizeRect.width(), resizeRect.height());
	if( KviApplication::overrideCursor() ) KviApplication::restoreOverrideCursor();

	if( m_state == Maximized ) {
		m_state = Normal;
		m_pMaximize->setType(KviMdiButton::Maximize);
		m_pManager->childRestored(this, true);
	}
	// Resync Qt
	ev.xany.window = winId();
	XSendEvent(ev.xany.display, ev.xany.window, false, ButtonReleaseMask, &ev);
}

void KviMdiChild::moveWindowOpaque(QPoint diff)
{
	if( m_state == Maximized ) {
		m_state = Normal;
		m_pMaximize->setType(KviMdiButton::Maximize);
	}
	setGeometry(x()+diff.x(), y()+diff.y(), width(), height());
}

// =========== moveWindow ==============
void KviMdiChild::moveWindow()
{
	// This is called by the caption on mousePressEvent
	QRect absoluteWinRect(x(), y(), width(), height());
	// Grab the pointer if possible
	KviApplication::setOverrideCursor(KviCursor::sizeAllCursor(), true);
	int grab = XGrabPointer(
		qt_xdisplay(), m_pManager->winId(), false, ButtonReleaseMask, GrabModeAsync,
		GrabModeAsync, m_pManager->winId(), None, CurrentTime
	);

	QPainter pa(m_pManager->parentWidget());
	kvi_drawDragRectangle(&pa, absoluteWinRect);

	XEvent ev;
	Window dummyWin1, dummyWin2;
	int dummy1, dummy2;
	unsigned int dummy3;
	QPoint oldMousePos = QCursor::pos();
	int mouseNewX, mouseNewY;
	QPoint mousePos;

	while( (!XCheckMaskEvent(qt_xdisplay(), (ButtonReleaseMask), &ev)) ) {
		XQueryPointer(
			qt_xdisplay(), qt_xrootwin(), &dummyWin1, &dummyWin2, &mouseNewX, &mouseNewY, &dummy1, &dummy2, &dummy3
		);

		mousePos = QPoint(mouseNewX, mouseNewY);
		if( oldMousePos != mousePos ) {
			kvi_drawDragRectangle(&pa, absoluteWinRect);
			int nX = absoluteWinRect.x() + (mousePos.x() - oldMousePos.x());
			nX = QMAX(nX, KVI_MDI_CHILD_MIN_VISIBLE_EDGE - width());
			nX = QMIN(nX, parentWidget()->width() - KVI_MDI_CHILD_MIN_VISIBLE_EDGE);
			int nY = absoluteWinRect.y() + (mousePos.y() - oldMousePos.y());
			nY = QMAX(nY, 0);
			nY = QMIN(nY, parentWidget()->height() - KVI_MDI_CHILD_MIN_VISIBLE_EDGE);
			absoluteWinRect = QRect(nX, nY, absoluteWinRect.width(), absoluteWinRect.height());
			kvi_drawDragRectangle(&pa, absoluteWinRect);
			oldMousePos = mousePos;
		}
	}

	kvi_drawDragRectangle(&pa, absoluteWinRect);

	if( grab == GrabSuccess ) XUngrabPointer(qt_xdisplay(), CurrentTime);

	if( m_state == Maximized ) {
		m_state = Normal;
		m_pMaximize->setType(KviMdiButton::Maximize);
	}
	setGeometry(absoluteWinRect.x(), absoluteWinRect.y(), width(), height());
	KviApplication::restoreOverrideCursor();
	ev.xany.window = winId();
	XSendEvent(ev.xany.display, ev.xany.window, false, ButtonReleaseMask, &ev);
}

// ================= getResizeCorner =============
int KviMdiChild::getResizeCorner(int ax, int ay)
{
	int ret = KVI_MDI_NORESIZE;
	if( (ax > 0         ) && (ax < (KVI_MDI_CHILD_BORDER + 2))              ) ret |= KVI_MDI_RESIZE_LEFT;
	if( (ax < width()   ) && (ax > (width() - (KVI_MDI_CHILD_BORDER+2)))    ) ret |= KVI_MDI_RESIZE_RIGHT;
	if( (ay > 0         ) && (ay < (KVI_MDI_CHILD_BORDER + 2))              ) ret |= KVI_MDI_RESIZE_TOP;
	if( (ay < (height())) && (ay > (height() - (KVI_MDI_CHILD_BORDER + 2))) ) ret |= KVI_MDI_RESIZE_BOTTOM;
	return ret;
}

// ============= maximizePressed ============

void KviMdiChild::maximizePressed()
{
	switch( m_state ) {
		case Maximized: setState(Normal);    break;
		case Normal:    setState(Maximized); break;
		case Minimized: setState(Maximized); break;
	}
}

// ============= minimizePressed ============
void KviMdiChild::minimizePressed()
{
	switch( m_state ) {
		case Minimized: setState(Normal);    break;
		case Normal:    setState(Minimized); break;
		case Maximized: setState(Minimized); break;
	}
}

// ============= closePressed ============
void KviMdiChild::closePressed()
{
	if( m_pClient ) m_pClient->close();
}

// ============ setState =================
void KviMdiChild::setState(MdiWindowState state, bool bAnimate)
{
	if( m_state == Normal ) { // Save the current rect
		m_restoredRect = QRect(x() > 0 ? x() : 0, y() > 0 ? y() : 0, width(), height());
	}
	QRect begin(x(), y(), width(), height());
	QRect end = begin;
	switch( state ) {
		case Normal: {
			switch( m_state ) {
				case Maximized:
					m_state = state;
					if( bAnimate ) m_pManager->animate(begin, m_restoredRect);
					m_pMaximize->setType(KviMdiButton::Maximize);
					setGeometry(m_restoredRect);
					m_pManager->childRestored(this, true);
					break;
				case Minimized:
					m_state = state;
					begin = QRect(x() + (width() >> 1), y() + (height() >> 1), 1, 1);
					if( bAnimate ) m_pManager->animate(begin, end);
					m_pMinimize->setType(KviMdiButton::Minimize);
					show();
					m_pManager->childRestored(this, false);
					break;
				case Normal:
					break;
			}
		} break;
		case Maximized: {
			end = QRect(
				-KVI_MDI_CHILD_MAXIMIZED_HIDDEN_EDGE,
				-(KVI_MDI_CHILD_BORDER + m_pCaption->heightHint() + 2),
				m_pManager->width()  + KVI_MDI_CHILD_DOUBLE_MAXIMIZED_HIDDEN_EDGE,
				m_pManager->height() + KVI_MDI_CHILD_BORDER + m_pCaption->heightHint() + KVI_MDI_CHILD_MAXIMIZED_HIDDEN_EDGE+3
			);
			switch( m_state ) {
				case Minimized:
					m_state = state;
					begin = QRect(x() + (width() >> 1), y() + (height() >> 1), 1, 1);
					if( bAnimate ) m_pManager->animate(begin, end);
					setGeometry(end);
					m_pMaximize->setType(KviMdiButton::Restore);
					m_pMinimize->setType(KviMdiButton::Minimize);
					show();
					m_pManager->childMaximized(this, true);
					break;
				case Normal:
					m_state = state;
					if( bAnimate ) m_pManager->animate(begin, end);
					m_pMaximize->setType(KviMdiButton::Restore);
					setGeometry(end);
					m_pManager->childMaximized(this, false);
					break;
				case Maximized:
					// HACK, needed because of wrong rect in window properties
					setGeometry(end);
					m_pMaximize->setType(KviMdiButton::Restore);
					m_pMinimize->setType(KviMdiButton::Minimize);
					m_pManager->childMaximized(this, false);
					break;
			}
		} break;
		case Minimized: {
			end = QRect(x() + (width() >> 1), y() + (height() >> 1), 1, 1);
			switch( m_state ) {
				case Maximized:
					m_state = state;
					hide();
					if( bAnimate ) m_pManager->animate(begin, end);
					setGeometry(m_restoredRect);
					m_pMinimize->setType(KviMdiButton::Restore);
					m_pMaximize->setType(KviMdiButton::Maximize);
					m_pManager->childMinimized(this, true);
					break;
				case Normal:
					m_state = state;
					hide();
					if( bAnimate ) m_pManager->animate(begin, end);
					m_pMinimize->setType(KviMdiButton::Restore);
					m_pManager->childMinimized(this, false);
					break;
				case Minimized:
					break;
			}
		} break;
	}
}

// ============ setCaption ===============
void KviMdiChild::setCaption(const char *text)
{
	m_pCaption->setCaption(text);
	m_pManager->fillWindowMenu();
}

const char *KviMdiChild::caption()
{
	return m_pCaption->m_szCaption;
}

// ============ enableClose ==============
void KviMdiChild::enableClose(bool bEnable)
{
	m_pClose->setEnabled(bEnable);
	m_pClose->repaint(false);
}

bool KviMdiChild::closeEnabled()
{
	return m_pClose->isEnabled();
}

// ============ setIconPointer =============
void KviMdiChild::setIconPointer(QPixmap *ptr)
{
	m_pCaption->m_pIcon = ptr;
	m_pCaption->repaint(false);
}

// ============ setClient ============
void KviMdiChild::setClient(QWidget *w)
{
	__range_valid(m_pClient == 0);
	__range_valid(w != 0);
	m_pClient = w;
	// Resize to match the client
	int clientYPos = m_pCaption->heightHint() + KVI_MDI_CHILD_SEPARATOR + KVI_MDI_CHILD_BORDER;
	resize(w->width() + KVI_MDI_CHILD_DOUBLE_BORDER, w->height() + KVI_MDI_CHILD_BORDER + clientYPos);
	// Reparent if needed
	if( w->parent() != this ) {
		// Reparent to this widget, no flags, point, show it
		QPoint pnt2(KVI_MDI_CHILD_BORDER, clientYPos);
		w->reparent(this, pnt2, true);
	} else w->move(KVI_MDI_CHILD_BORDER, clientYPos);
	setFocusProxy(w);
	 m_pCaption->setFocusProxy(w);
	m_pMinimize->setFocusProxy(w);
	m_pMaximize->setFocusProxy(w);
	   m_pClose->setFocusProxy(w);
	linkChildren(w);
	if( m_pClient->minimumSize().width()  > KVI_MDI_CHILD_MIN_WIDTH &&
		m_pClient->minimumSize().height() > KVI_MDI_CHILD_MIN_HEIGHT ) {
		setMinimumWidth(m_pClient->minimumSize().width()   + KVI_MDI_CHILD_DOUBLE_BORDER);
		setMinimumHeight(m_pClient->minimumSize().height() + KVI_MDI_CHILD_DOUBLE_BORDER + m_pCaption->heightHint() + KVI_MDI_CHILD_SEPARATOR);
	}
	KviStr tmp(KviStr::Format, "mdi_child_%s", w->name());
	setName(tmp.ptr());
}

// ============ unsetClient ============
void KviMdiChild::unsetClient()
{
	__range_valid(m_pClient != 0);
	if( !m_pClient ) return;
	// Reparent to desktop widget, no flags, point, show it
	unlinkChildren(m_pClient);
	setFocusProxy(0); // Remove the focus proxy...
	// Kewl... the reparent function has a small prob now..
	// The new toplevel widgets does not get reenabled for DnD
	m_pClient->reparent(0, QPoint(0, 0), true);
	m_pClient = 0;
	setName("mdi_child");
}

KviMdiChild::MdiWindowState KviMdiChild::state()
{
	return m_state;
}

void KviMdiChild::updateRects()
{
	resizeEvent(0);
}

// ============== linkChildren =============
void KviMdiChild::linkChildren(QWidget *w)
{
	// Recursive: installs an event filter to all children of the widget w
	QPtrList<QObject> *list = (QPtrList<QObject> *) (w->children());
	if( list ) {
		for( unsigned int i = 0; i < list->count(); i++ ) {
			QObject *o = list->at(i);
			if( o->inherits("QWidget") ) linkChildren(((QWidget *) o));
		}
	}
	w->installEventFilter(this);
}

// ============== unlinkChildren =============
void KviMdiChild::unlinkChildren(QWidget *w)
{
	// Recursive: installs an event filter to all children of the widget w
	QPtrList<QObject> *list = (QPtrList<QObject> *) (w->children());
	if( list ) {
		for( unsigned int i = 0; i < list->count(); i++ ) {
			QObject *o = list->at(i);
			if( o->inherits("QWidget") ) unlinkChildren(((QWidget *) o));
		}
	}
	w->removeEventFilter(this);
}

// ============== resizeEvent ===============
void KviMdiChild::resizeEvent(QResizeEvent *)
{
	// Resize the caption
	int captionHeight = m_pCaption->heightHint();
	int captionWidth = width() - KVI_MDI_CHILD_DOUBLE_BORDER;
	m_pCaption->setGeometry(
		KVI_MDI_CHILD_BORDER,
		KVI_MDI_CHILD_BORDER,
		captionWidth,
		captionHeight
	);
	// The buttons are caption children
	m_pClose->setGeometry((captionWidth-captionHeight) + 1, 1,captionHeight - 2, captionHeight - 2);
	m_pMaximize->setGeometry((captionWidth - (captionHeight << 1)) + 2, 1, captionHeight - 2, captionHeight - 2);
	m_pMinimize->setGeometry((captionWidth - (captionHeight*3)) + 3, 1, captionHeight - 2, captionHeight - 2);
	// Resize the client
	if( m_pClient ) {
		m_pClient->setGeometry(
			KVI_MDI_CHILD_BORDER,
			KVI_MDI_CHILD_BORDER + captionHeight + KVI_MDI_CHILD_SEPARATOR,
			captionWidth,
			height() - (KVI_MDI_CHILD_DOUBLE_BORDER + captionHeight + KVI_MDI_CHILD_SEPARATOR)
		);
	}
}

// ============== eventFilter ==============
bool KviMdiChild::eventFilter(QObject *o, QEvent *e)
{
	switch( e->type() ) {
		case QEvent::FocusIn: {
			// Still not sure... but seems a Qt bug. Even_FocusIn from a HIDDEN NoBackground widget!
			__range_valid(isVisible());

			m_pCaption->setActive(true);
			m_pManager->setTopChild(this, false);
			break;
		}
		case QEvent::Enter: {
			if( KviApplication::overrideCursor() )
				KviApplication::restoreOverrideCursor();
			break;
		}
		default: // ignore
			break;
	}
	return false;
}

// ============= focusInEvent ===============
void KviMdiChild::focusInEvent(QFocusEvent *)
{
	// We gained focus by click, tab or from the caption label.
	// Bring this child to top
	m_pCaption->setActive(true);
	m_pManager->setTopChild(this, false); // Do not focus now...
	// The client is our focusProxy! it should be focused by Qt!
#ifdef _KVI_DEBUG_CLASS_NAME_
	__range_valid(focusProxy() == m_pClient);
#endif
}

// =============== activate =================
void KviMdiChild::activate(bool bSetFocus)
{
	if( !m_pCaption->active() )
		m_pCaption->setActive(true);
	if( m_pManager->topChild() != this )
		m_pManager->setTopChild(this, bSetFocus);
	else if( bSetFocus )
		setFocus();
}

#include "m_kvi_mdi_child.moc"
