/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "ripping.h"
#include "settings_general.h"
#define AND  &&
#define OR  ||
#define NOT  !
#define EQUAL  ==

#include <KGlobal>
#include <KLocale>
#include <QDir>
#include <QFileInfo>
#include "proxyinfo.h"

ripping::ripping(QObject *parent) : streamripper_base(parent)
{
  internal_splitBehavior = QString::SkipEmptyParts;  /* makes sure that
  interpretate_console_output() doesn't get empty lines (which
  wouldn't make any sense, but only consume CPU time) */

  /* The internal variables who hold the properties must get initialized.
  *
  *  We use resetStreamripperProperties(), which uses the resetXXX()
  *  functions which use the setXXX() functions.
  *
  *  The resetStreamripperProperties() function and the resetXXX()
  *  functions are virtual. This means they are reimplemented in
  *  the inherited class "radiostation" to write directly to the
  *  config file. But it isn't desired. Instead we need only an
  *  initialation of the "PropertyValue internal_XXX;" members,
  *  and this initialation is necessary for the inherited class
  *  "get_stream_info" who relays on a working (and with "unset"
  *  initialised) properties system.
  *
  *  However we can safly call the reset functions in the constructor,
  *  because during the constructor call the reimplementations of
  *  the virtual functions are not yet available. */
  resetStreamripperProperties(); // this will emit signals just for the initialatation
                                  //(what's undesired),
                                  // but that's no problem, because during this constructor
                                  // is running, there can't still any slot be conneted!

  QObject::connect(&m_process,
                   SIGNAL(stateChanged(QProcess::ProcessState)),
                   this,
                   SLOT(streamripperStateChange(QProcess::ProcessState)));
  QObject::connect(&m_process,
                   SIGNAL(error(QProcess::ProcessError)),
                   this,
                   SLOT(errorOccured(QProcess::ProcessError)));
  QObject::connect(this,
                   SIGNAL(bitrateChanged(qlonglong, PropertyValue)),
                   this,
                   SLOT(emit_metaInterval_milliSecondsChanged()));
}

ripping::~ripping()
{
}

void ripping::emit_metaInterval_milliSecondsChanged()
{
  emit metaInterval_milliSecondsChanged(index(), metaInterval_milliSeconds());
}

void ripping::resetStreamripperProperties()
{
  setBitrate(default_value_of_bitrate());
  setDataSize(default_value_of_dataSize());
  setError(default_value_of_error());
  setMetaInterval(default_value_of_metaInterval());
  setRelayPort(default_value_of_relayPort());
  setServerName(default_value_of_serverName());
  setSong(default_value_of_song());
  setStatus(default_value_of_status());
  setStreamName(default_value_of_streamName());
}

void ripping::interpretate_console_output(QStringList & stringList)
{
  // variables
  QString my_line;
  // recognized data
  // (statusSet == true) means that setStatus() AND setSong() AND setDataSize() were called!
  bool statusSet=false;
  bool relayPortSet=false;
  bool streamNameSet=false;
  bool serverNameSet=false;
  bool bitrateSet=false;
  bool metaIntervalSet=false;
  bool errorSet=false;
  // help variables
  QString help_string;
  bool okay;
  qint64 helper_qint64;

  // code
  // recognize data
  while (!stringList.isEmpty()) {
    my_line = stringList.takeLast();  // gives back the last list entry and deletes it

    /* We can't do "my_line=my_line.toLower();" here (what would be very
    practical) because we need later the original string with upper AND lower cases when
    we want to recognize a song title! */
    if (my_line.toLower().startsWith(QString("[ripping...    ] "))) {
      if (!statusSet) {
        setStatus(is_ripping);
        // remove "[ripping...    ] " from the begin of the string:
        my_line = my_line.remove(0, 17);
        helper_interpretate_metainfo_and_datasize(my_line);
        statusSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("[buffering - | ] "))) {
      if (!statusSet) {
        setStatus(is_buffering);
        // remove "[buffering - | ] " from the begin of the string:
        my_line = my_line.remove(0, 17);
        setSong(my_line);
        setDataSize(0);
        statusSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("[skipping...   ] "))) {
      if (!statusSet) {
        setStatus(is_skipping);
        // remove "[skipping...   ] " from the begin of the string:
        my_line = my_line.remove(0, 17);
        helper_interpretate_metainfo_and_datasize(my_line);
        statusSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("connecting")) ||
              my_line.toLower().startsWith(QString("[getting track name"))) {
      if (!statusSet) {
        setStatus(is_connecting);
        setSong(default_value_of_song());
        setDataSize(default_value_of_dataSize());
        statusSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("shutting down")) ||
              my_line.toLower().startsWith(QString("bye.."))) {
      if (!statusSet) {
        setStatus(is_saving);
        setSong(default_value_of_song());
        setDataSize(default_value_of_dataSize());
        statusSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("relay port: "))) {
      if (!relayPortSet) {
        helper_qint64 = my_line.right(my_line.size()-12).toLongLong(&okay);
        if (NOT okay) {
          setRelayPort(-1);
        } else {
          setRelayPort(helper_qint64);
        };
        relayPortSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("stream: "))) {
      if (!streamNameSet) {
        setStreamName(my_line.right(my_line.size()-8));
        streamNameSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("server name: "))) {
      if (!serverNameSet) {
        setServerName(my_line.right(my_line.size()-13));
        serverNameSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("bitrate: "))) {
      if (!bitrateSet) {
        helper_qint64 = my_line.right(my_line.size()-9).toLongLong(&okay);
        if (NOT okay) {
          setBitrate(-1);
        } else {
          setBitrate(helper_qint64);
        };
        bitrateSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("declared bitrate: "))) {
      if (!bitrateSet) {
        helper_qint64 = my_line.right(my_line.size()-18).toLongLong(&okay);
        if (NOT okay) {
          setBitrate(-1);
        } else {
          setBitrate(helper_qint64);
        };
        bitrateSet = true;
      };
    }
    else if (my_line.toLower().startsWith(QString("meta interval: "))) {
      if (!metaIntervalSet) {
        helper_qint64 = my_line.right(my_line.size()-15).toLongLong(&okay);
        if (NOT okay) {
          setMetaInterval(-1);
        } else {
          setMetaInterval(helper_qint64);
        };
        metaIntervalSet = true;
      }
    }
    else if (my_line.toLower().startsWith(QString("error -"))) {
      if (!errorSet) {
        setError(my_line);
        errorSet = true;
      };
    } else {
      kDebug()
        << "could not recognize the following string:"
        << my_line
        << "size:"
        << my_line.size();
    };
  };
}

void ripping::helper_interpretate_metainfo_and_datasize(QString my_line)
{
  // help variables
  QString help_string;
  bool okay;
  qint64 dataSize;

  // code
  help_string = my_line.section('[', -1);  /* take all characters
  behind the last "[" in the string and put it to help_string */
  my_line.truncate(my_line.size() - help_string.size() - 2); /* removes
  the help_string from the end and also " [" before the help_string -> so
  there stays only the song (Okay, it's not really the song. It's the
  metadata.) itself. */

  // metainfo
  setSong(my_line);

  // data size
  // remove ']' and whitespace and convert tu uppercase letters:
  help_string=help_string.remove(']').simplified().toUpper();
  if (help_string.endsWith(QString("KB")) OR help_string.endsWith('K')) {
    if (help_string.endsWith(QString("KB"))) {
      help_string.truncate(help_string.size() - 2);
    } else {
      help_string.truncate(help_string.size() - 1);
    };
    dataSize = help_string.toLongLong(&okay) * 1024;
  } else {
    if (help_string.endsWith(QString("MB")) OR help_string.endsWith('M')) {
      if (help_string.endsWith(QString("MB"))) {
        help_string.truncate(help_string.size() - 2);
      } else {
        help_string.truncate(help_string.size() - 1);
      };
      dataSize = help_string.toDouble(&okay) * 1024 * 1024;
    } else {
      if (help_string.endsWith('B')) {
        // == ends with "B", but not "MB" or "KB" (see previous contitions)
        help_string.truncate(help_string.size() - 1);
        dataSize = help_string.toLongLong(&okay);
      } else {
        okay = false;
      };
    };
  };
  if (NOT okay) {
    setDataSize(-1);
  } else {
    setDataSize(dataSize);
  };
}

PropertyValue ripping::streamName() const
{
  return internal_streamName;
}

PropertyValue ripping::formatedStreamName(const QString & theStreamName)
{
    // variables
    PropertyValue temp_streamName;

    temp_streamName.internalValue = theStreamName;
    if (theStreamName == "Streamripper_rips") {
      temp_streamName.formatedValue = theStreamName;
      temp_streamName.type = PropertyValue::error;
      temp_streamName.toolTip = i18nc(
        "@info:tooltip Leave Streamripper_rips unchanged, it is a generic name of a directory",
        "Nameless stream. Using <emphasis>Streamripper_rips</emphasis> as replacement.");
      temp_streamName.whatsThis = i18nc(
        "@info:whatsthis Leave Streamripper_rips unchanged, it is a generic name of a directory",
        "This stream does not send a name in his meta data. "
          "<emphasis>Streamripper_rips</emphasis> is used as replacement, and "
          "you find the recorded files in the directory of the same name.");
    } else {
      if (theStreamName.isEmpty()) {
        temp_streamName.formatedValue = i18nc("@item", "not recognized");
        temp_streamName.type = PropertyValue::unset;
        temp_streamName.toolTip = i18nc(
          "@info:tooltip",
          "Could not connect to server.");
        temp_streamName.whatsThis = i18nc(
          "@info:whatsthis",
          "Could not connect to the specified server. So the stream name could not be recognized.");
      } else {
        temp_streamName.formatedValue = theStreamName;
        temp_streamName.type = PropertyValue::value;
      };
    };

    return temp_streamName;
}

void ripping::setStreamName(const QString & newStreamName)
{
  if (internal_streamName.internalValue.toString() != newStreamName) {
    internal_streamName = formatedStreamName(newStreamName);
    emit streamNameChanged(index(), internal_streamName);
  };
}

QString ripping::default_value_of_streamName()
{
  return QString();
}

PropertyValue ripping::formatedServerName(const QString & theServerName)
{
    // variables
    PropertyValue temp_serverName;

    // code
    temp_serverName.internalValue = theServerName;
    temp_serverName.formatedValue = theServerName;
    if (theServerName.isEmpty()) {
      temp_serverName.type = PropertyValue::unset;
    } else {
      temp_serverName.type = PropertyValue::value;
    };
    // there's never a toolTip or whatsThis, so no need to clear them.

    return temp_serverName;
}

PropertyValue ripping::serverName() const
{
  return internal_serverName;
}

void ripping::setServerName(const QString & newServerName)
{
  if (internal_serverName.internalValue.toString() != newServerName) {
    internal_serverName = formatedServerName(newServerName);
    emit serverNameChanged(index(), internal_serverName);
  };
}

QString ripping::default_value_of_serverName()
{
  return QString();
}

PropertyValue ripping::status() const
{
  return internal_status;
}

PropertyValue ripping::formatedStatus(const statusType theStatus)
{
    // variables
    PropertyValue temp_status;

    // code
    temp_status.internalValue.setValue(theStatus);
    if (theStatus == idle) {
      temp_status.type = PropertyValue::unset;
    } else {
      temp_status.type = PropertyValue::value;
    };
    switch (theStatus) {
      case idle:
        temp_status.formatedValue.clear();
        temp_status.toolTip.clear();
        temp_status.whatsThis.clear();
        break;
      case is_starting:
        temp_status.formatedValue = i18nc("@item status of streamripper", "Starting...");
        temp_status.toolTip = i18nc("@info:tooltip", "Invocing Streamripper...");
        temp_status.whatsThis = i18nc(
          "@info:whatsthis",
          "KRadioRipper is invocing Streamripper, the program used to "
            "perform the recording process.");
        break;
      case is_connecting:
        temp_status.formatedValue = i18nc("@item status of Streamripper", "Connecting...");
        temp_status.toolTip = i18nc("@info:tooltip", "Connecting with the stream server...");
        temp_status.whatsThis = i18nc("@info:whatsthis",
                                      "Streamripper is connecting with the stream server.");
        break;
      case is_buffering:
        temp_status.formatedValue = i18nc("@item status of Streamripper", "Buffering...");
        temp_status.toolTip = i18nc("@info:tooltip", "Buffering stream data...");
        temp_status.whatsThis = i18nc(
          "@info:whatsthis",
          "Stream data is buffered before starting the recording process.");
        break;
      case is_skipping:
        temp_status.formatedValue = i18nc("@item status of Streamripper", "Skipping...");
        temp_status.toolTip = i18nc("@info:tooltip", "Skipping the actual track...");
        temp_status.whatsThis = i18nc(
          "@info:whatsthis",
          "<para>KRadioRipper skips by default the first track, because this track will lack "
            "the begin.</para><para>You can change this behavior by editing the settings of the "
            "stream.</para>");
        break;
      case is_ripping:
        temp_status.formatedValue = i18nc("@item status of Streamripper", "Recording...");
        temp_status.toolTip = i18nc("@info:tooltip", "Recording the stream...");
        temp_status.whatsThis = i18nc("@info:whatsthis", "KRadioRipper is recording the stream.");
        break;
      case is_saving:
        temp_status.formatedValue = i18nc("@item status of Streamripper", "Saving...");
        temp_status.toolTip = i18nc("@info:tooltip", "Saving files...");
        temp_status.whatsThis = i18nc(
          "@info:whatsthis",
          "The buffer is processed and the last files are saved.");
        break;
    };

    return temp_status;
}

void ripping::setStatus(const statusType newStatus)
{
  if (internal_status.internalValue.value<statusType>() != newStatus) {
    internal_status = formatedStatus(newStatus);
    emit statusChanged(index(), internal_status);
    refreshRelayPort();
  };

  bool newIsRunning = (newStatus != idle);
  if (internal_isRunning != newIsRunning) {
    internal_isRunning = newIsRunning;
    if (newIsRunning) {
      emit running();
    } else {
      emit not_running();
    };
  };
}

ripping::statusType ripping::default_value_of_status()
{
  return idle;
}

PropertyValue ripping::error() const
{
  return internal_error;
}

PropertyValue ripping::formatedError(const QString & theError)
{
  // variables
  PropertyValue temp_error;
  QString streamripper_error_number;
  QString temp_message_of_unknown_id;

  // code
  temp_error.internalValue = theError;

  if (theError.isEmpty()) {
    temp_error.type = PropertyValue::unset;
  } else {
    temp_error.type = PropertyValue::value;
  };

  // message of format "error -123 bug description"
  if (theError.startsWith(QString("error -"))) {
    streamripper_error_number = theError.right(theError.size() - 7);  // without "error -"
    streamripper_error_number = streamripper_error_number.section(' ', 0, 0);
    switch (streamripper_error_number.toLongLong()) {
      case 3: // SR_ERROR_INVALID_URL
      case 6: // SR_ERROR_CANT_RESOLVE_HOSTNAME
        temp_error.formatedValue = i18nc("@item error message", "connection failed");
        temp_error.toolTip = i18nc("@info:tooltip error message", "Could not connect to server");
        temp_error.whatsThis = i18nc(
          "@info:whatsthis error message",
          "Either the URL is invalid or the corresponding server does not exist.");
        break;
      case 7: // SR_ERROR_RECV_FAILED
        temp_error.formatedValue = i18nc("@item error message", "incompatible stream");
        temp_error.toolTip = i18nc("@info:tooltip error message",
                                   "KRadioRipper can not record this type of stream.");
        temp_error.whatsThis = i18nc("@info:whatsthis error message",
                                      "Streamripper (and so also KRadioRipper) can only "
                                      "record shoutcast and icecast streams.");
        break;
      case 56: // HTTP:403 - Access Forbidden (try changing the UserAgent)
        temp_error.formatedValue = i18nc("@item error message", "connection refused");
        temp_error.toolTip = i18nc("@info:tooltip error message",
                                   "Try changing the user agent string");
        temp_error.whatsThis = i18nc(
          "@info:whatsthis error message",
          "<para>The server has refused the connection.</para><para>You can try to use another "
          "user agent string - maybe the server will accept this.</para>");
        break;
      case 64: // SR_ERROR_CANT_PARSE_PLS
        temp_error.formatedValue = i18nc("@item error message", "no stream");
        temp_error.toolTip = i18nc("@info:tooltip error message", "invalid playlist");
        temp_error.whatsThis = i18nc("@info:whatsthis error message",
                                     "The URL does not point directly to a stream, but to a "
                                       "playlist. And the playlist is invalid.");
        break;
      case 36: // SR_ERROR_CANT_CREATE_FILE
      case 1001: // BAD_DOWNLOAD_DIRECTORY
        temp_error.formatedValue = i18nc("@item error message", "bad download directory");
        temp_error.toolTip = i18nc(
          "@info:tooltip error message",
          "KRadioRipper could not write the file because the download directory is not writable.");
        temp_error.whatsThis = i18nc(
          "@info:whatsthis error message",
          "<para>The download directory is not accessible. Either "
            "it does not exist or you do not have sufficient access rights.</para><para>You "
            "can change the download directory at <emphasis>Settings</emphasis>, <emphasis>"
            "Configure KRadioRipper...</emphasis>, <emphasis>Saving</emphasis>"
            "</para>");
        break;
      default:
        temp_message_of_unknown_id = theError.right(theError.size() - 7);  // without "error -"
        temp_message_of_unknown_id = temp_message_of_unknown_id.section(' ', 1, 1);
        temp_error.formatedValue = i18nc("@item error message for an error that was not recognized "
                                           "(item 1: ID number of the error; item 2: error "
                                           "description in english",
                                         "Error %1: %2",
                                         streamripper_error_number.toLongLong(),
                                         temp_message_of_unknown_id);
        break;
    };
  } else {
    temp_error.formatedValue = theError;
  };

  return temp_error;
}

void ripping::setError(const QString & newError)
{
  if ((internal_error.internalValue.toString() != newError)) {
    /* we can't use != directly because QVariant doesn't support
    this for custom data types (but also doesn't generate a
    compiler error). */
    internal_error = formatedError(newError);
    emit errorChanged(index(), internal_error);
    refreshRelayPort();
  };
}

QString ripping::default_value_of_error()
{
  return QString();
}

PropertyValue ripping::song() const
{
  return internal_song;
}

PropertyValue ripping::formatedSong(const QString & theSong)
{
    // variables
    PropertyValue temp_song;

    // code
    temp_song.internalValue = theSong;
    if ((theSong.isEmpty()) OR (theSong EQUAL " - ")) {
      temp_song.type = PropertyValue::unset;
      temp_song.formatedValue.clear();
    } else {
      temp_song.type = PropertyValue::value;
      temp_song.formatedValue = theSong;
    };
    // there's never a toolTip or whatsThis, so no need to clear them.
    return temp_song;
}

void ripping::setSong(const QString & newSong)
{
  if (internal_song.internalValue.toString() != newSong) {
    internal_song = formatedSong(newSong);
    emit songChanged(index(), internal_song);
  };
}

QString ripping::default_value_of_song()
{
  return QString();
}

PropertyValue ripping::dataSize() const
{
  return internal_dataSize;
}

PropertyValue ripping::formatedDataSize(const qint64 theDataSize)
{
  // variables
  PropertyValue temp_dataSize;

  // code
  temp_dataSize.internalValue = theDataSize;
  if (theDataSize == (-1)) {
    temp_dataSize.type = PropertyValue::error;
    temp_dataSize.formatedValue = i18nc("@item", "error");
    temp_dataSize.toolTip = i18nc ("@info:tooltip", "Error determinating track size");
    temp_dataSize.whatsThis = i18nc (
      "@info:whatsthis",
      "The track size could not be determinated. Please report this as a bug.");
  } else {
    if (theDataSize >= 0) {
      temp_dataSize.type = PropertyValue::value;
      temp_dataSize.formatedValue = ki18nc("@item The unit is MiB instead of MB. See "
                                             "http://en.wikipedia.org/wiki/Binary_prefix "
                                             "for details.",
                                           "%1 MiB")
          .subs(double(theDataSize) / (1024 * 1024), 0, 'f', 2).toString();
      temp_dataSize.whatsThis = i18nc("@info:whatsthis",
                                       "<para>The size of the track in MiB.</para><para>MiB "
                                       "has a binary prefix which means 1024 * 1024 B = "
                                       "1048576 B (different from MB which would mean "
                                       "1000000 B).</para>");
    } else { // theDataSize < -1
      temp_dataSize.type = PropertyValue::unset;
      temp_dataSize.formatedValue.clear();
      temp_dataSize.whatsThis.clear();
    };
    temp_dataSize.toolTip.clear();
  };
  temp_dataSize.formatedValueAlignment = (Qt::AlignRight | Qt::AlignVCenter);

  return temp_dataSize;
}

void ripping::setDataSize(const qint64 newDataSize)
{
  if (internal_dataSize.internalValue.toLongLong() != newDataSize) {
    internal_dataSize = formatedDataSize(newDataSize);
    emit dataSizeChanged(index(), internal_dataSize);
  };
}

qint64 ripping::default_value_of_dataSize()
{
  return (-2);
}

bool ripping::default_value_of_isRunning()
{
  return false;
}

bool ripping::isRunning() const
{
  return internal_isRunning;
}

PropertyValue ripping::relayPort() const
{
  return internal_relayPort;
}

PropertyValue ripping::formatedRelayPort(const qint64 theRelayPort)
{
    // variables
    PropertyValue temp_relayPort;

    // code
    temp_relayPort.internalValue = theRelayPort;
    if (theRelayPort == (-1)) {
      temp_relayPort.type = PropertyValue::error;
      temp_relayPort.formatedValue = i18nc(
        "@item",
        "error");
      temp_relayPort.toolTip = i18nc (
        "@info:tooltip",
        "Error determinating relay server port");
      temp_relayPort.whatsThis = i18nc (
        "@info:whatsthis",
        "The port of the relay server could not be determinated. Please report this as a bug.");
    } else {
      if (theRelayPort >= 0) {
        temp_relayPort.type = PropertyValue::value;
        temp_relayPort.formatedValue = KGlobal::locale()->formatLong(theRelayPort);
      } else { // relayPort < -1
        temp_relayPort.type = PropertyValue::unset;
        temp_relayPort.formatedValue.clear();
      };
      temp_relayPort.toolTip.clear();
      temp_relayPort.whatsThis.clear();
    };
    temp_relayPort.formatedValueAlignment = (Qt::AlignRight | Qt::AlignVCenter);

    return temp_relayPort;
}

void ripping::refreshRelayPort()
{
  statusType actualStatus = status().internalValue.value<statusType>();
  if ((actualStatus == is_skipping || actualStatus == is_ripping) &&
      (error().type == PropertyValue::unset)) {
    internal_relayPort = formatedRelayPort(lastRecognizedRelayPort);
  } else {
    internal_relayPort = formatedRelayPort(-2);
  };
  emit relayPortChanged(index(), internal_relayPort);
}

void ripping::setRelayPort(const qint64 newRelayPort)
{
  if (lastRecognizedRelayPort != newRelayPort) {
    lastRecognizedRelayPort = newRelayPort;
    refreshRelayPort();
  };
}

qint64 ripping::default_value_of_relayPort()
{
  return (-2);
}

PropertyValue ripping::bitrate() const
{
  return internal_bitrate;
}

PropertyValue ripping::formatedBitrate(const qint64 theBitrate)
{
    // variables
    PropertyValue temp_bitrate;

    // code
    temp_bitrate.internalValue=theBitrate;

    if (theBitrate >= 1) {  // a valid bitrate...
      temp_bitrate.formatedValue = i18ncp(
        "@item This makes a nicly formated string for the bitrate of a stream - %1 is an integer. "
          "WARNING: Unit has changed! It is now kbit instead of Kibit. "
          "This means 1000 bit (NOT 1024).",
        "%1 kbit/s",
        "%1 kbit/s",
        theBitrate);
      temp_bitrate.type = PropertyValue::value;
      temp_bitrate.toolTip = i18nc("@info:tooltip", "declared bit rate");
      temp_bitrate.whatsThis = i18nc(
        "@info:whatsthis WARNING Unit has changed from binary prefix to SI prefix",
        "<para>The declared bit rate of the stream in kbit/s.</para><para>kbit has an SI prefix "
          "which means 1000 bit (different from Kibit which would mean 1024 bit). So 1 kbit/s "
          "means 1000 bits per second.</para>");
    } else {
      if (theBitrate >= -1) {  // "0" or "-1" (error during recognization)
        temp_bitrate.formatedValue = i18nc(
          "@item This makes a nicly formated string for the bitrate of a stream.",
          "Unable to recognize bitrate.");
        temp_bitrate.type = PropertyValue::error;
        temp_bitrate.toolTip = i18nc (
          "@info:tooltip",
          "Error determinating bit rate");
        temp_bitrate.whatsThis = i18nc (
          "@info:whatsthis",
          "The bit rate could not be determinated. Please report this as a bug.");
      } else { // "-2" or smaller (no value set)
        temp_bitrate.formatedValue.clear();
        temp_bitrate.type = PropertyValue::unset;
        temp_bitrate.toolTip.clear();
        temp_bitrate.whatsThis.clear();
      }
    };

    temp_bitrate.formatedValueAlignment = (Qt::AlignRight | Qt::AlignVCenter);

    return temp_bitrate;
}

void ripping::setBitrate(const qint64 newBitrate)
{
  if (internal_bitrate.internalValue.toLongLong() != newBitrate) {
    internal_bitrate = formatedBitrate(newBitrate);
    emit bitrateChanged(index(), internal_bitrate);
  };
}

qint64 ripping::default_value_of_bitrate()
{
  return (-2);
}

PropertyValue ripping::metaInterval() const
{
  return internal_metaInterval;
}

PropertyValue ripping::formatedMetaInterval(const qint64 theMetaInterval)
{
    // variables
    PropertyValue temp_metaInterval;

    // code
    temp_metaInterval.internalValue = theMetaInterval;
    if (theMetaInterval > 0) {     // metaInterval is 1 or bigger
      temp_metaInterval.formatedValue = //KGlobal::locale()->formatLong(theMetaInterval);
        ki18nc("@item The unit is KiB instead of kB. See "
                 "http://en.wikipedia.org/wiki/Binary_prefix for details.",
               "%1 KiB")
          .subs(qint64(theMetaInterval / 1024)).toString();
      temp_metaInterval.type = PropertyValue::value;
    } else {
      if (theMetaInterval >= -1) { // metaInterval is 0 or -1
        temp_metaInterval.formatedValue = i18nc(
          "@item",
          "error");
        temp_metaInterval.type = PropertyValue::error;
      } else {
        temp_metaInterval.formatedValue.clear();
        temp_metaInterval.type = PropertyValue::unset;
      };
    };
    temp_metaInterval.formatedValueAlignment = (Qt::AlignRight | Qt::AlignVCenter);

    return temp_metaInterval;
}

void ripping::setMetaInterval(const qint64 newMetaInterval)
{
  if (internal_metaInterval.internalValue != newMetaInterval) {
    internal_metaInterval = formatedMetaInterval(newMetaInterval);
    emit metaIntervalChanged(index(), internal_metaInterval);
  };
}

qint64 ripping::default_value_of_metaInterval()
{
  return (-2);
}

PropertyValue ripping::metaInterval_milliSeconds() const
{
  // variables
  qint64 temp;

  // code
  if (bitrate().type == PropertyValue::value &&  // both are values...
      metaInterval().type == PropertyValue::value &&
      bitrate().internalValue.toLongLong() > 0) { // Prevent from dividing by 0. Is redundant to
                                            // bitrate.type == PropertyValue::value.
                                            // Just to be sure, also with further changes...
    /* bitrate has the unit kbit/s...
    * kbit/s = (1000 bit)/s
    *        = (1000 bit)/(1000 ms)
    *        = bit/ms.
    * Conclusion: kbit/s=bit/ms, so we can think of bitrate as bit/ms.
    *
    * metaInterval has the unit Byte. So we take the value *8, so we are
    * in the unit bit.
    *
    * So we calculate: (metaInterval*8)/bitrate. This works fine for the units:
    * bit/(bit/ms)=ms, so we get the metaInterval in ms! */
    temp = (metaInterval().internalValue.toLongLong() * 8) / bitrate().internalValue.toLongLong();
  } else if (bitrate().type == PropertyValue::unset ||  // at least one is unset...
             metaInterval().type == PropertyValue::unset) {
    temp = -2;
  } else {
    temp = -1;
  };
  return formatedMetaInterval_milliSeconds(temp);
}

PropertyValue ripping::formatedMetaInterval_milliSeconds(const qint64 theMetaInterval)
{
    // variables
    PropertyValue temp_metaInterval_milliSeconds;

    // code
    temp_metaInterval_milliSeconds.internalValue = theMetaInterval;
    if (theMetaInterval > 0) {     // metaInterval is 1 or bigger
      temp_metaInterval_milliSeconds.formatedValue =
        ki18nc("@item milliseconds", "%1 ms").subs(theMetaInterval).toString();
      temp_metaInterval_milliSeconds.type = PropertyValue::value;
    } else {
      if (theMetaInterval >= -1) { // metaInterval is 0 or -1
        temp_metaInterval_milliSeconds.formatedValue = i18nc(
          "@item",
          "error");
        temp_metaInterval_milliSeconds.type = PropertyValue::error;
      } else {
        temp_metaInterval_milliSeconds.formatedValue.clear();
        temp_metaInterval_milliSeconds.type = PropertyValue::unset;
      };
    };
    temp_metaInterval_milliSeconds.formatedValueAlignment = (Qt::AlignRight | Qt::AlignVCenter);

    return temp_metaInterval_milliSeconds;
}

void ripping::errorOccured(const QProcess::ProcessError error)
{
  // variables
  QFileInfo m_file_info(settings_general::streamripperCommand());

  // code
  switch (error) {
    case QProcess::FailedToStart:
      if (m_file_info.exists()) {
        setError(i18nc(
          "@item streamripper error",
          "Insufficient permissions to invoke Streamripper."));
      } else {
        setError(i18nc("@item streamripper error", "Streamripper binary not found."));
      };
      break;
    case QProcess::Crashed:
      setError(i18nc("@item streamripper error", "Streamripper crashed."));
      break;
    case QProcess::Timedout:
      setError(i18nc("@item streamripper error", "Streamripper does not react."));
      break;
    default:  //ReadError, WriteError, UnknownError
      setError(i18nc("@item streamripper error", "Error accessing Streamripper."));
      break;
  };
}

void ripping::streamripperStateChange(const QProcess::ProcessState newState)
{
  if (newState == QProcess::NotRunning) {
    setStatus(default_value_of_status());
    setSong(default_value_of_song());
    /* Here, we can't determinate
    *  - if the program has terminated normally (good) or
    *  - if it has crashed or
    *  - if it couldn't even be started, because the binary wasn't found or
    *    wasn't executable.
    *
    *  In the last 2 cases, we should display an error message, depending on
    *  _which_ error occurred. But we can't use m_process.error() to determinate
    *  the error type, because QProcess emits the signal stateChanged() (to which
    *  this slot is connected) _before_ it actualizes the property error().
    *  Because of this, the error message is set by the slot errorOccured(), who
    *  is connected to the signal error(). */
    setDataSize(default_value_of_dataSize());
    setRelayPort(default_value_of_relayPort());
  };
}

QStringList ripping::parameterList() const
{
  // variables
  QStringList parameters;
  QString temp;

  //code
  parameters.append(serverUri());

  parameters.append(QString("-t")); // don't override files in the incomplete dir
                                    // but make instead a save copy of the old file
                                    // by appending an index. I'm not sure if this
                                    // is absolutly needed, but at least it's not bad.

  temp = proxyinfo::proxyserver(serverUri()).at(0);
  if (temp != "direct://") {
    parameters.append(QString("-p"));
    parameters.append(temp);
  };

  if (settings_general::createRelayServer()) { // create relay server
    // create a relay server at (or up to) port X!
    parameters.append(QString("-r"));
    parameters.append(QString::number(settings_general::preferedPortForRelayServer()));

    // the number of connections is limited to which number?
    parameters.append(QString("-R"));
    if (settings_general::limitConnections()) {
      parameters.append(QString::number(settings_general::limitConnectionsToX()));
    } else {
      parameters.append(QString("0"));
    };
  };

  parameters.append(QString("-c"));  // don't auto-reconnect  TODO: implement this on my own way!

  return parameters;
}

QString ripping::streamripperCommand() const
{
  return settings_general::streamripperCommand();
}

void ripping::startStreamripper()
{
  // variables
  QFileInfo * dir;
  QDir *dirCreator;
  QString temp;

  // code
  if (m_process.state() == QProcess::NotRunning) { // TODO else if the process is
    // yet shutting down or has yet an error -> einreihen für späteren automatischen Neustart!
    setError(QString());
    setStatus(is_starting);
    temp = workingDirectory();
    if (temp.startsWith(QLatin1String("file://"))) {
      temp.remove(0, 7);
    };
    dir = new QFileInfo(temp);
    // If dir doesn't exist and is an absulute path: Create it (if possible).
    if ((!dir->exists()) && dir->isAbsolute()) {
      dirCreator = new QDir(/*dir*/);
      dirCreator->mkpath(dir->filePath());
      dir->refresh();
    };
    if (dir->exists() && dir->isAbsolute() && dir->isDir() &&
        dir->isReadable() && dir->isWritable()) {
      m_process.setWorkingDirectory(workingDirectory());
      streamripper_base::startStreamripper();
    } else {
      // Emulate streamripper error message to get automatically a formated message:
      setError("error -1001 [BAD_DOWNLOAD_DIRECTORY]");
      setStatus(idle);
    };
  };
}

void ripping::shutDown()
{
  m_process.terminate();
  /* TODO eventuelle Einreihung für Neustart wieder löschen! (wenn jemand startet,
  während gerade runtergefahren wird) */
}

bool ripping::doesTheUserWantsThatTheStreamIsRipping(ripping::statusType theStatus)
{
  return (!((theStatus == ripping::idle) OR (theStatus == ripping::is_saving)));
}

bool ripping::doesTheUserWantsThatTheStreamIsRipping()
{
  return doesTheUserWantsThatTheStreamIsRipping(
           internal_status.internalValue.value<statusType>());
}
