/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mainwindow.h"

#include <QClipboard>
#include <QDockWidget>
#include <QMimeData>
#include <QTreeView>
#include <KApplication>
#include <KAction>
#include <KLocale>
#include <KActionCollection>
#include <KTipDialog>
#include <KStatusBar>
#include <KStandardDirs>
#include "icecast.h"
#include "settings_general_dialog.h"
#include "streamdirectoryproxymodel.h"

MainWindow::MainWindow(QWidget *parent) : KXmlGuiWindow(parent)
{
  //initialise central widget
  m_streamlistwidget = new stationlistWidget(this);
  setCentralWidget(m_streamlistwidget);

  //setup actions
  KStandardAction::preferences(this, SLOT(display_global_settings_dialog()),
                                actionCollection());

  KStandardAction::quit(kapp, SLOT(quit()), actionCollection());

  KAction * recordAction = new KAction(this);
  recordAction->setText(i18nc("@action", "&Record"));
  recordAction->setIcon(KIcon("media-record"));
  QList<QKeySequence> recordAction_tempShortcutList;
  recordAction_tempShortcutList.append(Qt::Key_R);
  recordAction_tempShortcutList.append(Qt::Key_MediaRecord);
  recordAction_tempShortcutList.append(Qt::CTRL + Qt::Key_R);
  recordAction->setShortcuts(recordAction_tempShortcutList);
  recordAction->setDisabled(true);
  connect(recordAction, SIGNAL(triggered(bool)),
           m_streamlistwidget, SLOT(record()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
           recordAction, SLOT(setDisabled(bool)));
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
           recordAction, SLOT(setEnabled(bool)));
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
           recordAction, SLOT(setEnabled(bool)));
  actionCollection()->addAction("record", recordAction);
  m_streamlistwidget->streamContextMenu.addAction(recordAction);

  KAction * stopRecordAction = new KAction(this);
  stopRecordAction->setText(i18nc("@action", "&Stop"));
  stopRecordAction->setIcon(KIcon("media-playback-stop"));
  QList<QKeySequence> stopRecordAction_tempShortcutList;
  stopRecordAction_tempShortcutList.append(Qt::Key_S);
  stopRecordAction_tempShortcutList.append(Qt::Key_MediaRecord);
  stopRecordAction_tempShortcutList.append(Qt::CTRL + Qt::Key_S);
  stopRecordAction_tempShortcutList.append(Qt::META + Qt::Key_V);
  stopRecordAction->setShortcuts(stopRecordAction_tempShortcutList);
  stopRecordAction->setDisabled(true);
  connect(stopRecordAction, SIGNAL(triggered(bool)),
           m_streamlistwidget, SLOT(stopRecord()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
           stopRecordAction, SLOT(setDisabled(bool)));
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
           stopRecordAction, SLOT(setEnabled(bool)));
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
           stopRecordAction, SLOT(setEnabled(bool)));
  actionCollection()->addAction("stopRecord", stopRecordAction);
  m_streamlistwidget->streamContextMenu.addAction(stopRecordAction);

  KAction * startListenInAction = new KAction(this);
  startListenInAction->setText(
    i18nc("@action listening in the stream that is actually recorded (and has been selected)",
          "&Listen in"));
  startListenInAction->setIcon(KIcon("audio-volume-high"));
  /*QList<QKeySequence> startListenInAction_tempShortcutList;
  startListenInAction_tempShortcutList.append(Qt::Key_S);
  startListenInAction_tempShortcutList.append(Qt::Key_MediaRecord);
  startListenInAction_tempShortcutList.append(Qt::CTRL + Qt::Key_S);
  startListenInAction_tempShortcutList.append(Qt::META + Qt::Key_V);
  startListenInAction->setShortcuts(startListenInAction_tempShortcutList);*/
  startListenInAction->setDisabled(true);
  connect(startListenInAction, SIGNAL(triggered(bool)),
          m_streamlistwidget->stationlistmodel(), SLOT(enableListeningIn()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
          startListenInAction, SLOT(setDisabled(bool)));
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
          startListenInAction, SLOT(setEnabled(bool)));
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
          startListenInAction, SLOT(setDisabled(bool)));
  actionCollection()->addAction("startListenIn", startListenInAction);
  m_streamlistwidget->streamContextMenu.addAction(startListenInAction);

  KAction * stopListenInAction = new KAction(this);
  stopListenInAction->setText(i18nc("@action listening in the stream that is actually recorded",
                                    "&Mute"));
  stopListenInAction->setIcon(KIcon("audio-volume-muted"));
  /*QList<QKeySequence> stopListenInAction_tempShortcutList;
  stopListenInAction_tempShortcutList.append(Qt::Key_S);
  stopListenInAction_tempShortcutList.append(Qt::Key_MediaRecord);
  stopListenInAction_tempShortcutList.append(Qt::CTRL + Qt::Key_S);
  stopListenInAction_tempShortcutList.append(Qt::META + Qt::Key_V);
  stopListenInAction->setShortcuts(stopListenInAction_tempShortcutList);*/
  stopListenInAction->setDisabled(true);
  connect(stopListenInAction, SIGNAL(triggered(bool)),
          m_streamlistwidget->stationlistmodel(), SLOT(disableListeningIn()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
          stopListenInAction, SLOT(setDisabled(bool)));
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
          stopListenInAction, SLOT(setEnabled(bool)));
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
          stopListenInAction, SLOT(setDisabled(bool)));
  actionCollection()->addAction("stopListenIn", stopListenInAction);
  m_streamlistwidget->streamContextMenu.addAction(stopListenInAction);

  KAction * newStreamAction = KStandardAction::openNew(m_streamlistwidget,
                                                       SLOT(addNewStation()),
                                                       this);
  newStreamAction->setText(i18nc("@action", "&New stream"));
  actionCollection()->addAction("newStream", newStreamAction);
  m_streamlistwidget->globalContextMenu.addAction(newStreamAction);
  m_streamlistwidget->streamContextMenu.addAction(newStreamAction);

  KAction * deleteStreamAction = new KAction(this);
  deleteStreamAction->setText(i18nc("@action", "&Delete stream"));
  deleteStreamAction->setIcon(KIcon("edit-delete"));
  deleteStreamAction->setShortcut(Qt::SHIFT + Qt::Key_Delete);
  deleteStreamAction->setDisabled(true);
  connect(deleteStreamAction, SIGNAL(triggered(bool)),
          m_streamlistwidget, SLOT(deleteStation()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
          deleteStreamAction, SLOT(setDisabled(bool)));
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
          deleteStreamAction, SLOT(setEnabled(bool)));
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
          deleteStreamAction, SLOT(setEnabled(bool)));
  actionCollection()->addAction("deleteStream", deleteStreamAction);
  m_streamlistwidget->streamContextMenu.addAction(deleteStreamAction);

  pasteAction = actionCollection()->addAction(KStandardAction::Paste,
                                              "paste",
                                              m_streamlistwidget->stationlistmodel(),
                                              SLOT(paste()));
  m_streamlistwidget->globalContextMenu.addAction(pasteAction);
  m_streamlistwidget->streamContextMenu.addAction(pasteAction);
  connect(QApplication::clipboard(), SIGNAL(dataChanged()),
          this, SLOT(reloadStatusOfPasteAction()));
  reloadStatusOfPasteAction();

  KAction * selectAllAction = KStandardAction::selectAll(m_streamlistwidget,
                                                         SLOT(selectAll()),
                                                         actionCollection());
  m_streamlistwidget->globalContextMenu.addAction(selectAllAction);
  m_streamlistwidget->streamContextMenu.addAction(selectAllAction);

  KAction * streamSettingsAction = new KAction(this);
  streamSettingsAction->setText(i18nc("@action properties of a stream)", "&Properties"));
  streamSettingsAction->setIcon(KIcon("document-properties"));
  QList<QKeySequence> streamSettingsAction_tempShortcutList;
  streamSettingsAction_tempShortcutList.append(Qt::ALT + Qt::Key_Return);
  streamSettingsAction_tempShortcutList.append(Qt::Key_MediaStop);
  streamSettingsAction->setShortcuts(streamSettingsAction_tempShortcutList);
  streamSettingsAction->setDisabled(true);
  connect(streamSettingsAction, SIGNAL(triggered(bool)),
           m_streamlistwidget, SLOT(displayStreamSettings()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
           streamSettingsAction, SLOT(setDisabled(bool)));
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
           streamSettingsAction, SLOT(setEnabled(bool)));
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
           streamSettingsAction, SLOT(setDisabled(bool)));
  actionCollection()->addAction("streamSettings", streamSettingsAction);
  m_streamlistwidget->streamContextMenu.addAction(streamSettingsAction);

  KStandardAction::tipOfDay(this, SLOT(displayTipOfDay()), actionCollection());

  //setup GUI
  setupGUI();  // creates also some further actions automatically
  QAction *helpContentsAction = actionCollection()->action("help_contents");
  if (!(helpContentsAction == 0)) {
    helpContentsAction->setEnabled(false);
  };

  statusBar()->insertPermanentItem(QString(), 0);
  statusBar()->insertPermanentItem(QString(), 1);
  connect(m_streamlistwidget->stationlistmodel(),
           SIGNAL(bandwidthChanged()),
           this,
           SLOT(actualize_bandwidth_in_statusBar()));
  connect(m_streamlistwidget->stationlistmodel(),
           SIGNAL(numberOfActiveStreamsChanged()),
           this,
           SLOT(actualize_numberOfActiveStreams_in_statusBar()));
  actualize_bandwidth_in_statusBar();  // initialization
  actualize_numberOfActiveStreams_in_statusBar();  // initialization

  QDockWidget *m_streamDirectory = new QDockWidget(
    i18nc("@title:window Dock name of the directory of radio stations", "stream catalog"),
    this);
  m_streamDirectory->setObjectName("theRadiostationDirectory");  // to avoid a warning on stdout
  m_streamDirectory->setAllowedAreas(Qt::LeftDockWidgetArea | Qt::RightDockWidgetArea |
    Qt::BottomDockWidgetArea);
  QTreeView *m_directoryView = new QTreeView();
  m_streamDirectory->setWidget(m_directoryView);
  icecast *m_icecast_model = new icecast(m_directoryView/*TODO, the_icon*/);
  streamDirectoryProxyModel *m_proxy_model = new streamDirectoryProxyModel(m_icecast_model);
  m_proxy_model->setSourceModel(m_icecast_model);
  m_directoryView->setModel(m_proxy_model);
  m_directoryView->setVerticalScrollMode(QAbstractItemView::ScrollPerPixel);
  m_directoryView->setHorizontalScrollMode(QAbstractItemView::ScrollPerPixel);
  m_directoryView->setAnimated(true);
  m_directoryView->setAlternatingRowColors(true);
  m_directoryView->setSortingEnabled(true);
  m_directoryView->setUniformRowHeights(true); // makes the widget more performant
  m_directoryView->setSelectionMode(QAbstractItemView::ExtendedSelection);
  m_directoryView->setDragEnabled(true);
  addDockWidget(Qt::LeftDockWidgetArea, m_streamDirectory);

  KTipDialog::showTip(this, "kradioripper/tips");
}

MainWindow::~MainWindow()
{
}

void MainWindow::actualize_numberOfActiveStreams_in_statusBar()
{
  statusBar()->changeItem(
    i18ncp("@info:status",
            "Recording %1 stream.",
            "Recording %1 streams.",
            m_streamlistwidget->stationlistmodel()->numberOfActiveStreams()),
    0);
}

void MainWindow::actualize_bandwidth_in_statusBar()
{
  statusBar()->changeItem(
    i18ncp("@info:status The unit is kbit (=1000 bit) instead of Kibit (=1024 bit). "
              "See http://en.wikipedia.org/wiki/Binary_prefix for details.",
            "%1 kbit/s",
            "%1 kbit/s",
            m_streamlistwidget->stationlistmodel()->bandwidth()),
    1);
}

bool MainWindow::queryClose()
{
  return m_streamlistwidget->queryClose();
}

void MainWindow::saveProperties(KConfigGroup & m_configGroup)
{
  m_streamlistwidget->saveProperties(m_configGroup);
}

void MainWindow::readProperties(const KConfigGroup & m_configGroup)
{
  m_streamlistwidget->readProperties(m_configGroup);
}

void MainWindow::display_global_settings_dialog()
{
  // if the dialog object yet exists, we can display it directly and return.
  if (KConfigDialog::showDialog("settings_general_dialog")) {
    return;
  };

  // else we have to construct and configure it before displaying it and return
  settings_general_dialog * dialog = new settings_general_dialog(this,
                                                                  "settings_general_dialog");
  dialog->show();
}

void MainWindow::displayTipOfDay()
{
  KTipDialog::showTip(this, "kradioripper/tips", true);
}

void MainWindow::reloadStatusOfPasteAction()
{
  bool isAccepted = false;
  const QStringList clipboardFormats = QApplication::clipboard()->mimeData()->formats();
  const QStringList acceptedFormats = m_streamlistwidget->stationlistmodel()->mimeTypes();
  const int numberOfAcceptedFormats = acceptedFormats.size();
  int i = 0;
  while ((!isAccepted) && (i < numberOfAcceptedFormats)) {
    isAccepted = clipboardFormats.contains(acceptedFormats.at(i));
    ++i;
  };
  pasteAction->setEnabled(isAccepted);
}
