/***************************************************************************
 *   Copyright (C) 2005 by Conrad Hoffmann                                 *
 *   conrausch@gmx.de                                                      *
 *   http://conrausch.doesntexist.org                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include <qlabel.h>
#include <qlayout.h>
#include <qsize.h>
#include <qptrlist.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qpainter.h>
#include <qfont.h>
#include <qevent.h>
#include <qtooltip.h>

#include <kdebug.h>
#include <krootpixmap.h>
#include <kiconloader.h>
#include <kimageeffect.h>

#include <kopetecontact.h>
#include <kopetemetacontact.h>
#include <kopetecontactlist.h>

#include "kopetedesklistplugin.h"
#include "kopetedesklistitem.h"
#include "kopetedesklistrootpixmap.h"
#include "kopete_desklist.h"


KopeteDesklistItem::KopeteDesklistItem( Kopete::MetaContact* metaContact, bool forceSingleLine, QWidget* parent, const char* name)
		: QWidget(parent, name),
		m_metaContact( metaContact ),
		m_contactIconsLayout( 0L )
{
	KopeteDesklistKcfg::enableSmoothCorners() ? applyRoundedRootPixmap( this ) : applyRootPixmap( this );
	setFocusPolicy( QWidget::NoFocus );

	/* determine layout parameters */
	int rows = forceSingleLine ? 1 : 2;
	int cols = forceSingleLine ? 5 : 2; // the 5 is actually a nasty hack to gain better alignment...
	int padding = KopeteDesklistKcfg::contactBoxPadding();
	m_outerLayout = new QGridLayout( this, rows, cols, padding, 3 );

	/* create the label that displays the name */
	m_nameLabel = new QLabel( this );
	m_nameLabel->setFont( KopeteDesklistKcfg::contactBoxFont() );
	m_nameLabel->setPaletteForegroundColor( KopeteDesklistKcfg::contactBoxTextColor() );
	if ( KopeteDesklistKcfg::enableTruncateNames() && ( m_metaContact->displayName().length() > KopeteDesklistKcfg::truncateNamesLength() ) )
	{
		QString s = m_metaContact->displayName();
		s.truncate( KopeteDesklistKcfg::truncateNamesLength() );
		m_nameLabel->setText( s.append( KopeteDesklistKcfg::truncateNamesAppendix() ) );
	}
	else
	{
		m_nameLabel->setText( m_metaContact->displayName() );
	}
	applyRootPixmap( m_nameLabel );

	/* create the label that displays the picture */
	m_photoLabel = new QLabel( this );
	applyRootPixmap( m_photoLabel );
	m_photoLabel->setPixmap( KopeteDesklistItem::generatePixmap( m_metaContact ) );

	/* create a seperate layout with labels displaying the contact icons */
	m_contactIconsLayout = new QHBoxLayout();
	QPtrList<Kopete::Contact> allContacts = m_metaContact->contacts();
	QLabel* label;
	for ( Kopete::Contact* c = allContacts.first(); c != 0; c = allContacts.next() )
	{
		QPixmap accountIcon = c->onlineStatus().iconFor( c->account() );
		label = new QLabel( this );
		label->setPixmap( accountIcon );
		applyRootPixmap( label );
		m_contactIconsLayout->addWidget( label );
		m_contacts.insert( c->contactId(), label );
	}

	/* determine layout flags from preferences */
	Qt::AlignmentFlags detailsAlignment = KopeteDesklistKcfg::detailsAlignmentLeft() ? Qt::AlignLeft : Qt::AlignRight;

	/* lay it all out */
	if ( KopeteDesklistKcfg::photoPositionLeft() )
	{
		if ( forceSingleLine )
		{
			m_outerLayout->addMultiCellWidget( m_photoLabel, 0, 0, 0, 0, Qt::AlignLeft );
			m_outerLayout->addMultiCellWidget( m_nameLabel, 0, 0, 1, 3, detailsAlignment );
			m_outerLayout->addMultiCellLayout( m_contactIconsLayout, 0, 0, 4, 4, detailsAlignment );
		}
		else
		{
			m_outerLayout->addMultiCellWidget( m_photoLabel, 0, 1, 0, 0, Qt::AlignLeft );
			m_outerLayout->addMultiCellWidget( m_nameLabel, 0, 0, 1, 1, detailsAlignment );
			m_outerLayout->addMultiCellLayout( m_contactIconsLayout, 1, 1, 1, 1, detailsAlignment );
		}
	}
	else
	{
		if ( forceSingleLine )
		{
			m_outerLayout->addMultiCellWidget( m_nameLabel, 0, 0, 0, 2, detailsAlignment );
			m_outerLayout->addMultiCellLayout( m_contactIconsLayout, 0, 0, 3, 3, detailsAlignment );
			m_outerLayout->addMultiCellWidget( m_photoLabel, 0, 0, 4, 4, Qt::AlignRight );
		}
		else
		{
			m_outerLayout->addMultiCellWidget( m_nameLabel, 0, 0, 0, 0, detailsAlignment );
			m_outerLayout->addMultiCellLayout( m_contactIconsLayout, 1, 1, 0, 0, detailsAlignment );
			m_outerLayout->addMultiCellWidget( m_photoLabel, 0, 1, 1, 1, Qt::AlignRight );
		}
	}

	generateToolTips();
	show();
	QObject::connect( m_metaContact, SIGNAL( displayNameChanged( const QString&, const QString& ) ), this, SLOT( slotNameChanged() ) );
	QObject::connect( m_metaContact, SIGNAL( contactStatusChanged( Kopete::Contact*, const Kopete::OnlineStatus& ) ), this, SLOT( slotContactStatusChanged( Kopete::Contact*, const Kopete::OnlineStatus& ) ) );
	QObject::connect( m_metaContact, SIGNAL( photoChanged() ), this, SLOT( slotPhotoChanged() ) );
}


KopeteDesklistItem::~KopeteDesklistItem()
{
}

// static
QPixmap KopeteDesklistItem::generatePixmap( const Kopete::MetaContact *mc )
{
	QImage photoImg;
	if ( mc == 0L || mc->photo() == 0L )
	{
		KIconLoader *loader = KGlobal::iconLoader();
		photoImg = loader->loadIcon( QString::fromLatin1( "presence_online" ), KIcon::Toolbar, KIcon::SizeHuge ).convertToImage();
	}
	else
	{
		photoImg = mc->photo();
	}

	// make sure height is as desired, currently just don't care about width
	photoImg = photoImg.smoothScale( 1, KopeteDesklistKcfg::photoWidth(), QImage::ScaleMax );

	// the effects known from the contact list
	KImageEffect *effect = 0L;
	switch ( mc->status() )
	{
		case Kopete::OnlineStatus::Online:
		break;
		case Kopete::OnlineStatus::Away:
			effect = new KImageEffect();
			effect->fade(photoImg, 0.5, Qt::white);
		break;
		case Kopete::OnlineStatus::Offline:
			effect = new KImageEffect();
			effect->fade(photoImg, 0.4, Qt::white);
			effect->toGray(photoImg);
		break;
		case Kopete::OnlineStatus::Unknown:
		default:
			effect = new KImageEffect();
			effect->fade(photoImg, 0.8, Qt::white);
	}
	delete effect;

	QPixmap photoPixmap;
	photoPixmap = photoImg;


	/* if someone is interested in drawing the black border around contact photos,
	 * like on the contact list, uncomment these lines. but i recommend partially
	 * transparent pngs w/o border for full eye candy effect ;) */
// 	QPainter p(&photoPixmap);
// 	p.setPen(Qt::black);
// 	p.drawLine(0, 0, photoPixmap.width()-1, 0);
// 	p.drawLine(0, photoPixmap.height()-1, photoPixmap.width()-1, photoPixmap.height()-1);
// 	p.drawLine(0, 0, 0, photoPixmap.height()-1);
// 	p.drawLine(photoPixmap.width()-1, 0, photoPixmap.width()-1, photoPixmap.height()-1);

	return photoPixmap;
}


void KopeteDesklistItem::applyRootPixmap( QWidget *w )
{
	KRootPixmap *krp = new KRootPixmap( w );
	krp->setFadeEffect( KopeteDesklistKcfg::contactBoxOpacity() * 0.01, KopeteDesklistKcfg::contactBoxTintColor() );
	krp->start();
}


void KopeteDesklistItem::applyRoundedRootPixmap( QWidget *w )
{
	KopeteDesklistRootPixmap *krp = new KopeteDesklistRootPixmap( w );
	krp->setFadeEffect( KopeteDesklistKcfg::contactBoxOpacity() * 0.01, KopeteDesklistKcfg::contactBoxTintColor() );
	krp->start();
}


void KopeteDesklistItem::generateToolTips()
{
	if ( KopeteDesklistKcfg::showMetaToolTips() )
	{
		QString toolTip = QString::fromLatin1("<html><head></head><body><table><tr><td>");
	
		if ( ! m_metaContact->photo().isNull() )
		{
			QString photoName = QString::fromLatin1("kopete-metacontact-photo:%1").arg( KURL::encode_string( m_metaContact->metaContactId() ));
			toolTip += QString::fromLatin1("<img src=\"%1\" style=\"margin-bottom:10px;\">").arg( photoName );
		}
	
	
		toolTip += QString::fromLatin1("</td><td><b><font size=\"+1\">%1</font></b><br><br>").arg( m_metaContact->displayName() );
		QPtrList<Kopete::Contact> contactList = m_metaContact->contacts();
		for ( Kopete::Contact* c = contactList.first(); c != 0L; c = contactList.next() )
		{
			toolTip += c->toolTip() + QString::fromLatin1("<br><br>");
		}
		toolTip += QString::fromLatin1("</body><html>");
	
		QToolTip::add( m_photoLabel, toolTip );
	}

	if ( KopeteDesklistKcfg::showContactToolTips() )
	{
		QPtrList<Kopete::Contact> contactList = m_metaContact->contacts();
		for ( Kopete::Contact* c = contactList.first(); c != 0L; c = contactList.next() )
		{
			QLabel* l = m_contacts.find( c->contactId() );
			if ( l )
			{
				QToolTip::add( l, c->toolTip() );
			}
		}
	}
}


void KopeteDesklistItem::mouseDoubleClickEvent( QMouseEvent *e )
{
	if ( e->button() == Qt::LeftButton )
	{
		kdDebug() << k_funcinfo << "Trying to initiate chat..." << endl;
		m_metaContact->startChat();
	}
}


void KopeteDesklistItem::contextMenuEvent( QContextMenuEvent* e )
{
	KopeteDesklistPlugin::self()->showContextMenu( m_metaContact, e->globalPos() );
}


void KopeteDesklistItem::slotNameChanged()
{
	KopeteDesklistPlugin::self()->requestGroupRefresh( m_metaContact );
}

void KopeteDesklistItem::slotPhotoChanged()
{
	m_photoLabel->setPixmap( KopeteDesklistItem::generatePixmap( m_metaContact ) );
}

void KopeteDesklistItem::slotContactStatusChanged( Kopete::Contact* contact, const Kopete::OnlineStatus& status )
{
	QLabel* l = m_contacts.find( contact->contactId() );
	if ( l )
	{
		l->setPixmap( contact->onlineStatus().iconFor( contact->account() ) );
	}
}

void KopeteDesklistItem::updateMetaStatus()
{
	m_photoLabel->setPixmap( KopeteDesklistItem::generatePixmap( m_metaContact ) );
}

#include "kopetedesklistitem.moc"
