/***************************************************************************
 *   Copyright (C) 2007-2009 by Thomas Thelliez <admin.kontrol@gmail.com>  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as               *
 *   published by the Free Software Foundation; either version 2.0 of      *
 *   the License, or (at your option) any later version.                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public             *
 *   License along with this program; if not, write to                     *
 *   the Free Software Foundation, Inc.,                                   *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Server.h"

Server::Server(QObject *parent)
     : QTcpServer(parent)
{
    clientList = new QMap<int, Client*>();
    clientNameList = new QMap<int, QString>();
    blackList = new QMap<int, QString>();
    filteredIpList = XmlDataStore::getFilteredIpsData();
    xmlCreator = new XmlCreator(this);
    serverStatus = true;
    alreadyWarm = false;
    filteredBool = false;
    connect(this, SIGNAL(newConnection()), this, SLOT(addNewClient()));
    nbrClient = 0;
}

void Server::addNewClient()
 {
    nbrClient++;
    QTcpSocket *socket = this->nextPendingConnection();
    Client *client = new Client(socket, nbrClient, this);
    QObject::connect(this, SIGNAL(stopClient(int)), client, SLOT(stoppedByServer(int)));
    QObject::connect(socket, SIGNAL(readyRead()), client, SLOT(readResults()));
    QObject::connect(client, SIGNAL(sendClientParametersToAddTab(QStringList,int)), this, SLOT(getClientParameters(QStringList,int)));
    QObject::connect(client, SIGNAL(sendClientContext(QStringList,int)), this, SLOT(getClientContext(QStringList,int)));
    QObject::connect(client, SIGNAL(error(QString)), this, SLOT(error(QString)));
    QObject::connect(client, SIGNAL(sendMessage(QString)), this, SLOT(sendMessage(QString)));
    QObject::connect(client, SIGNAL(sendResultsToDisplay(QString*,int)), this, SLOT(getResults(QString*,int)));
    QObject::connect(client, SIGNAL(removeClientTab(QString,int)), this, SLOT(removeClientTab(QString,int)));
    QObject::connect(client, SIGNAL(clientBytesSent(int, int)), this, SLOT(clientBytesSent(int, int)));
    QObject::connect(client, SIGNAL(endDownload(int)), this, SLOT(endDownload(int)));
    clientList->insert(nbrClient, client);
 }

int Server::send_to_client(QString buffer, int nbr)
{
    QByteArray block;
    QDataStream out(&block, QIODevice::WriteOnly);
    out.setVersion(QDataStream::Qt_4_0);
    out << QString(xmlCreator->createXmlCommandFlow(buffer));
    if (clientList->contains (nbr))
    {
        Client *client = clientList->value(nbr);
        client->getSocket()->write(block);
        client->newResults = true;
        return 1;
    }
    return 0;
}

void Server::send_file_to_client(QString filePath, int fileSize, int nbr)
{
    QByteArray block;
    QDataStream out(&block, QIODevice::WriteOnly);
    out.setVersion(QDataStream::Qt_4_0);
    out << QString(xmlCreator->createXmlSendFileFlow(filePath, fileSize));
    if (clientList->contains (nbr))
    {
        Client *client = clientList->value(nbr);
        client->getSocket()->write(block);
        client->getSocket()->flush();
        client->newResults = true;
    }
}

int Server::send_to_all_client(QString buffer)
{
    QByteArray block;
    QDataStream out(&block, QIODevice::WriteOnly);
    out.setVersion(QDataStream::Qt_4_0);
    out << QString(xmlCreator->createXmlCommandFlow(buffer));
    for (int j = 0; j < 100; ++j) {
        if (clientList->contains (j))
        {
            Client *client = clientList->value(j);
            client->getSocket()->write(block);
            client->getSocket()->flush();
            client->newResults = true;
        }
    }
    return 0;
}

void Server::sendFile(const QString&fn, int nbr)
{
   if (clientList->contains (nbr))
    {
        Client *client = clientList->value(nbr);
        client->newResults = true;
        QFile f(fn);
        if (!f.open(QIODevice::ReadOnly))
        {
            emit warning(tr("An error occured while sending file : The file has not been correctly opened"));
            emit fileSent(nbr);
            return;
        }
        while (!f.atEnd())
        {
            QByteArray b = f.read(1024);
            client->getSocket()->write(b);
            emit bytesSent(b.size());
            if(client->getSocket()->waitForBytesWritten(5000)==false) 
            {
                emit warning(tr("An error occured while sending file : The socket is busy."));
                emit fileSent(nbr);
                f.close();
                return;
            }
        }
        emit fileSent(nbr);
        f.close();
    }
}

void Server::ask_file_from_client(QString fileName, int nbr)
{
    QByteArray block;
    QDataStream out(&block, QIODevice::WriteOnly);
    out.setVersion(QDataStream::Qt_4_0);
    out << QString(xmlCreator->createXmlAskFileFlow(fileName));
   if (clientList->contains (nbr))
    {
        Client *client = clientList->value(nbr);
        client->getSocket()->write(block);
        client->getSocket()->flush();
    }
}

int Server::remove_client(int nbr)
{
    if (clientList->contains(nbr))
    {
        clientList->remove(nbr);
        blackList->remove(nbr);
        emit stopClient(nbr);
    }
    return 1;
}

int Server::stop_server()
{
    serverStatus = false;
    nbrClient = 0;
    if (!clientList->empty())
    {
        for (int j = 0; j <= 100; ++j) {
            if (clientList->contains(j))
            {
                    remove_client(j);
            }
        }
    }
    close();
    return 0;
}

bool Server::getServerStatus()
{
    return serverStatus;
}

int Server::get_client_number()
{
    return clientList->size();
}

QMap<int, Client*>* Server::get_clientList()
{
    return clientList;
}

void Server::sendMessage(QString msg)
{
    emit displayFileTransfered(msg);
}

void Server::getResults(QString *results, int clientNbr)
{
    emit sendResults(results, clientNbr);
}

void Server::getClientParameters(QStringList params, int nbr)
{
    bool existingClient = false;
    QString ip = params.at(2);
    for (int j = 2; j < params.size(); ++j) {
        if (params.at(j).length() <= 15)
            ip = params.at(j);
    }
    for (int j = 0; j < blackList->size(); ++j) {
            if (QString::compare(blackList->value(j), ip, Qt::CaseSensitive) != 0) {
                existingClient = true;
                if (alreadyWarm == false)
                    emit warning(tr("Another client workstation with an already registered IP address tryed to connect to this server and has been rejected"));
                alreadyWarm = true;
                remove_client(nbr);
            }
    }
    for (int j = 0; j < filteredIpList.size(); ++j) {
            if (strcmp(qPrintable(filteredIpList.at(j)), qPrintable(ip)) == 0) {
                existingClient = true;
                if (filteredBool == false)
                    emit warning(tr("A filtered IP address tryed to connect. The KontrolPack server rejected the remote client"));
                filteredBool = true;
                remove_client(nbr);
            }
    }
    if (!existingClient) {
        emit sendClientParameters(params, nbr);
        clientNameList->insert(nbr, params.at(0));
        for (int j = 2; j < params.size(); ++j) {
            if (params.at(j).length() <= 15)
                blackList->insert(nbr, params.at(0));
        }
    }
}

void Server::removeClientTab(QString title, int clientNbr)
{	
    if (clientList->contains(clientNbr))
    {
            clientList->remove(clientNbr);
            blackList->remove(clientNbr);
    }
    if (nbrClient > 0)
    {
            nbrClient = clientList->size();
    }
    emit removeTab(title, clientNbr);
}

void Server::error(QString error)
{
    emit errorOccuredWhileStartingServer(error);
}

void Server::getClientContext(QStringList context, int clientNbr)
{
    emit sendClientContextToGui(context, clientNbr);
}

void Server::clientBytesSent(int bytesNbr, int fileSize)
{
    emit clientBytesSentSignal(bytesNbr, fileSize);
}

void Server::endDownload(int clientNbr)
{
    emit endDownloadSignal(clientNbr);
}
