/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "auth.h"
#include "connection.h"
#include "connectioninfomanager.h"
#include "crypto.h"
#include "tool.h"

#include "user_record.h"

using namespace Form;

Entry UserRecord::loadEntry( const QString &id, PagePolicy::State )
{
  QString filter( "objectClass=*" );
  QLdapResponse response = Connection::self()->search( id, QLdap::Base, filter );

  if ( !response.isValid() || response.entries().isEmpty() )
    return Entry();

  QLdapEntry ldapEntry = response.entries().first();
  Entry entry = Entry::fromLdapEntry( ldapEntry );

  QStringList parts = id.split( "," );
  if ( parts.contains( "cn=groups" ) )
    entry.setValue( "accttype", "group" );
  else if ( parts.contains( "cn=resources" ) )
    entry.setValue( "accttype", "resource" );
  else if ( parts.contains( "cn=internal" ) )
    entry.setValue( "accttype", "internalUser" );
  else
    entry.setValue( "accttype", "user" );

  if ( entry.values( "kolabInvitationPolicy" ).isEmpty() )
    entry.setValue( "kolabInvitationPolicy", "ACT_MANUAL" );

  return entry;
}

bool UserRecord::saveEntry( const Entry &entry, PagePolicy::State state, QString &errorMsg )
{
  if ( state == PagePolicy::Add || state == PagePolicy::Modify ) {

    QLdapEntry ldapEntry;
    ldapEntry.addValue( "objectClass", "top" );
    ldapEntry.addValue( "objectClass", "inetOrgPerson" );
    ldapEntry.addValue( "objectClass", "kolabInetOrgPerson" );

    ldapEntry.setValue( "givenName", entry.value( "givenName" ) );
    ldapEntry.setValue( "sn", entry.value( "sn" ) );
    ldapEntry.setValue( "cn", entry.value( "givenName" ) + " " + entry.value( "sn" ) );

    // handle password
    if ( !entry.value( "password" ).isEmpty() ) {
      QByteArray data = "{sha}" + Crypto::sha1( entry.value( "password" ).toUtf8() ).toBase64();
      ldapEntry.setValue( "userPassword", QString::fromUtf8( data ) );
    } else {
      ldapEntry.setValue( "userPassword", entry.value( "userPassword" ) );
    }

    ldapEntry.setValue( "mail", entry.value( "mail" ).toLower() );
    ldapEntry.setValue( "uid", entry.value( "uid" ).toLower() );

    if ( state == PagePolicy::Add ) {
      if ( entry.value( "uid" ).isEmpty() )
        ldapEntry.setValue( "uid", entry.value( "mail" ).toLower() );
    }

    ldapEntry.setValue( "kolabHomeServer", entry.value( "kolabHomeServer" ) );

    if ( !entry.value( "title" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "title", entry.value( "title" ) );

    if ( !entry.value( "o" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "o", entry.value( "o" ) );

    if ( !entry.value( "ou" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "ou", entry.value( "ou" ) );

    if ( !entry.value( "roomNumber" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "roomNumber", entry.value( "roomNumber" ) );

    if ( !entry.value( "street" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "street", entry.value( "street" ) );

    if ( !entry.value( "postOfficeBox" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "postOfficeBox", entry.value( "postOfficeBox" ) );

    if ( !entry.value( "postalCode" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "postalCode", entry.value( "postalCode" ) );

    if ( !entry.value( "l" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "l", entry.value( "l" ) );

    if ( !entry.value( "c" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "c", entry.value( "c" ) );

    if ( !entry.value( "telephoneNumber" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "telephoneNumber", entry.value( "telephoneNumber" ) );

    if ( !entry.value( "facsimileTelephoneNumber" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "facsimileTelephoneNumber", entry.value( "facsimileTelephoneNumber" ) );

    if ( !entry.value( "kolabFreeBusyFuture" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "kolabFreeBusyFuture", entry.value( "kolabFreeBusyFuture" ) );

    const QStringList invitationPolicies = entry.values( "kolabInvitationPolicy" );
    for ( int i = 0; i < invitationPolicies.count(); ++i )
      ldapEntry.addValue( "kolabInvitationPolicy", invitationPolicies[ i ] );

    const QStringList alias = entry.values( "alias" );
    for ( int i = 0; i < alias.count(); ++i )
      ldapEntry.addValue( "alias", alias[ i ] );

    const QStringList delegates = entry.values( "kolabDelegate" );
    for ( int i = 0; i < delegates.count(); ++i )
      ldapEntry.addValue( "kolabDelegate", delegates[ i ] );

    if ( !entry.value( "cyrus-userquota" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "cyrus-userquota", entry.value( "cyrus-userquota" ) );

    QString dnPrefix;

    if ( entry.value( "accttype" ) == "user" )
      dnPrefix = "";
    else if ( entry.value( "accttype" ) == "internalUser" )
      dnPrefix = "cn=internal,";
    else if ( entry.value( "accttype" ) == "group" )
      dnPrefix = "cn=groups,";
    else if ( entry.value( "accttype" ) == "resource" )
      dnPrefix = "cn=resources,";

    QString domainDn = dnPrefix + Connection::self()->baseDn();


    if ( state == PagePolicy::Modify ) {
      QString newDn;
      if ( !ldapEntry.value( "cn" ).isEmpty() )
        newDn = QString( "cn=%1,%2" ).arg( ldapEntry.value( "cn" ), domainDn );
      else
        newDn = entry.id();

      /**
       * The 'cn' of the user has changed, so we have to perform some
       * extra checks.
       */
      if ( entry.id() != newDn ) {
        /**
         * Check for distribution lists with this user as member.
         */
        QString filter = QString( "(&(objectClass=kolabGroupOfNames)(member=%1))" ).arg( entry.id() );
        QLdapResponse response = Connection::self()->search( Connection::self()->baseDn(), QLdap::Sub, filter );

        if ( !response.isValid() ) {
          errorMsg = Connection::self()->errorString();
          return false;
        }

        /**
         * There is at least one matching distribution list...
         */
        if ( !response.entries().isEmpty() ) {
          QStringList listNames;

          QLdapEntry::List entries = response.entries();
          for ( int i = 0; i < entries.count(); ++i ) {
            listNames.append( entries[ i ].value( "cn" ) );
          }

          errorMsg = QObject::tr( "Account could not be modified because the following distribution lists depend on it: " )
                                .arg( listNames.join( ", " ) );
          return false;
        }

        /**
         * Now rename the old entry to the new one by using a temporary entry to
         * avoid data loose on race condition.
         */
        QStringList dnParts = entry.id().split( "," );
        QString randomString = QString::fromLocal8Bit( dnParts.first().toLocal8Bit().toBase64() );
        randomString.replace( "=", "_" );
        randomString.replace( ",", "_" );

        dnParts.removeFirst(); // remove the 'cn=' entry
        QString tmpBaseDn = dnParts.join( "," );

        QString tmpDn = QString( "cn=%1" ).arg( randomString );

        if ( !Connection::self()->rename( entry.id(), tmpDn, tmpBaseDn, false ) ) {
          errorMsg = QObject::tr( "Could not rename %1 to %2: %3" )
                                .arg( entry.id(), tmpDn, Connection::self()->errorString() );
          return false;
        }
        if ( !Connection::self()->add( newDn, ldapEntry ) ) {
          errorMsg = QObject::tr( "Could not add %1 as %2: %3" )
                                .arg( entry.id(), newDn, Connection::self()->errorString() );
          return false;
        }
        if ( !Connection::self()->remove( QString( "%1,%2" ).arg( tmpDn, tmpBaseDn ) ) ) {
          errorMsg = QObject::tr( "Could not remove old entry %1,%2: %3" )
                                .arg( tmpDn, tmpBaseDn, Connection::self()->errorString() );
          return false;
        }
      }
      ldapEntry.setDn( newDn );

      /**
       * Users are not allowed to modify some attributes, so
       * remove them here from the query.
       */
      if ( Auth::self()->group() == Auth::Users ) {
        ldapEntry.clearValue( "sn" );
        ldapEntry.clearValue( "cn" );
        ldapEntry.clearValue( "mail" );
        ldapEntry.clearValue( "uid" );
        ldapEntry.clearValue( "kolabHomeServer" );
      }

      if ( !Connection::self()->modifyAttributes( ldapEntry.dn(), ldapEntry ) ) {
        errorMsg = Connection::self()->errorString();
        return false;
      } else {
        return true;
      }
    } else if ( state == PagePolicy::Add ) {
      QString dn = QString( "cn=%1,%2" ).arg( ldapEntry.value( "cn" ), domainDn );

      ldapEntry.setDn( dn );
      if ( !Connection::self()->add( ldapEntry.dn(), ldapEntry ) ) {
        errorMsg = Connection::self()->errorString();
        return false;
      }

      return true;
    }
  } else if ( state == PagePolicy::Delete ) {
    /**
     * Check for distribution lists with this user as member.
     */
    QString filter = QString( "(&(objectClass=kolabGroupOfNames)(member=%1))" ).arg( entry.id() );
    QLdapResponse response = Connection::self()->search( Connection::self()->baseDn(), QLdap::Sub, filter );

    if ( !response.isValid() ) {
      errorMsg = Connection::self()->errorString();
      return false;
    }

    QLdapEntry::List entries = response.entries();
    for ( int i = 0; i < entries.count(); ++i ) {
      QLdapEntry distributionList = entries[ i ];
      if ( distributionList.values( "member" ).count() == 1 ) {
        /**
         * The user is the only member of this group, so don't remove it.
         */
        errorMsg = QObject::tr( "Account could not be deleted, distribution list '%1' depends on it" )
                              .arg( distributionList.value( "cn" ) );
        return false;
      } else {
        /**
         * Remove the user from this distribution list.
         */
        QLdapEntry deleteEntry;
        deleteEntry.setDn( distributionList.dn() );
        deleteEntry.setValue( "member", entry.id() );

        if ( !Connection::self()->removeAttributes( deleteEntry.dn(), deleteEntry ) ) {
          errorMsg = Connection::self()->errorString();
          return false;
        }
      }
    }

    /**
     * Finally delete the user.
     */
    QString ldapError;
    if ( !Tool::deleteObject( Connection::self(), entry.id(), ldapError ) ) {
      errorMsg = QObject::tr( "Could not delete user '%1': %2" )
                            .arg( entry.value( "cn" ), ldapError );
      return false;
    }
  }

  return true;
}
