/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QtCore/QTime>

#include "qldap.h"

#include "tool.h"

static bool deleteCommonObject( QLdap* connection, const QString &dn, QString &errorMsg, bool deleteNow, bool nukePassword )
{
  if ( deleteNow ) {
    if ( !connection->remove( dn ) ) {
      errorMsg = connection->errorString();
      return false;
    }
  } else {
    const QLdapEntry kolabEntry = Tool::kolabSettings( connection );

    QLdapEntry entry;

    /**
     * Set all defined kolabHosts in the kolabDeleteflag.
     */
    QStringList kolabHosts = kolabEntry.values( "kolabHost" );
    for ( int i = 0; i < kolabHosts.count(); ++i )
      entry.addValue( "kolabDeleteflag", kolabHosts[ i ] );

    /**
     * To block the login of a deleted object between the deletion and the cleanup
     * we fill the password with random garbage.
     */
    if ( nukePassword ) {
      entry.setValue( "userPassword", QString( "{sha}%1" ).arg( Tool::randomString( 32 ) ) );
    }

    if ( !connection->modifyAttributes( dn, entry ) ) {
      errorMsg = connection->errorString();
      return false;
    }
  }

  return true;
}

int Tool::numberOfMails( QLdap* connection, const QString &baseDn, const QString &email, const QStringList &excludedDns )
{
  /**
   * At first count the matching users...
   */
  const QString filter = QString( "(|(mail=%1)(alias=%2)(uid=%3))" ).arg( email, email, email );
  const QStringList attributes( "dn" );

  const QLdapResponse response = connection->search( baseDn, QLdap::Sub, filter, attributes );

  int count = 0;

  const QLdapEntry::List entries = response.entries();
  if ( !excludedDns.isEmpty() ) {
    for ( int i = 0; i < entries.count(); ++i ) {
      if ( excludedDns.contains( entries[ i ].dn() ) )
        continue;

      count++;
    }
  } else
    count += entries.count();

  return count;
};

QString Tool::dnForMailOrAlias( QLdap* connection, const QString &value )
{
  const QString filter = QString( "(&(objectClass=kolabInetOrgPerson)(|(mail=%1)(alias=%2)))" ).arg( value, value );
  const QStringList attributes( "dn" );

  const QLdapResponse response = connection->search( connection->baseDn(), QLdap::Sub, filter, attributes );
  if ( !response.isValid() || response.entries().isEmpty() )
    return QString();

  return response.entries().first().dn();
}

bool Tool::deleteObject( QLdap* connection, const QString &dn, QString &errorMsg, bool deleteNow )
{
  return deleteCommonObject( connection, dn, errorMsg, deleteNow, true );
}

bool Tool::deleteSharedFolder( QLdap* connection, const QString &dn, QString &errorMsg, bool deleteNow )
{
  return deleteCommonObject( connection, dn, errorMsg, deleteNow, false );
}

bool Tool::deleteGroupOfNames( QLdap* connection, const QString &dn, QString &errorMsg, bool deleteNow )
{
  return deleteCommonObject( connection, dn, errorMsg, deleteNow, false );
}

QLdapEntry Tool::kolabSettings( QLdap* connection )
{
  const QString filter = QString( "objectClass=*" );
  const QString dn = QString( "k=kolab,%1" ).arg( connection->baseDn() );

  const QLdapResponse response = connection->search( dn, QLdap::Base, filter );
  if ( !response.isValid() || response.entries().isEmpty() )
    return QLdapEntry();

  return response.entries().first();
}

QString Tool::randomString( int length )
{
  char *characters = "abcdefghijklmnopqrstuvwABCDEFGHIJKLMNOPQRSTUVW0123456789";

  // initiate seed
  qsrand( QTime::currentTime().msec() );

  QString retval;
  for ( int i = 0; i < length; ++i ) {
    retval.append( characters[ qrand() % 56 ] );
  }

  return retval;
}

QString Tool::escape( const QString &value )
{
  /**
    From RFC-2254:

    If a value should contain any of the following characters

    Character       ASCII value
    ---------------------------
    *               0x2a
    (               0x28
    )               0x29
    \               0x5c
    NUL             0x00

   the character must be encoded as the backslash '\' character (ASCII
   0x5c) followed by the two hexadecimal digits representing the ASCII
   value of the encoded character. The case of the two hexadecimal
   digits is not significant.
   */
  QString retval( value );

  retval.replace( '\\', QChar( 0x5c ) );
  retval.replace( '*', QChar( 0x2a ) );
  retval.replace( '(', QChar( 0x28 ) );
  retval.replace( ')', QChar( 0x29 ) );
  retval.replace( '\0', QChar( 0x00 ) );

  return retval;
}

QString Tool::escapeDn( const QString &dn )
{
  /**
    DN component escaping as described in RFC-2253
   */
  QString retval( dn );

  retval.replace( '\\', "\\\\" );
  retval.replace( ',', "\\," );
  retval.replace( '+', "\\," );
  retval.replace( '<', "\\<" );
  retval.replace( '>', "\\>" );
  retval.replace( ';', "\\;" );

  if ( retval.startsWith( '#' ) )
    retval.prepend( '\\' );

  return retval;
}
