/***************************************************************************
                          msnnotificationconnection.h  -  description
                             -------------------
    begin                : Thu Jan 23 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef MSNNOTIFICATIONCONNECTION_H
#define MSNNOTIFICATIONCONNECTION_H

#include <qobject.h>
#include <qptrlist.h>
#include <qtimer.h>
#include <qdom.h>

#include "chatinformation.h"
#include "msnconnection.h"
#include "tcpconnectionbase.h"

#include "soap/httpsoapconnection.h"


// Forward declarations
class ChatInformation;
class ChatMessage;
class Contact;
class ContactList;
class KConfig;
class KMessTest;
class MimeMessage;
class PassportLoginService;
class OfflineImService;

/**
 * @brief The primary connection to the MSN server.
 *
 * The notification connection is initialized when the
 * user signs in to the MSN Messenger service.
 * When this connection is dropped, the user is signed off as well.
 * The notification connection is used for the following actions:
 * - managing the contact list
 * - starting new chat sessions
 * - receive invitations for a chat session
 * - receive status updates about contacts, e.g.:
 *   - online/offline changes
 *   - new e-mail
 *   - personal status messages
 *   - now playing information
 *   - new offline-im messages
 *
 * Most protocol commands are echo'ed back to the client.
 * This allows a Model-View-Controller based protocol handling,
 * see MsnConnection::parseCommand() for details.
 *
 * When the connection to the notification server is established,
 * the the PassportLoginService class is used to retrieve an authentication token.
 * This token is provided to the <code>USR</code> command complete the login process.
 *
 * When a contact is updated, it's information in the
 * ContactList class is updated automatically.
 * Contacts are organized in multiple "lists".
 * This differs from the "groups" seen in the main window!
 * See the Contact class for information about "lists" and "groups".
 *
 * New chat invitations are handled by the <code>XFR</code> command.
 * The server response contains the login data for a chat session (or "chat room")
 * at a switchboard server. This information is used to start the chat session.
 * The information about new chats is forwarded to the ChatMaster first.
 * This class looks for existing chats with a contact,
 * or opens a new chat window instead when needed.
 * The actual chat session is handled by the MsnSwitchboardConnection class.
 * Once the chat session is created, the other contact
 * is invited to the session at the switchboard server.
 *
 * Some contact commands include an MsnObject argument.
 * This is the "ticket" to request a display picture later,
 * by "inviting" the contact to transfer the data belonging to the MsnObject.
 * Invitations for file and picture transfers are not handled by the notification connection however.
 * A chat session needs to be started first. The invitation for the picture or file transfer can be sent there.
 *
 * Note that each connection starts with a version exchange.
 * Clients inform the server about the versions of the protocol it supports.
 * The commands documented here may differ from other clients using a different protocol version.
 * Currently KMess uses <code>MSNP12</code> for all the notification connection.
 * New features are added to new protocol versions only.
 * Therefore, something as simple as a 'personal status message' required
 * an upgrade to a complete new protocol version. The <code>UBX</code>
 * command - which contains the personal status message - is only sent as of <code>MSNP11</code>.
 * 
 * The <code>CHG</code> and <code>NLN</code> commands include a bitwise flag
 * with the supported client-to-client features. This includes the
 * MSNC version for the invitation system, the type of the client ('normal', 'mobile', 'web messenger') and features like 'winks', 'webcam', 'folder sharing'.
 * By advertising these values, a client promoses the feature is fully supported.
 * Changing the advertised MSNC version changes the type of P2P messages sent by another client.
 * Hence, changing the capabilities value could pose a major impact.
 * The list of known flags can be found in the ContactBase::MsnClientCapabilities enum.
 *
 * @author Mike K. Bennett
 * @ingroup NetworkCore
 */
class MsnNotificationConnection : public MsnConnection
{
  Q_OBJECT

  friend class KMessTest;

  public:
    // The constructor
                         MsnNotificationConnection();
    // The destructor
                        ~MsnNotificationConnection();
    // Request a change in the user's status
    void                 changeStatus( const QString& newStatus );
    // Close the connection with the server
    virtual void         closeConnection();
    // Return the contact list
    const ContactList*   getContactList() const;
    // Initialize the object
    bool                 initialize();
    // Open a connection to the server
    bool                 openConnection();
    // Save contact list properties
    void                 saveProperties(KConfig *config);

  public slots: // Public slots
    // Copy contact to another group.
    void                 addContactToGroup(QString handle, QString groupId);
    // Add an existing contact to the friends list
    void                 addExistingContact(QString handle);
    // Add a new contact to the list
    void                 addNewContact(QString handle);
    // Adds the given contact to the allow list.
    void                 allowContact(QString handle);
    // Add a new group
    void                 addGroup(QString name);
    // Block the given contact
    void                 blockContact(QString handle);
    // Change the current media of the user.
    void                 changeCurrentMedia( const QString &application, const QString &mediaType, bool enabled,
                                             const QString &formatString, const QStringList arguments );
    // The msn object has required, requires a notification message
    void                 changedMsnObject();
    // Change the friendly name of the user.
    void                 changeFriendlyName( QString newName );
    // Change the friendly name of another contact.
    void                 changeFriendlyName( QString handle, QString newName );
    // Change the personal message of the user.
    void                 changePersonalMessage( QString newMessage );
    // Change a user property
    void                 changeProperty( QString type, QString value );
    // Change a contact property
    void                 changeProperty( QString handle, QString type, QString value );
    // Move the contact to another group.
    void                 moveContact(QString handle, QString fromGroupId, QString toGroupId);
    // Request a switchboard to connect to a contact
    void                 requestSwitchboard( QString handle, ChatInformation::ConnectionType type );
    // Rename a group
    void                 renameGroup(QString id, QString newName);
    // Remove a contact from the contact list completely
    void                 removeContact(QString handle, bool block = true);
    // Remove a contact from a single group
    void                 removeContactFromGroup(QString handle, QString groupId);
    // Remove the current media advertising in the personal status.
    void                 removeCurrentMedia();
    // Remove a group
    void                 removeGroup(QString id);
    // Unblock the given contact
    void                 unblockContact(QString handle);

  protected: // Protected methods
    // Test whether the given command is a payload command
    virtual bool         isPayloadCommand(const QString &command) const;

  protected slots: // Protected slots
    // The socket connected, so send the version command.
    void                 connectionSuccess();

  private: // Private methods
    // Register a SOAP client to the class.
    void                 addSoapClient( HttpSoapConnection *client );
    // Create the Offline-IM service
    OfflineImService    *createOfflineImService();
    // Create the passport login handler
    PassportLoginService *createPassportLoginService();
    // Show a .net status message.
    void                 dotNetMessage(QString message);
    // Go online
    void                 goOnline();
    // Received the addition of a contact to a list or list and group
    void                 gotAdc(const QStringList& command);
    // Received the addition of a group
    void                 gotAdg(const QStringList& command);
    // Received confirmation of the user's status change
    void                 gotChg(const QStringList& command);
    // Received a challenge from the server
    void                 gotChl(const QStringList& command);
    // Received a version update from the server
    void                 gotCvr(const QStringList& command);
    // Received notice that a contact went offline
    void                 gotFln(const QStringList& command);
    // Received notice that a contact is already online
    void                 gotIln(const QStringList& command);
    // Received group information
    void                 gotLsg(const QStringList& command);
    // Received contact information
    void                 gotLst(const QStringList& command);
    // Received notice that a contact has changed status
    void                 gotNln(const QStringList& command);
    // Received notice of disconnetion from the server
    void                 gotOut(const QStringList& command);
    // Received a propertu change (friendly name, phone number, etc..)
    void                 gotPrp(const QStringList& command);
    // Received confirmation of change in a group's name
    void                 gotReg(const QStringList& command);
    // Received confirmation of a conact's removal from list or list and group
    void                 gotRem(const QStringList& command);
    // Received confirmation of a group's removal
    void                 gotRmg(const QStringList& command);
    // Received a chat request from a contact
    void                 gotRng(const QStringList& command);
    // Received synchronization information
    void                 gotSyn(const QStringList& command);
    // Received details of a contact's personal message
    void                 gotUbx(const QStringList &command, const QByteArray &payload);
    // Received the folder and command info for Hotmail's inbox or compose
    void                 gotUrl(const QStringList& command);
    // Received user authentication information
    void                 gotUsr(const QStringList& command);
    // Received version information
    void                 gotVer(const QStringList& command);
    // Received server transfer information
    void                 gotXfr(const QStringList& command);
    // Add a contact to the given list or, if applicable, group
    int                  putAdc(QString handle, QString list, QString groupId = QString::null);
    // Remove a contact from the given list or, if applicable, group
    int                  putRem(QString handle, QString list, QString groupId = QString::null);
    // Send the personal status and current media fields.
    void                 putUux();
    // Send the version command
    void                 putVer();
    // Parse a regular command
    void                 parseCommand(const QStringList& command);
    // Parse a message command
    void                 parseMessage(const QStringList& command, const MimeMessage &message);
    // Parse a payload command
    virtual void         parsePayloadMessage(const QStringList &command, const QByteArray &payload);

  private slots: // Private slots
    // Check whether login was successful within time limit
    void                 checkLogin();
    // Removes any expired switchboard requests, so new ones can be made
    void                 checkSwitchboardsTimeout();
    // The passport login failed, username/password was incorrect
    void                 loginIncorrect();
    // The passport login succeeded
    void                 loginSucceeded(QString authentication);
    // Received the Mail-Data field over SOAP or at the notification connection.
    void                 receivedMailData( QDomElement mailData );
    // An Offline-IM message was downloaded
    void                 receivedOfflineIm( const QString &messageId, const QString &from, const QString &to,
                                            const QDateTime &date, const QString &body,
                                            const QString &runId, int sequenceNum );
    // A SOAP request failed with an error message
    void                 soapRequestFailed(HttpSoapConnection *client, const QString &reason);
    // A SOAP request failed with an SOAP fault
    void                 soapRequestFailed(HttpSoapConnection *client, const QString &faultCode,
                                           const QString &faultString, QDomElement faultNode);

  private: // Private attributes

    struct OfflineImMessage
    {
      QString   messageId;
      QString   from;
      QDateTime date;
      QString   body;
      QString   runId;
      int       sequenceNum;
    };

    // The 'p' value for the login
    QString              authP_;
    // The 't' value for the login
    QString              authT_;
    // A pointer to the list of contacts
    ContactList         *contactList_;
    // Whether or not the object was initialized
    bool                 initialized_;
    // The last current media set.
    QString              lastCurrentMedia_;
    // The last Personal Status Message set.
    QString              lastPsm_;
    // A timer to timeout the user's logon
    QTimer               loginTimer_;
    // The number of contact read in
    int                  noContacts_;
    // The received offline messages.
    QPtrList<OfflineImMessage> offlineImMessages_;
    // The Offline-IM SOAP client
    OfflineImService    *offlineImService_;
    // A list of switchboard requests which haven't yet been given servers
    QPtrList<ChatInformation> openRequests_;
    // The object than handles the passport login
    PassportLoginService *passportLoginService_;
    // The pending offline-im messages
    QStringList          pendingOfflineImMessages_;
    // The persistent soap clients.
    QPtrList<HttpSoapConnection> soapClients_;
    // The total number of contacts in the contact list
    int                  totalNoContacts_;

  signals: // Public signals
    // Signal a chat message, used for offline-im messages
    void                 offlineMessage( const ChatMessage &message );
    // Signal that the connection has been made sucessfully
    void                 connected();
    // Signal that a contact added you
    void                 contactAddedUser(const Contact *contact);
    // Signal that a new email has been received
    void                 newEmail(QString sender, QString subject, bool inInbox, QString command, QString folder, QString url);
    // A new switchboard with a contact has been opened
    void                 startedSwitchboard( const ChatInformation &chatInfo );
};

#endif
