# Copyright (c) 2008 Hideki Ikemoto
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

class ThreadView < Qt::Widget
  slots 'reload()'
  slots 'show_write_dialog()'
  slots 'remove_self()'

  attr_reader :dat_url

  def initialize(parent)
    super(parent)

    layout = Qt::VBoxLayout.new(self)

    buttons_layout = Qt::HBoxLayout.new
    layout.addLayout(buttons_layout)

    @write_button = Qt::ToolButton.new(self)
    @write_button.setPixmap(KDE::SmallIcon("pencil"))
    @write_button.setEnabled(false)
    buttons_layout.addWidget(@write_button)
    connect(@write_button, SIGNAL("clicked()"), self, SLOT("show_write_dialog()"))

    @reload_button = Qt::ToolButton.new(self)
    @reload_button.setPixmap(KDE::SmallIcon("reload"))
    @reload_button.setEnabled(false)
    buttons_layout.addWidget(@reload_button)
    connect(@reload_button, SIGNAL("clicked()"), self, SLOT("reload()"))

    spacer = Qt::SpacerItem.new(30, 20,
                                Qt::SizePolicy::Expanding,
                                Qt::SizePolicy::Minimum)
    buttons_layout.addItem(spacer)

    @close_button = Qt::ToolButton.new(self)
    @close_button.setPixmap(KDE::SmallIcon("fileclose"))
    @close_button.setEnabled(false)
    buttons_layout.addWidget(@close_button)
    connect(@close_button, SIGNAL("clicked()"), self, SLOT("remove_self()"))

    @html_part = KitaHTMLPart.new(self)
    @html_part.set_view(self)
    layout.addWidget(@html_part.view)

    @html_part.begin()
    @html_part.end()
  end

  def show_thread(dat_url)
    @dat_url = dat_url

    @dat = DatInfo.new(@dat_url)
    @dat.load

    write(@dat)

    ViewMediator.set_url(Util.dat_url_to_thread_url(dat_url))
    enable_buttons
  end

  def write(dat)
    @html_part.begin()
    @html_part.write("<html><head><style type=\"text/css\">")
    @html_part.write(Const::Default_Style_Sheet)
    @html_part.write("</style></head>")
    dat.each {|comment|
      @html_part.write(comment.to_html)
    }
    @html_part.write("</html>")
    @html_part.end()
  end

  def enable_buttons
    @write_button.setEnabled(true)
    @reload_button.setEnabled(true)
    @close_button.setEnabled(true)
  end

  def reload
    dat = DatInfo.new(@dat_url)
    dat.reload

    write(dat)
  end

  def show_write_dialog
    dialog = WriteDialog.instance
    dialog.dat_url = @dat_url
    dialog.thread_name = Util.truncate_str(thread_name, 20)
    dialog.board_name = BoardManager.instance.get_name(Util.dat_url_to_board_url(@dat_url))
    dialog.show
    dialog
  end

  def show_write_dialog_with_str(str)
    dialog = show_write_dialog
    dialog.body_edit.text = str
  end

  def thread_name
    @dat.thread_name
  end

  def remove_self
    ViewMediator::remove_thread_view(self)
  end

  def body(num)
    @dat.body(num)
  end

  def copy
    selected_text = @html_part.selectedText
    return if selected_text == nil
    return if selected_text.empty?

    clipboard = Qt::Application.clipboard
    clipboard.setText(selected_text, Qt::Clipboard::Clipboard)
  end
end
