#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <glib.h>
#include <glib-object.h>

#include "kpcalendarentry.h"
#include "kputil.h"
#include "kpresults.h"

enum {
  RESULT_ADDED_SIGNAL,
  RESULT_REMOVED_SIGNAL,
  LAST_SIGNAL
};

/* GObject stuff */
static guint      kp_results_signals[LAST_SIGNAL] = { 0 };

static void       kp_results_class_init           (GObjectClass *klass,
                                                   gpointer data);
static void       kp_results_instance_init        (GObject *object,
                                                   gpointer data);
static void       kp_results_instance_finalize    (GObject *object);

/* KPCalendarEntry implementations */
static gchar     *kp_results_get_human_name       (KPCalendarEntry *entry);
static gchar     *kp_results_get_icon_name        (KPCalendarEntry *entry);
static xmlNodePtr kp_results_to_xml               (KPCalendarEntry *entry);
static G_CONST_RETURN gchar *
                  kp_results_to_string            (KPCalendarEntry *entry);
static gboolean   kp_results_parse                (KPCalendarEntry *entry,
                                                   xmlNodePtr node);


GType
kp_results_get_type ()
{
  static GType kp_results_type = 0;

  if (!kp_results_type) {
    static const GTypeInfo kp_results_info = {
      sizeof (KPResultsClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_results_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPResults),
      0,
      (GInstanceInitFunc) kp_results_instance_init,
      NULL
    };
    kp_results_type = g_type_register_static (KP_TYPE_CALENDAR_ENTRY,
                                             "KPResults",
                                             &kp_results_info,
                                              0);
  }
  return kp_results_type;
}


static void
kp_results_class_init (GObjectClass *klass, gpointer data)
{
  KPCalendarEntryClass *entry_class;
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_results_instance_finalize;

  entry_class = KP_CALENDAR_ENTRY_CLASS (klass);
  entry_class->get_human_name = kp_results_get_human_name;
  entry_class->get_icon_name = kp_results_get_icon_name;
  entry_class->to_string = kp_results_to_string;
  entry_class->to_xml = kp_results_to_xml;
  entry_class->parse = kp_results_parse;

  kp_results_signals[RESULT_ADDED_SIGNAL]
    = g_signal_new ("result-added",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPResultsClass, result_added),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__UINT_POINTER,
                    G_TYPE_NONE,
                    2,
                    G_TYPE_UINT,
                    G_TYPE_POINTER);
  
  kp_results_signals[RESULT_REMOVED_SIGNAL]
    = g_signal_new ("result-removed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPResultsClass, result_removed),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__UINT_POINTER,
                    G_TYPE_NONE,
                    2,
                    G_TYPE_UINT,
                    G_TYPE_POINTER);
}



static void
kp_results_instance_init (GObject *object, gpointer data)
{
  KPResults *res;
  res = KP_RESULTS (object);
  res->list = NULL;
  res->location = NULL;
  
  res->order = KP_RESULT_ORDER_ASC;
  res->type = KP_RESULT_TYPE_TIME;
}


  
static void
kp_results_instance_finalize (GObject *object)
{
  GObjectClass *parent_class;
  KPResults *res = KP_RESULTS (object);

  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);

  if (res->name)
    g_free (res->name);
}

/**
 * kp_results_new:
 * 
 * Create a new unset instance of #KPResults.
 * 
 * Returns: A new #KPResults.
 */
KPResults *
kp_results_new (const gchar *title, KPResultType type)
{
  KPResults *res;

  res = g_object_new (kp_results_get_type (), NULL);
  kp_results_set_title (res, title);
  
  return res;
}


KPResults *
kp_results_copy (KPResults *results)
{
  KPResults *copy;
  KPResult *res_copy;
  KPResult *res;
  GList *node;
  
  copy = kp_results_new (kp_results_get_title (results), results->type);
   
  for (node = results->list; node; node = node->next) {
    res = KP_RESULT (node->data);
    res_copy = g_new0 (KPResult, 1);
    res_copy->type = res->type;
    res_copy->comment = g_strdup (res->comment);
    res_copy->result = res->result;
    res_copy->competitor = g_new0 (KPCompetitor, 1);
    res_copy->competitor->name = g_strdup (res->competitor->name);
    if (res->competitor->organization)
      res_copy->competitor->organization = g_strdup (res->competitor->organization);

    kp_results_add_result (copy, res_copy);  
  }

  return copy;
}



G_CONST_RETURN gchar *
kp_results_get_title (KPResults *res)
{
  g_return_val_if_fail (KP_IS_RESULTS (res), NULL);
  
  return (G_CONST_RETURN gchar *) res->name;
}


G_CONST_RETURN gchar *
kp_results_get_location (KPResults *res)
{
  g_return_val_if_fail (KP_IS_RESULTS (res), NULL);

  return (G_CONST_RETURN gchar *) res->location;
}


void
kp_results_set_title (KPResults *res, const gchar *name)
{
  g_return_if_fail (KP_IS_RESULTS (res));

  if (res->name)
    g_free (res->name);

  res->name = g_strdup (name);
}


void
kp_results_set_location (KPResults *res, const gchar *location)
{
  g_return_if_fail (KP_IS_RESULTS (res));

  if (res->location)
    g_free (res->location);

  res->location = g_strdup (location);
}


gchar *
kp_results_get_as_string (KPResults *res)
{
  return NULL;
}

gint 
result_cmp_func (gconstpointer a, gconstpointer b)
{
  KPResult *r1, *r2;

  g_assert (a != NULL);
  g_assert (b != NULL);
  
  r1 = KP_RESULT (a);
  r2 = KP_RESULT (b);

  g_return_val_if_fail (r1->type == r2->type, 0);
  
  switch (r1->type) {
    case KP_RESULT_TYPE_RAW:
      if (r1->result < r2->result)
        return 1;
      else if (r1->result > r2->result)
        return -1;
      else
        return 0;

    case KP_RESULT_TYPE_TIME:
      if (r1->result > r2->result)
        return 1;
      else if (r1->result < r2->result)
        return -1;
      else
        return 0;

    default:
      g_assert_not_reached ();
  }
}

void
kp_results_add_result (KPResults *res, KPResult *r)
{
  gint n;
  
  g_return_if_fail (KP_IS_RESULTS (res));
  g_return_if_fail (r != NULL);
  g_return_if_fail (r->competitor != NULL);
  g_return_if_fail (r->competitor->organization != NULL);
  g_return_if_fail (r->competitor->name != NULL);
  g_return_if_fail (r->comment != NULL);
  
  res->list = g_list_insert_sorted (res->list, r, result_cmp_func);
  n = g_list_index (res->list, r);
  
  g_signal_emit (G_OBJECT (res), kp_results_signals[RESULT_ADDED_SIGNAL],
                 0, (n >= 0) ? (guint) n : 11111, r);
}

void
kp_results_remove_result (KPResults *res, KPResult *r)
{
  gint n;
  
  n = g_list_index (res->list, r);
  g_return_if_fail (n >= 0);
  g_signal_emit (G_OBJECT (res), kp_results_signals[RESULT_REMOVED_SIGNAL],
                 0, (n >= 0) ? n : 100, r);
  res->list = g_list_remove (res->list, r);
}


void
kp_results_add_competitor (KPResults *res, KPCompetitor *comp, gint result)
{
  KPResult *r;
  
  g_return_if_fail (KP_IS_RESULTS (res));
  g_return_if_fail (comp != NULL);

  r = g_new0 (KPResult, 1);
  r->competitor = comp;
  r->result = result;
  r->type = res->type;
  
  res->list = g_list_insert_sorted (res->list, r, result_cmp_func); 
}


guint
kp_results_get_top_result (KPResults *res)
{
  g_return_val_if_fail (KP_IS_RESULTS (res), 0);
  g_return_val_if_fail (res->list != NULL, 0);

  return KP_RESULT (res->list->data)->result;
}


GList *
kp_results_get_list (KPResults *res)
{
  g_return_val_if_fail (KP_IS_RESULTS (res), NULL);
  return res->list;
}

void
kp_results_print (KPResults *res)
{
  GList *node;
  guint i;
  g_return_if_fail (KP_IS_RESULTS (res));

  g_print ("Printing results...\n");
  
  for (i=1, node = res->list; node; i++, node = node->next)
    g_print ("%3d  %40s  %u\n", i, 
            KP_RESULT (node->data)->competitor->name,
            KP_RESULT (node->data)->result);
  
}

static gchar *
kp_results_get_human_name (KPCalendarEntry *entry)
{
  return g_strdup ("Results");
}


static gchar *
kp_results_get_icon_name (KPCalendarEntry *entry)
{
  return g_strdup ("runner.xpm");
}


static xmlNodePtr 
kp_results_to_xml (KPCalendarEntry *entry)
{
  GList *cn;
  xmlNodePtr node, list, comp;
  gchar buf[64];
  guint i;
  
  node = xmlNewNode (NULL, BAD_CAST ("results"));
  list = xmlNewChild (node, NULL, BAD_CAST ("list"), NULL);
 
  (void) xmlNewTextChild (node, NULL, BAD_CAST ("title"),
                          BAD_CAST (kp_results_get_title (KP_RESULTS (entry))));
  (void) xmlNewTextChild (node, NULL, BAD_CAST ("location"),
                          BAD_CAST (kp_results_get_location (KP_RESULTS (entry))));
  
  kp_calendar_time_to_xml (entry->datetime, node); 
  
  for (i=1, cn = KP_RESULTS (entry)->list; cn; i++, cn = cn->next) {
    comp = xmlNewChild (list, NULL, BAD_CAST ("competitor"), NULL);
  
    /* <place> */
    g_snprintf (buf, sizeof (buf) - 1, "%u", i);
    (void) xmlNewTextChild (comp, NULL, BAD_CAST ("place"), BAD_CAST (buf));
    
    /* <result> */
    g_snprintf (buf, sizeof (buf) - 1, "%u", KP_RESULT (cn->data)->result);
    (void) xmlNewTextChild (comp, NULL, BAD_CAST ("result"), BAD_CAST (buf));
    
    /* <name> */
    (void) xmlNewTextChild (comp, NULL, BAD_CAST ("name"), 
                            BAD_CAST (KP_RESULT (cn->data)->competitor->name));
    /* <organization */
    (void) xmlNewTextChild (comp, NULL, BAD_CAST ("organization"), 
                            BAD_CAST (KP_RESULT (cn->data)->competitor->organization));
    /* <comment> */
    if (KP_RESULT (cn->data)->comment)
      (void) xmlNewTextChild (comp, NULL, BAD_CAST ("comment"),
                              BAD_CAST (KP_RESULT (cn->data)->comment));
  }
  return node;
}


static G_CONST_RETURN gchar *
kp_results_to_string (KPCalendarEntry *entry)
{
  gchar *location;
  gchar *title;
  gchar *str;

  location = g_markup_escape_text (kp_results_get_location (KP_RESULTS (entry)), -1);
  title = g_markup_escape_text (kp_results_get_title (KP_RESULTS (entry)), -1);
  
  str = g_strdup_printf ("Results: %s @ %s", 
                         (title) ? title : "Unnamed race",
                         (location) ? location : "Somewhere");

  g_free (location);
  g_free (title);

  return str;
}


gboolean
read_competitors (KPResults *results, xmlNodePtr ptr)
{
  xmlNodePtr node;
  xmlNodePtr cn;
  KPResult *r = NULL;
  gchar *str;
  gdouble val;
  
  g_return_val_if_fail (KP_IS_RESULTS (results), FALSE);
  g_return_val_if_fail (ptr != NULL, FALSE);
  
  for (node = ptr->children; node; node = node->next) {
    if (KP_TAG_MATCH (node, "competitor")) {
      r = g_new0 (KPResult, 1);
      r->competitor = g_new0 (KPCompetitor, 1);
      r->type = KP_RESULT_TYPE_TIME;
       
      for (cn = node->children; cn; cn = cn->next) {
        /* Place tag isn't needed to do the results */
        
        if (KP_TAG_MATCH (cn, "result")) {
          str = (gchar *) xmlNodeGetContent (cn);
          val = kp_number (str);
          g_free (str);
          
          if (val >= 0)
            r->result = (guint) val;
          else
            goto clean;
        }
        if (KP_TAG_MATCH (cn, "name")) {
          r->competitor->name = (gchar *) xmlNodeGetContent (cn); 
        }
        if (KP_TAG_MATCH (cn, "organization")) {
          r->competitor->organization = (gchar *)xmlNodeGetContent (cn);
        }
        if (KP_TAG_MATCH (cn, "comment")) {
          r->comment = (gchar *) xmlNodeGetContent (cn); 
        }
      }
        
      /* These are always needed! */
      if (!r->competitor->name || !r->result)
        goto clean;
       
      results->list = g_list_insert_sorted (results->list, r, result_cmp_func);
    }
  }
  return TRUE;

clean:
  if (r != NULL) {
    if (r->competitor != NULL)
      g_free (r->competitor);
    g_free (r);
  }
  return FALSE;
}


gboolean
kp_results_parse (KPCalendarEntry *entry, xmlNodePtr ptr)
{
  xmlNodePtr node;
  xmlChar *location;
  xmlChar *title;
  xmlChar *date;
  xmlChar *type;
  
  type = xmlGetProp (ptr, BAD_CAST ("type"));
  
  KP_RESULTS (entry)->unit = (gchar *) xmlGetProp (ptr, BAD_CAST ("unit"));
  KP_RESULTS (entry)->type = (type != NULL && strcmp ((const gchar *)type, "time") == 0)
      ? KP_RESULT_TYPE_TIME 
      : KP_RESULT_TYPE_RAW;
  
  for (node = ptr->children; node; node = node->next) {
    if (KP_TAG_MATCH (node, "datetime")) { 
      date = xmlNodeGetContent (node);
      if (!kp_calendar_time_set_datetime (KP_CALENDAR_ENTRY (entry)->datetime,
                                          (const gchar *) date))
        return FALSE;
      g_free (date);
    }

    if (KP_TAG_MATCH (node, "title")) {
      title = xmlNodeGetContent (node);
      kp_results_set_title (KP_RESULTS (entry), (const gchar *) title);
      g_free (title);
    }

    if (KP_TAG_MATCH (node, "location")) {
      location = xmlNodeGetContent (node);
      kp_results_set_location (KP_RESULTS (entry), (const gchar *) location);
      g_free (location);
    }
  
    if (KP_TAG_MATCH (node, "list")) {
      if (!read_competitors (KP_RESULTS (entry), node))
        return FALSE;
    }
  }

  return TRUE;
}
