#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <glib.h>
#include <glib-object.h>

#include "kputil.h"
#include "kpparam.h"

enum {
  CHANGED_SIGNAL,
  LAST_SIGNAL
};

static guint      kp_param_signals[LAST_SIGNAL] = { 0 };

/* GObject stuff */
static void       kp_param_class_init           (GObjectClass *klass,
                                                 gpointer data);
static void       kp_param_instance_init        (GObject *object,
                                                 gpointer data);
static void       kp_param_instance_finalize    (GObject *object);


GType
kp_param_get_type ()
{
  static GType kp_param_type = 0;

  if (!kp_param_type) {
    static const GTypeInfo kp_param_info = {
      sizeof (KPParamClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_param_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPParam),
      0,
      (GInstanceInitFunc) kp_param_instance_init,
      NULL
    };

    kp_param_type = g_type_register_static (G_TYPE_OBJECT,
                                           "KPParam",
                                           &kp_param_info,
                                            0);
  }
  return kp_param_type;
}


static void
kp_param_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_param_instance_finalize;

  kp_param_signals[CHANGED_SIGNAL]
    = g_signal_new ("changed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPParamClass, changed),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__VOID,
                    G_TYPE_NONE,
                    0);
}

  
static void
kp_param_instance_init (GObject *object, gpointer data)
{
  KPParam *param;
  param = KP_PARAM (object);
  
  param->v_int = 0;
  param->v_uint = 0;
  param->v_double = 0;
  param->v_bool = FALSE;
  param->v_string = NULL;
  param->v_pointer = NULL;
  param->v_object = NULL;
}

static void
kp_param_instance_finalize (GObject *object)
{
  GObjectClass *parent_class;
  KPParam *param = KP_PARAM (object);

  kp_param_unset (param);

  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}

/**
 * kp_param_new:
 * 
 * Create a new unset instance of #KPParam.
 * 
 * Returns: A new #KPParam.
 */
KPParam *
kp_param_new (const gchar *name)
{
  KPParam *param;

  param = g_object_new (kp_param_get_type (), NULL);
  kp_param_set_name (param, name);
  
  return param;
}



/**
 * kp_param_isset:
 * param: A #KPParam
 * type: A #KPParamType
 *
 * Check if param is set and it's set for correct type.
 * 
 * Returns: TRUE or FALSE
 */
gboolean
kp_param_isset (KPParam *param, KPParamType type)
{
  g_return_val_if_fail (KP_IS_PARAM (param), FALSE);
  
  return (param->type == type);
}


G_CONST_RETURN gchar *
kp_param_get_name (KPParam *param)
{
  g_return_val_if_fail (KP_IS_PARAM (param), NULL);
  
  return (G_CONST_RETURN gchar *) param->name;
}


void
kp_param_set_name (KPParam *param, const gchar *name)
{
  g_return_if_fail (KP_IS_PARAM (param));

  if (param->name)
    g_free (param->name);

  param->name = g_strdup (name);

  g_signal_emit (G_OBJECT (param), kp_param_signals[CHANGED_SIGNAL], 0);
}


/**
 * kp_param_copy:
 * @param: A #KPParam
 *
 * Copies all the data of @param and returns the copied param.
 *
 * Returns: A newly-allocated copy of @param.
 */
KPParam *
kp_param_copy (KPParam *param)
{
  KPParam *copy;
  
  g_return_val_if_fail (KP_IS_PARAM (param), NULL);
 
  copy = kp_param_new (kp_param_get_name (param));
  
  copy->type = param->type;
  copy->v_int = param->v_int;
  copy->v_uint = param->v_uint;
  copy->v_double = param->v_double;
  copy->v_bool = param->v_bool;
  
  if (param->v_string)
    copy->v_string = g_strdup (param->v_string);

  if (param->v_pointer)
    g_warning ("%s: Pointers can't be copied!\n", __PRETTY_FUNCTION__);
  
  if (copy->v_object)
    g_warning ("%s: Objects can't be copied!\n", __PRETTY_FUNCTION__);
  
  return copy;
}

/**
 * kp_param_unset:
 * @param: A #KPParam
 *
 * Unset value, make the type KP_PARAM_TYPE_UNSET
 * and free all possibly allocated memory.
 */
void
kp_param_unset (KPParam *param)
{
  g_return_if_fail (KP_IS_PARAM (param));
  
  param->type = KP_PARAM_TYPE_UNSET;
  param->v_int = 0;
  param->v_uint = 0;
  param->v_double = 0.0;
  param->v_bool = FALSE;
  
  if (param->v_object != NULL && G_IS_OBJECT (param->v_object)) {
    g_object_unref (param->v_object);
    param->v_object = NULL;
  }
  
  if (param->v_pointer)
    g_free (param->v_pointer);
  if (param->v_string)
    g_free (param->v_string);

  param->v_pointer = param->v_string = NULL;
}

  
void
kp_param_set_int (KPParam *param, gint value)
{
  g_return_if_fail (KP_IS_PARAM (param));
  
  kp_param_unset (param);
  param->type = KP_PARAM_TYPE_INT;
  param->v_int = value;
  g_signal_emit (G_OBJECT (param), kp_param_signals[CHANGED_SIGNAL], 0);
}


void
kp_param_set_uint (KPParam *param, guint value)
{
  g_return_if_fail (KP_IS_PARAM (param));
  
  kp_param_unset (param);
  param->type = KP_PARAM_TYPE_UINT;
  param->v_uint = value;
  g_signal_emit (G_OBJECT (param), kp_param_signals[CHANGED_SIGNAL], 0);
}


void
kp_param_set_double (KPParam *param, gdouble value)
{
  g_return_if_fail (KP_IS_PARAM (param));

  kp_param_unset (param);
  param->type = KP_PARAM_TYPE_DOUBLE;
  param->v_double = value;
  g_signal_emit (G_OBJECT (param), kp_param_signals[CHANGED_SIGNAL], 0);
}

  
void
kp_param_set_boolean (KPParam *param, gboolean value)
{
  g_return_if_fail (KP_IS_PARAM (param));

  kp_param_unset (param);
  param->type = KP_PARAM_TYPE_BOOLEAN;
  param->v_bool = value;
  g_signal_emit (G_OBJECT (param), kp_param_signals[CHANGED_SIGNAL], 0);
}


void
kp_param_set_string (KPParam *param, const gchar *value)
{
  g_return_if_fail (KP_IS_PARAM (param));
  
  kp_param_unset (param);
  param->type = KP_PARAM_TYPE_STRING;
  param->v_string = g_strdup (value);
  g_signal_emit (G_OBJECT (param), kp_param_signals[CHANGED_SIGNAL], 0);
}


void
kp_param_set_pointer (KPParam *param, gpointer value)
{
  g_return_if_fail (KP_IS_PARAM (param));
  
  kp_param_unset (param);
  param->type = KP_PARAM_TYPE_POINTER;
  param->v_pointer = value;
  g_signal_emit (G_OBJECT (param), kp_param_signals[CHANGED_SIGNAL], 0);
}

  
void
kp_param_set_time (KPParam *param, guint value)
{
  g_return_if_fail (KP_IS_PARAM (param));
  
  kp_param_unset (param);
  param->type = KP_PARAM_TYPE_TIME;
  param->v_uint = value;
  g_signal_emit (G_OBJECT (param), kp_param_signals[CHANGED_SIGNAL], 0);
}

/* Get */

gint
kp_param_get_int (KPParam *param)
{
  g_return_val_if_fail (KP_IS_PARAM (param), 0.0);
  return param->v_int + (gint) param->v_uint + (gint) param->v_double;
}


guint
kp_param_get_uint (KPParam *param)
{
  g_return_val_if_fail (KP_IS_PARAM (param), 0.0);
  return param->v_uint + (guint) param->v_int + (guint) param->v_double;
}


gdouble 
kp_param_get_double (KPParam *param)
{
  g_return_val_if_fail (KP_IS_PARAM (param), 0.0);
  return param->v_double + (gdouble) param->v_int + (gdouble) param->v_uint;
}


gboolean
kp_param_get_boolean (KPParam *param)
{
  g_return_val_if_fail (KP_IS_PARAM (param), FALSE);
  return (param->type == KP_PARAM_TYPE_BOOLEAN) ? param->v_bool : FALSE;
}


G_CONST_RETURN gchar*
kp_param_get_string (KPParam *param)
{
  g_return_val_if_fail (KP_IS_PARAM (param), NULL);
  return (param->type == KP_PARAM_TYPE_STRING) ? param->v_string : NULL;
}


gpointer
kp_param_get_pointer (KPParam *param)
{
  g_return_val_if_fail (KP_IS_PARAM (param), NULL);
  return (param->type == KP_PARAM_TYPE_POINTER) ? param->v_pointer : NULL;
}


GObject *
kp_param_get_object (KPParam *param)
{
  g_return_val_if_fail (KP_IS_PARAM (param), NULL);
  return (param->type == KP_PARAM_TYPE_OBJECT) ? param->v_object : NULL;
}


guint
kp_param_get_time (KPParam *param)
{
  g_return_val_if_fail (KP_IS_PARAM (param), 0);

  return (param->type == KP_PARAM_TYPE_TIME) ? 
    param->v_uint + (guint) param->v_double + (guint) param->v_int : 0;
}


void
kp_param_set_automatic_as_string (KPParam *param, const gchar *string)
{
  guint msec;
  gdouble d;

  g_return_if_fail (KP_IS_PARAM (param));
  kp_param_unset (param);

  d = kp_number (string);

  if (d >= 0) {
    if (floor (d) != d) 
      kp_param_set_double (param, d);
    else if (abs ((gint) d) != (gint) d)
      kp_param_set_int (param, (gint) d);
    else
      kp_param_set_uint (param, (guint) d);
    return;
  }
  
  if ((msec = kp_duration_str_to_ms (string))) {
    kp_param_set_time (param, msec);
    return;
  }
  
  if (strcmp (string, "true") == 0) {
    kp_param_set_boolean (param, TRUE);
    return;
  }
  
  if (strcmp (string, "false") == 0) {
    kp_param_set_boolean (param, FALSE);
    return;
  }
  
  kp_param_set_string (param, string);
}

gchar *
kp_param_get_as_string (KPParam *param)
{
  gchar buf[128];
  g_return_val_if_fail (KP_IS_PARAM (param), NULL);

  switch (param->type)
  {
    case KP_PARAM_TYPE_UINT:
      return g_strdup_printf ("%u", param->v_uint);

    /* Use this locale independent way */
    case KP_PARAM_TYPE_DOUBLE:
      g_ascii_formatd (buf, sizeof (buf)-1, "%.3f", param->v_double);
      return g_strdup (buf);

    case KP_PARAM_TYPE_STRING:
      return g_strdup (param->v_string);

    case KP_PARAM_TYPE_TIME:
      return kp_date_mseconds_to_std_string (param->v_uint);

    case KP_PARAM_TYPE_POINTER:
      return g_strdup ("[pointer]");

    case KP_PARAM_TYPE_OBJECT:
      return g_strdup ("[object]");

    case KP_PARAM_TYPE_UNSET:
      return g_strdup ("[unset]");

    default:
      return g_strdup ("[invalid type]");
  }
}


void
kp_param_export_as_xml (KPParam *param, xmlNodePtr parent)
{
  xmlNodePtr child;
  gchar *type;
  gchar *val = NULL;

  switch (param->type)
  {
    case KP_PARAM_TYPE_UINT:
      type = "uint";
      break;

    case KP_PARAM_TYPE_DOUBLE:
    case KP_PARAM_TYPE_INT:
      type = "float";
      break;

    case KP_PARAM_TYPE_STRING:
    case KP_PARAM_TYPE_TIME:
      type = "str";
      break;

    case KP_PARAM_TYPE_BOOLEAN:
      type = "bool";
      break;

    case KP_PARAM_TYPE_POINTER:
    case KP_PARAM_TYPE_OBJECT:
    case KP_PARAM_TYPE_UNSET:
      type = "str";
      val = "[invalid]";
      break;

    default:
      type = "str";
      val = "[invalid]";
  }
  
  child = xmlNewNode (NULL, BAD_CAST ("param"));
  xmlSetProp (child, BAD_CAST ("type"), BAD_CAST (type));
  xmlSetProp (child, BAD_CAST ("name"), BAD_CAST (kp_param_get_name (param)));
  
  if (val) 
    xmlNodeSetContent (child, BAD_CAST (val));
  else {
    val = kp_param_get_as_string (param);
    xmlNodeSetContent (child, BAD_CAST (val));
    g_free (val);
  }

  xmlAddChild (parent, child);
}
