#include <string.h>

#include "page-entry.h"
#include "kphtmlutil.h"
#include "kphtmlwriter.h"

#include "../kpsplitworkout.h"
#include "../kpworkoutmodel.h"
#include "../kipina-i18n.h"
#include "../kpresults.h"
#include "../kpcomment.h"
#include "../kpsplit.h"


static KPHtmlNaviData navi_data = 
{
  1, 1, 1,
  { NULL, NULL, NULL, NULL, NULL },
  { NULL, NULL, NULL, NULL, NULL }
};


static void 
splits_table (KPHtmlWriter *writer, KPSplitWorkout *wo, FILE *fp)
{
  KPSplit *split;
  gchar *duration;
  gchar *distance;
  gchar *speed;
  gchar *pace;
  gchar *str;
  GSList *l;

  kp_html_h1 (fp, _("Splits"));
  kp_html_table_head (fp, _("Number"), _("Sport"), _("Type"), _("Distance"),
                          _("Duration"), _("Pace"), _("Speed"), _("Comment"),
                            NULL);

  for (l = wo->splits; l; l = l->next) {
    split = KP_SPLIT (l->data);
   
    if (split->duration > 0)
      duration = kp_date_mseconds_to_std_string (split->duration);
    else
      duration = NULL;

    if (split->distance > 0.0)
      distance = g_strdup_printf ("%.3f km", split->distance);
    else
      distance = NULL;
   
    pace = kp_date_mseconds_to_std_string (kp_split_get_pace (split));
    speed = g_strdup_printf ("%.1f km/h", kp_split_get_speed (split));
    str = g_strdup_printf ("%u", split->n);
    
    kp_html_table_row (fp, 
                       str,
                       NOT_NULL (split->sport),
                       NOT_NULL (split->type),
                       NOT_NULL (distance),
                       NOT_NULL (duration),
                       NOT_NULL (pace),
                       NOT_NULL (speed),
                       NOT_NULL (split->comment),
                       NULL);
    g_free (str);
  }

  kp_html_table_foot (fp);
}

  
static void
kp_workout_table (KPHtmlWriter *writer, KPWorkout *wo, FILE *fp)
{
  gchar buf[128];
  KPDate date;
  gchar *str;
  gdouble dval;
  guint ival;
  
  kp_calendar_entry_get_date (KP_CALENDAR_ENTRY (wo), &date);
  kp_html_table_head (fp, _("Key"), _("Value"), NULL);

  /* Date */
  kp_date_to_locale_string (&date, buf, sizeof (buf)-1);
  kp_html_table_row (fp, _("<b>Date:</b>"), buf, NULL);

  /* Sport */
  kp_html_table_row (fp, _("<b>Sport:</b>"), kp_workout_get_sport (wo), NULL);

  /* Duration */
  str = kp_param_list_get_as_string (wo->param_list, "duration");
  if (str) {
    kp_html_table_row_start (fp);
    kp_html_table_cell (fp, "<b>%s</b>", _("Duration:"));
    kp_html_table_cell (fp, str, NULL);
    kp_html_table_row_end (fp);
    g_free (str);
  }

  /* Distance */
  dval = kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo));
  if (dval) {
    kp_html_table_row_start (fp);
    kp_html_table_cell (fp, "<b>%s</b>", _("Distance:"));
    kp_html_table_cell (fp, "%.1f km", dval);
    kp_html_table_row_end (fp);
  }

  /* Velocity */
  dval = kp_workout_model_get_speed (KP_WORKOUT_MODEL (wo));
  if (dval) {
    kp_html_table_row_start (fp);
    kp_html_table_cell (fp, "<b>%s</b>", _("Speed:"));
    kp_html_table_cell (fp, "%.1f km/h", dval);
    kp_html_table_row_end (fp);
  }
  /* Pace */
  ival = kp_workout_model_get_pace (KP_WORKOUT_MODEL (wo));
  if (ival) {
    str = kp_date_mseconds_to_std_string (ival);
    kp_html_table_row_start (fp);
    kp_html_table_cell (fp, "<b>%s</b>", _("Pace:"));
    kp_html_table_cell (fp, "%s / km", str);
    kp_html_table_row_end (fp);
    g_free (str);
  }
 
  /* Comment */
  str = kp_workout_get_comment (KP_WORKOUT (wo));
  if (str) {
    kp_html_table_row_start (fp);
    kp_html_table_cell (fp, "<b>%s</b>",_("Comment:"));
    kp_html_table_cell (fp, str);
    kp_html_table_row_end (fp);
    g_free (str);
  }
  
  kp_html_table_foot (fp);

  if (KP_IS_SPLIT_WORKOUT (wo))
    splits_table (writer, KP_SPLIT_WORKOUT (wo), fp);
}


static void
kp_comment_table (KPHtmlWriter *writer, KPComment *comment, FILE *fp)
{
  gchar buf[128];
  KPDate date;
  gchar *str;
  gchar *title;
  gchar *text;
  
  kp_calendar_entry_get_date (KP_CALENDAR_ENTRY (comment), &date);
 
  kp_html_table_head (fp, _("Key"), _("Value"), NULL);
  
  /* Date */
  kp_date_to_locale_string (&date, buf, sizeof (buf)-1);
  kp_html_table_row (fp, _("<b>Date:</b>"), buf, NULL);

  text = kp_comment_get_text (comment);
  title = kp_comment_get_title (comment);  

  str = g_strdup_printf ("<pre>\n%s</pre>\n", text);
  
  kp_html_table_row (fp, _("<b>Title:</b>"), title, NULL);
  kp_html_table_row (fp, _("<b>Text:</b>"), str, NULL);
    
  g_free (title);
  g_free (text);
  g_free (str);
  
  kp_html_table_foot (fp);
}

  
static void
kp_results_table (KPHtmlWriter *writer, KPResults *res, FILE *fp)
{
  gchar buf[128];
  KPDate date;
  gchar *diff;
  gchar *str;
  KPResult *r;
  GList *l;
  guint i;
  
  kp_calendar_entry_get_date (KP_CALENDAR_ENTRY (res), &date);
  kp_html_table_head (fp, _("Key"), _("Value"), NULL);
  
  /* Date */
  kp_date_to_locale_string (&date, buf, sizeof (buf)-1);
  kp_html_table_row (fp, _("<b>Date:</b>"), buf, NULL);
  kp_html_table_row (fp, _("<b>Title:</b>"), kp_results_get_title (res), NULL);
  kp_html_table_row (fp, _("<b>Location:</b>"), kp_results_get_location (res), NULL);
  kp_html_table_foot (fp);
  
  kp_html_h1 (fp, _("List"));
  kp_html_table_head (fp, _("Rank"), _("Name"), _("Organization"), _("Result"),
                          _("Difference"), NULL);

  for (i=1, l = res->list; l; l = l->next, i++) {
    r = KP_RESULT (l->data);
  
    g_snprintf (buf, sizeof (buf)-1, "%u.", i);
    str = kp_date_mseconds_to_std_string (r->result);
    /* Count the difference */
    diff = kp_date_mseconds_to_std_string (r->result - KP_RESULT (res->list->data)->result);
    
    kp_html_table_row (fp, 
                       buf,
                       NOT_NULL (r->competitor->name),
                       NOT_NULL (r->competitor->organization),
                       NOT_NULL (str),
                       NOT_NULL (diff),
                       NULL);
    g_free (diff);
    g_free (str);
  }



  
  kp_html_table_foot (fp);
}

void
page_entry (KPHtmlWriter *writer, KPCalendarEntry *entry, GError **err)
{
  GError *tmp_error = NULL;
  KPDate date;
  FILE *fp;
  gchar buf[32];
  gchar *str;

  g_return_if_fail (KP_IS_HTML_WRITER (writer));
  g_return_if_fail (KP_IS_CALENDAR_ENTRY (entry));
 
  g_snprintf (buf, sizeof (buf)-1, "entry-%u.html", kp_calendar_entry_get_id (entry));
  fp = kp_html_writer_open (writer, buf, &tmp_error);
 
  if (!fp) {
    g_propagate_error (err, tmp_error);
    return;
  }
  
  kp_calendar_entry_get_date (KP_CALENDAR_ENTRY (entry), &date);
  kp_html_navi (fp, &navi_data, date.y, date.m, kp_calendar_entry_get_id (entry));
 
  /* Heading 1 */
  str = g_strdup_printf ("%s %u.%u.%u", 
                         kp_calendar_entry_get_human_name (KP_CALENDAR_ENTRY (entry)),
                         date.d, date.m, date.y);
  kp_html_h1 (fp, str);
  g_free (str);

  /* Data table */
  if (KP_IS_WORKOUT (entry))
    kp_workout_table (writer, KP_WORKOUT (entry), fp);
  else if (KP_IS_COMMENT (entry))
    kp_comment_table (writer, KP_COMMENT (entry), fp);
  else if (KP_IS_RESULTS (entry))
    kp_results_table (writer, KP_RESULTS (entry), fp);
  else 
    g_warning ("HTML Output function for type '%s' not implemented!\n", 
                G_OBJECT_TYPE_NAME (entry));

  kp_html_writer_close (fp);
}


