#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "kpsplitsview.h"

#include "../kpsplitworkout.h"
#include "../kipina-i18n.h"
#include "../kputil.h"

enum {
  SPLIT_SELECTED_SIGNAL,
  LAST_SIGNAL
};

static guint      kp_splits_view_signals[LAST_SIGNAL] = { 0 }; 
static void       kp_splits_view_class_init       (KPSplitsViewClass *klass);
static void       kp_splits_view_init             (KPSplitsView *lv);
static void       kp_splits_view_finalize         (GObject *object);
static void       selection_changed_cb            (GtkTreeSelection *selection,
                                                   KPSplitsView *sv);


typedef struct KPSplitsViewPrivateData_
{
  GSList           *splits;
  GtkListStore     *store;
} KPSplitsViewPrivateData;

#define KP_SPLITS_VIEW_PRIVATE_DATA(widget) (((KPSplitsViewPrivateData*) \
      (KP_SPLITS_VIEW (widget)->private_data)))

enum {
  COLUMN_SPLIT_N,
  COLUMN_SPORT,
  COLUMN_TYPE,
  COLUMN_DISTANCE,
  COLUMN_DURATION,
  COLUMN_PACE,
  COLUMN_SPEED,
  COLUMN_COMMENT,
  COLUMN_N
};

GType
kp_splits_view_get_type (void)
{
  static GType        kp_splits_view_type = 0;

  if (!kp_splits_view_type) {
    static const GTypeInfo kp_splits_view_info = {
      sizeof (KPSplitsViewClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_splits_view_class_init,
      NULL,
      NULL,
      sizeof (KPSplitsView),
      0,
      (GInstanceInitFunc) kp_splits_view_init,
      NULL
    };
    kp_splits_view_type = g_type_register_static (GTK_TYPE_TREE_VIEW,
                                                  "KPSplitsView",
                                                  &kp_splits_view_info, 0);
  }
  return kp_splits_view_type;
}

static void
kp_splits_view_class_init (KPSplitsViewClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_splits_view_finalize;

  kp_splits_view_signals[SPLIT_SELECTED_SIGNAL]
    = g_signal_new ("split-selected",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPSplitsViewClass, split_selected),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__POINTER,
                    G_TYPE_NONE,
                    1,
                    G_TYPE_POINTER);
}

static GtkTreeViewColumn *
new_column (KPSplitsView *lv, const gchar *title, guint column,
            GtkTreeCellDataFunc func)
{
  GtkTreeViewColumn *col;
  GtkCellRenderer *ren;

  ren = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_column_new_with_attributes (title, ren, "markup", column,
                                                  NULL);
  gtk_tree_view_column_set_sort_column_id (col, column);
  gtk_tree_view_append_column (GTK_TREE_VIEW (lv), col);

  gtk_tree_view_column_set_resizable (col, TRUE);

  return col;
}

static void
kp_splits_view_init (KPSplitsView *lv)
{
  KPSplitsViewPrivateData *p_data;
  GtkTreeSelection *selection;
  
  lv->private_data = g_new (KPSplitsViewPrivateData, 1);
  p_data = KP_SPLITS_VIEW_PRIVATE_DATA (lv);
  p_data->splits = NULL;

  (void) new_column (lv, _("Num"), COLUMN_SPLIT_N, NULL);
  (void) new_column (lv, _("Sport"), COLUMN_SPORT, NULL);
  (void) new_column (lv, _("Type"), COLUMN_TYPE, NULL);
  (void) new_column (lv, _("Distance"), COLUMN_DISTANCE, NULL);
  (void) new_column (lv, _("Duration"), COLUMN_DURATION, NULL);
  (void) new_column (lv, _("Pace"), COLUMN_PACE, NULL);
  (void) new_column (lv, _("Speed"), COLUMN_SPEED, NULL);
  (void) new_column (lv, _("Comment"), COLUMN_COMMENT, NULL);
  
  p_data->store = gtk_list_store_new (COLUMN_N,
                                      G_TYPE_UINT,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING);
  
  gtk_tree_view_set_model (GTK_TREE_VIEW (lv), GTK_TREE_MODEL (p_data->store));
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (lv), TRUE);
  gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (lv), TRUE);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (lv));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (selection_changed_cb), lv);
}


static void
kp_splits_view_finalize (GObject *object)
{
  KPSplitsViewPrivateData *p_data;

  p_data = KP_SPLITS_VIEW_PRIVATE_DATA (object);

  KP_SPLITS_VIEW (object)->private_data = NULL;
  g_free (KP_SPLITS_VIEW (object)->private_data);  
}

GtkWidget *
kp_splits_view_new (GSList *splits)
{
  KPSplitsViewPrivateData *p_data;
  GtkWidget *widget;

  widget = g_object_new (kp_splits_view_get_type (), NULL);

  p_data = KP_SPLITS_VIEW_PRIVATE_DATA (widget);
  p_data->splits = splits;
  
  kp_splits_view_set_splits (KP_SPLITS_VIEW (widget), splits);
  
  return widget;
}


GSList *
kp_splits_view_get_splits (KPSplitsView *view)
{
  KPSplitsViewPrivateData *p_data;
  p_data = KP_SPLITS_VIEW_PRIVATE_DATA (view);
  return p_data->splits;
}


void
kp_splits_view_set_splits (KPSplitsView *view, GSList *splits)
{
  KPSplitsViewPrivateData *p_data;
  KPSplit *split;
  gchar *distance, *duration, *pace;
  gchar *speed;
  GtkTreeIter iter;  
  GSList *node;
  guint n = 1;
 
  g_return_if_fail (KP_IS_SPLITS_VIEW (view));
  
  p_data = KP_SPLITS_VIEW_PRIVATE_DATA (view);
  
  /* NULL is an empty list */
  if (splits == NULL) {
    gtk_list_store_clear (p_data->store);    
    return;
  }
  
  p_data->splits = splits;
  gtk_list_store_clear (p_data->store);
  
  for (node = splits; node; node = node->next)
  {
    split = KP_SPLIT (node->data);

    g_return_if_fail (split != NULL);   
      
    if (split->duration > 0)
      duration = kp_date_mseconds_to_std_string (split->duration);
    else
      duration = NULL;
   
    if (split->distance > 0.0)
      distance = g_strdup_printf ("%.3f km", split->distance);
    else
      distance = NULL;
   
    pace = kp_date_mseconds_to_std_string (kp_split_get_pace (split));
    speed = g_strdup_printf ("%.1f km/h", kp_split_get_speed (split));
    
    gtk_list_store_append (p_data->store, &iter);
    gtk_list_store_set (p_data->store, &iter, 
                        COLUMN_SPLIT_N, split->n,
                        COLUMN_DISTANCE, NOT_NULL (distance), 
                        COLUMN_DURATION, NOT_NULL (duration), 
                        COLUMN_PACE, NOT_NULL (pace),
                        COLUMN_SPEED, NOT_NULL (speed),
                        COLUMN_COMMENT, NOT_NULL (split->comment), 
                        COLUMN_SPORT, NOT_NULL (split->sport),
                        COLUMN_TYPE, NOT_NULL (split->type),
                        -1);
    g_free (speed);
    
    if (distance)
      g_free (distance);
    if (duration)
      g_free (duration);
    if (pace)
      g_free (pace);
    
    n++;
  }
}


static void
selection_changed_cb (GtkTreeSelection *selection, KPSplitsView *sv)
{
  KPSplitsViewPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GSList *node;
  guint n;

  p_data = KP_SPLITS_VIEW_PRIVATE_DATA (sv);
  
  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    return;

  gtk_tree_model_get (GTK_TREE_MODEL (model), &iter, COLUMN_SPLIT_N, &n, -1);
  
  for (node = p_data->splits; node; node = node->next) {
    if (KP_SPLIT (node->data)->n == n) {
      g_signal_emit (G_OBJECT (sv), 
                     kp_splits_view_signals[SPLIT_SELECTED_SIGNAL], 0,
                     KP_SPLIT (node->data));
      return;
    }
  }
  g_warning ("Can't find split N = %u !\n", n);
}

/**
 * kp_splits_view_get_selected_split_n:
 * @view: A #KPSplitsView
 *
 * Get the number of the currently selected split.
 * 
 * Returns: the split number (1 -> ) and 0 if there is no selected split.
 */
guint
kp_splits_view_get_selected_split_n (KPSplitsView *view)
{
  KPSplitsViewPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint n;

  p_data = KP_SPLITS_VIEW_PRIVATE_DATA (view);
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  
  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    return 0;

  gtk_tree_model_get (GTK_TREE_MODEL (model), &iter, COLUMN_SPLIT_N, &n, -1);

  return n;
}
  
