#include <gtk/gtk.h>
#include <glade/glade.h>

#include "kptreeview.h"
#include "kpmainwindow.h"
#include "kpresultsmanager.h"
#include "kpresultsview.h"
#include "kpaddcompetitordialog.h"
#include "kplogstore.h"
#include "kpguiutils.h"
#include "kpresultseditor.h"

#include "../kptraininglog.h"
#include "../kipina-i18n.h"

static void     kp_results_manager_class_init     (KPResultsManagerClass *klass);
static void     kp_results_manager_init           (KPResultsManager *dialog);
static void     kp_results_manager_finalize       (GObject *object);

/* Callbacks */
static void     results_tree_selection_changed    (GtkTreeSelection *selection,
                                                   KPResultsManager *manager);

static void     log_set                           (KPViewModel *model,
                                                   KPTrainingLog *log,
                                                   KPResultsManager *manager);
static void     log_unset                         (KPViewModel *model,
                                                   KPResultsManager *manager);
static void     new_button_clicked                (GtkButton *button,
                                                   KPResultsManager *manager);
static void     delete_button_clicked             (GtkButton *button,
                                                   KPResultsManager *manager);
static void     update_states                     (KPResultsManager *manager);



typedef struct KPResultsManagerPrivateData_
{
  KPTrainingLog       *log;
  KPResultsView       *view;
  KPTreeView          *tree;
  KPResults           *selected_results;
  
  GtkWidget           *button_new;
  GtkWidget           *button_delete;
  GtkWidget           *button_information;
  GtkWidget           *button_export;
  GtkWidget           *sw_tree;
  GtkWidget           *sw_view;

  GtkWidget           *closebutton;

} KPResultsManagerPrivateData;

#define KP_RESULTS_MANAGER_PRIVATE_DATA(widget) \
  (((KPResultsManagerPrivateData*) \
  (KP_RESULTS_MANAGER (widget)->private_data)))

static GObjectClass *parent_class = NULL;

GType
kp_results_manager_get_type (void)
{
  static GType kp_results_manager_type = 0;

  if (kp_results_manager_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPResultsManagerClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_results_manager_class_init,
      NULL,
      NULL,
      sizeof (KPResultsManager),
      0,
      (GInstanceInitFunc) kp_results_manager_init,
      NULL,
    };

    kp_results_manager_type = g_type_register_static (GTK_TYPE_WINDOW,
                                                     "KPResultsManager",
                                                     &our_info, 0);
  }
  return kp_results_manager_type;
}


static void
kp_results_manager_class_init (KPResultsManagerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_results_manager_finalize;
}


static void
kp_results_manager_init (KPResultsManager *dialog)
{
  KPResultsManagerPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkWidget *paned;
  KPLogStore *store;
  GladeXML *xml;
  KPView *view;
  
  xml = kp_gui_load ("results", "resultsbox");
  
  gtk_container_add (GTK_CONTAINER (dialog), KP_W (xml, "resultsbox"));
  
  dialog->private_data = g_new0 (KPResultsManagerPrivateData, 1);
  p_data = KP_RESULTS_MANAGER_PRIVATE_DATA (dialog);

  p_data->closebutton = KP_W (xml, "button_close");
  
  g_signal_connect_swapped (G_OBJECT (p_data->closebutton), "clicked",
                            G_CALLBACK (gtk_widget_destroy), dialog);
  
  p_data->log = kp_main_window_get_log ();
  g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
 
  /* Put widgets to private data */
  p_data->button_new = KP_W (xml, "new");
  p_data->button_delete = KP_W (xml, "delete");
  p_data->button_information = KP_W (xml, "information");
  p_data->button_export = KP_W (xml, "export");
  p_data->sw_tree = KP_W (xml, "sw_tree");
  p_data->sw_view = KP_W (xml, "sw_view");
 
  p_data->tree = KP_TREE_VIEW (kp_tree_view_new (NULL));
  p_data->view = KP_RESULTS_VIEW (kp_results_view_new (NULL));
  p_data->selected_results = NULL;
  
  paned = KP_W (xml, "hpaned");

  gtk_widget_set_size_request (GTK_WIDGET (paned), 600, -1);
  
  gtk_paned_set_position (GTK_PANED (paned), 
                          200);

  gtk_paned_pack1 (GTK_PANED (paned), GTK_WIDGET (p_data->tree), TRUE, FALSE);
  gtk_paned_add2 (GTK_PANED (paned), GTK_WIDGET (p_data->view));
 
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->tree));
  gtk_tree_selection_set_mode (GTK_TREE_SELECTION (selection),
                               GTK_SELECTION_SINGLE);
  
  store = kp_log_store_new ();
  gtk_tree_view_set_model (GTK_TREE_VIEW (p_data->tree), GTK_TREE_MODEL (store));
  
  kp_log_store_attach_log (store, p_data->log, kp_results_get_type ());
  kp_view_model_set_log (KP_VIEW_MODEL (p_data->tree), p_data->log);
  gtk_widget_set_size_request (GTK_WIDGET (p_data->tree), 200, 300);
  gtk_widget_set_size_request (GTK_WIDGET (p_data->view), 300, 300);
  
  gtk_widget_show (GTK_WIDGET (p_data->tree));

  gtk_container_add (GTK_CONTAINER (p_data->sw_tree), 
                     GTK_WIDGET (p_data->tree));
  gtk_container_add (GTK_CONTAINER (p_data->sw_view), 
                     GTK_WIDGET (p_data->view));

  gtk_widget_set_size_request (KP_W (xml, "frame_tree"), 200, -1);
  gtk_widget_set_size_request (KP_W (xml, "frame_view"), 300, -1);
  
  view = kp_main_window_get_view ();

  g_signal_connect (G_OBJECT (p_data->button_new), "clicked",
                    G_CALLBACK (new_button_clicked), dialog);
  g_signal_connect (G_OBJECT (p_data->button_delete), "clicked",
                    G_CALLBACK (delete_button_clicked), dialog);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (results_tree_selection_changed), dialog);
  g_signal_connect (G_OBJECT (view), "log-set",
                    G_CALLBACK (log_set), dialog);
  g_signal_connect (G_OBJECT (view), "log-unset",
                    G_CALLBACK (log_unset), dialog);

  update_states (dialog);
  
  g_object_unref (G_OBJECT (xml));
}


static void
kp_results_manager_finalize (GObject *object)
{
  KPResultsManagerPrivateData *p_data;
  KPResultsManager *dialog;
  KPView *view;

  p_data = KP_RESULTS_MANAGER_PRIVATE_DATA (object);
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_RESULTS_MANAGER (object));

  dialog = KP_RESULTS_MANAGER (object);

  view = kp_main_window_get_view ();
  
  g_signal_handlers_disconnect_by_func (view, log_unset, dialog);
  g_signal_handlers_disconnect_by_func (view, log_set, dialog);

  g_return_if_fail (dialog->private_data != NULL);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


/**
 * kp_results_manager_new:
 *
 * Create an instance of #KPResultsManager.
 *
 * Returns: A #KPResultsManager
 */
GtkWidget *
kp_results_manager_new (void)
{
  return g_object_new (KP_TYPE_RESULTS_MANAGER, NULL);
}


static void
update_states (KPResultsManager *manager)
{
  KPResultsManagerPrivateData *p_data;
  gboolean results_active;
  
  p_data = KP_RESULTS_MANAGER_PRIVATE_DATA (manager);
  
  results_active = (p_data->selected_results != NULL);
  
  gtk_widget_set_sensitive (p_data->button_delete, results_active);
}


static void
log_set (KPViewModel *model, KPTrainingLog *log, KPResultsManager *manager)
{
  KPResultsManagerPrivateData *p_data;
  p_data = KP_RESULTS_MANAGER_PRIVATE_DATA (manager);
  kp_view_model_set_log (KP_VIEW_MODEL (p_data->tree), log);
}


static void
log_unset (KPViewModel *model, KPResultsManager *manager)
{
  KPResultsManagerPrivateData *p_data;
  p_data = KP_RESULTS_MANAGER_PRIVATE_DATA (manager);

  g_print ("Is the problem really here?\n");
    
  if (KP_IS_RESULTS_MANAGER (manager))
    kp_view_model_unset_log (KP_VIEW_MODEL (p_data->tree));
}


static void
results_tree_selection_changed (GtkTreeSelection *selection, 
                                KPResultsManager *manager)
{
  KPResultsManagerPrivateData *p_data;
  KPLogStoreRecordType type;
  KPCalendarEntry *entry;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *mark;
  guint d, m, y;
  
  p_data = KP_RESULTS_MANAGER_PRIVATE_DATA (manager);

  if (!gtk_tree_selection_get_selected (selection, &model, &iter)) {
    p_data->selected_results = NULL;
    update_states (manager);
    return;
  }
  type = kp_log_store_get_iter_type (KP_LOG_STORE (model), &iter);

  switch (type) {
    case KP_LOG_STORE_REC_INVALID:
    case KP_LOG_STORE_REC_ROOT:
    case KP_LOG_STORE_REC_YEAR:
    case KP_LOG_STORE_REC_MONTH:
    case KP_LOG_STORE_REC_DAY:
    default:
      p_data->selected_results = NULL;
      update_states (manager);
      return;

    case KP_LOG_STORE_REC_ENTRY:
      gtk_tree_model_get (model, &iter, KP_LOG_STORE_COL_RECORD, &mark, -1);
      kp_log_store_get_date (KP_LOG_STORE (model), &iter, &d, &m, &y);
     
      entry = kp_training_log_get_entry (p_data->log, d, m, y, mark);
      g_free (mark);
     
      if (G_OBJECT_TYPE (entry) == kp_results_get_type ()) {
        kp_results_view_set_results (p_data->view, KP_RESULTS (entry));
        p_data->selected_results = KP_RESULTS (entry);
        update_states (manager);
      }
  }
}


static void
new_button_clicked (GtkButton *button, KPResultsManager *manager)
{
  GtkWidget *editor;
  KPDate date;

  date.d = 12;
  date.m = 12;
  date.y = 2002;

  editor = kp_results_editor_new (&date, NULL);
  gtk_window_set_transient_for (GTK_WINDOW (editor), NULL);
  gtk_widget_show (editor);
}


static void
delete_button_clicked (GtkButton *button, KPResultsManager *manager)
{
  KPResultsManagerPrivateData *p_data;
  p_data = KP_RESULTS_MANAGER_PRIVATE_DATA (manager);

  if (p_data->selected_results) {
    kp_training_log_remove (p_data->log, 
                            KP_CALENDAR_ENTRY (p_data->selected_results));
    p_data->selected_results = NULL;
  }
}


