#include <string.h>
#include <stdlib.h>
#include <time.h>

#include <gtk/gtk.h>
#include <gconf/gconf-client.h>

#include "kppreferencesdialog.h"
#include "kphtmloutputdialog.h"
#include "kpviewpopupmodel.h"
#include "kpresultsmanager.h"
#include "kploginfodialog.h"
#include "kpmainwindow.h"
#include "kpguiutils.h"
#include "kpentries.h"

#include "kipina-version.h"
#include "../kptraininglog.h"
#include "../kipina-i18n.h"

#if HAS_LIBGNOME
#include <libgnome/libgnome.h>
#endif


#define GET_VIEW(void)  KP_VIEW_MODEL(kp_main_window_get_view())

#define KEY_WINDOW_GEOMETRY   "/apps/kipina/preferences/save_window_geometry"
#define KEY_WINDOW_HEIGHT     "/apps/kipina/preferences/window_height"
#define KEY_WINDOW_WIDTH      "/apps/kipina/preferences/window_width"



GtkActionGroup *action_group = NULL;
GtkWindow *parent_window = NULL;


KPCalendarEntry *
get_currently_active_entry (void)
{
  KPViewModel *model = kp_view_get_current_viewer (KP_VIEW (GET_VIEW ()));
  g_return_val_if_fail (KP_IS_VIEW_MODEL (model), NULL);

  return kp_view_popup_model_get_active_entry (KP_VIEW_POPUP_MODEL (model));
}


static void
save_window_geometry (void)
{
  GConfClient *client;
  gint width, height;

  client = gconf_client_get_default ();
  g_return_if_fail (client != NULL);
  
  if (gconf_client_get_bool (client, KEY_WINDOW_GEOMETRY, NULL) == FALSE)
    return;
  
  gtk_window_get_size (parent_window, &width, &height);

  gconf_client_set_int (client, KEY_WINDOW_WIDTH, width, NULL);
  gconf_client_set_int (client, KEY_WINDOW_HEIGHT, height, NULL);
}


gchar *
get_filename_from_user (gboolean allow_overwrite)
{
  gchar *file;
  
  file = kp_gui_get_file_to_save (GTK_WINDOW (parent_window));
  if (!file)
    return NULL;
  
  /* File with that name doesn't exist */
  if (g_file_test (file, G_FILE_TEST_EXISTS) == FALSE)
    return file;

  if (allow_overwrite == FALSE)
    if (kp_gui_ask_if_user_wants_to_replace (GTK_WINDOW (parent_window), file))
      return file;
  
  g_free (file);
  return NULL;
}

enum {
  SAVE_RESPONSE_CANCEL,
  SAVE_RESPONSE_ERROR,
  SAVE_RESPONSE_OK
};


guint
save_log (GError **err)
{
  KPTrainingLog *log;
  GError *error = NULL;
  const gchar *file;
  gboolean ret;

  log = kp_main_window_get_log ();
  g_return_val_if_fail (log != NULL, SAVE_RESPONSE_ERROR);
 
  if ((file = kp_training_log_get_filename (log))) {
    kp_debug ("Log has file name: %s", file);
    goto save;
  }

  /* If the log is already written with that name, it can be saved over itself. */
  if ((file = get_filename_from_user (kp_training_log_is_written_to_disk (log))))
    goto save;
    
  return SAVE_RESPONSE_CANCEL;


save:
  ret = kp_training_log_save (log, file, &error);
  
  if (ret == TRUE) {
    kp_statusbar_set_format_message (kp_main_window_get_statusbar (), 
                                    "Log saved: \"%s\".", file);
    return SAVE_RESPONSE_OK;
  }
  else {
    g_propagate_error (err, error);
    return SAVE_RESPONSE_ERROR;
  }
}



/* File -> New */
static void
new_action (void)
{
  KPTrainingLog *log; 
  GError *err = NULL;
  
  if ((log = kp_main_window_get_log ()) == NULL)
    goto new;
  
  if (kp_training_log_is_modified (log) == FALSE)
    goto new;
    
  if (!kp_gui_ask_if_user_wants_to_save_old_log (GTK_WINDOW (parent_window)))
    goto new;
  
  switch (save_log (&err)) {
    case SAVE_RESPONSE_OK:
      goto new;
    case SAVE_RESPONSE_CANCEL:
      return;
    case SAVE_RESPONSE_ERROR:
      /* Prevent a compiler warning */
      break;
  }

  /* If saving the old fails, don't create a new log! */
  kp_gui_err_message (GTK_WINDOW (parent_window),
                     "Can't save the old log: %s; so not creating a new one..",
                     (err) ? err->message : _("Unknown reason"));
  if (err)
    g_error_free (err);
  return;

new:
  kp_main_window_init_log (kp_main_window_get_window (), NULL);
}


/* File -> Open */
static void
open_action (void)
{
  KPTrainingLog *log;
  GError *error;
  gchar *file;
  guint ret;

  log = kp_main_window_get_log ();
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  
  if ((file = kp_gui_get_file_to_open (GTK_WINDOW (parent_window))) == NULL)
    return;
  
  if (kp_training_log_is_modified (log) == FALSE)
    goto open_log;
 
  switch (kp_gui_ask_if_user_wants_to_save_old_log (GTK_WINDOW (parent_window)))
  {
    case GTK_RESPONSE_CANCEL:
      return;

    case GTK_RESPONSE_YES:
      error = NULL;
      ret = save_log (&error);
      
      if (ret == SAVE_RESPONSE_OK)
        goto open_log;
      else if (ret == SAVE_RESPONSE_CANCEL)
        return;
      else {
        kp_gui_err_message (GTK_WINDOW (parent_window), 
                           "Can't save the old log: %s; so not creating a new one..",
                           (error) ? error->message : _("Unknown reason"));
        return;
      }

    case GTK_RESPONSE_NO:
      goto open_log;

    default:
      g_return_if_reached ();
  }

open_log:
  kp_main_window_init_log (kp_main_window_get_window (), file);
}


/* File -> Save */
static void
save_action (void)
{
  GError *err = NULL;

  if (save_log (&err) == SAVE_RESPONSE_ERROR)
    kp_gui_err_message (GTK_WINDOW (parent_window),
                       "Can't save the log: %s", 
                       (err) ? err->message : _("Unknown reason"));
}


/* File -> Save As */
static void
save_as_action (void)
{
  KPTrainingLog *log;
  GError *error = NULL;
  const gchar *file;

  log = kp_main_window_get_log ();
  
  /* Don't allow overwrite without asking */
  if ((file = get_filename_from_user (FALSE)))
    kp_training_log_save (log, file, &error);
}


/* File -> Quit */
static void     
quit_action ()
{
  GError *err = NULL;
  const gchar *file;
  KPTrainingLog *log = kp_main_window_get_log ();
 
  if (kp_training_log_is_modified (log) == FALSE)
    goto out;

  switch (kp_gui_ask_if_user_wants_to_save (GTK_WINDOW (parent_window)))
  {
    case GTK_RESPONSE_CANCEL:
      return;

    case GTK_RESPONSE_CLOSE:
      goto out;

    /* User wants to save */
    case GTK_RESPONSE_OK:
      if ((file = kp_training_log_get_filename (log)) == NULL)
        file = get_filename_from_user (FALSE);
      if (file == NULL)
        return; /* Log is modified but no filename, abort! */

      switch (save_log (&err)) 
      {
        case SAVE_RESPONSE_ERROR:
          kp_gui_err_message (GTK_WINDOW (parent_window),
                             "Can't save the log: %s", 
                             (err) ? err->message : _("Unknown reason"));
          if (err)
            g_error_free (err);
          return;

        case SAVE_RESPONSE_CANCEL:
          return;

        case SAVE_RESPONSE_OK:
          goto out;
      }
      
    default:
      g_return_if_reached ();
  }; 

out:
  save_window_geometry ();
  gtk_main_quit ();
}


/* View -> Statusbar */
static void 
statusbar_action (GtkToggleAction *action)
{
  kp_main_window_set_statusbar_visible (gtk_toggle_action_get_active (action));
}


/* View -> Toolbar */
static void
toolbar_action (GtkToggleAction *action)
{
  kp_main_window_set_toolbar_visible (gtk_toggle_action_get_active (action));
}


/* View -> Sidebar */
static void
sidebar_action (GtkToggleAction *action)
{
  kp_main_window_set_sidebar_visible (gtk_toggle_action_get_active (action));
}


/* View -> Log Meta Info */
static void
metainfo_action (void)
{
  GtkWidget *dialog = kp_log_info_dialog_new ();
  g_signal_connect_swapped (G_OBJECT (dialog), "response",
                            G_CALLBACK (gtk_widget_destroy), dialog);
  gtk_widget_show (dialog);
}


/* Entry -> Edit */
static void
edit_action (void)
{
  KPCalendarEntry *entry = get_currently_active_entry ();
  g_return_if_fail (KP_IS_CALENDAR_ENTRY (entry));

  kp_entries_show_edit_dialog (entry);
}


/* Entry -> Remove */
static void
remove_action (void)
{
  KPCalendarEntry *entry = get_currently_active_entry ();
  g_return_if_fail (KP_IS_CALENDAR_ENTRY (entry));

  kp_training_log_remove (kp_main_window_get_log (), entry);
}


/* Entry -> Properties */
static void
properties_action (void)
{
  KPCalendarEntry *entry = get_currently_active_entry ();
  g_return_if_fail (KP_IS_CALENDAR_ENTRY (entry));

  kp_entries_show_prop_dialog (entry);
}


/* Insert -> Add Workout 
 * Insert -> Add Split Workout
 * Insert -> Add Comment 
 * Insert -> Add Results */
struct {
  const gchar *human_name;
  const gchar *action;
}
action_name_map[] = {
  { N_("Workout"),          "WorkoutAction"       },
  { N_("Split Workout"),    "SplitWorkoutAction"  },
  { N_("Comment"),          "CommentAction"       },
  { N_("Results"),          "ResultsAction"       }
};

static void
add_entry_action (GtkAction *action)
{
  KPEntryData *data;
  KPDate date;
  guint i;

  for (data = NULL, i = 0; i < G_N_ELEMENTS (action_name_map); i++)
    if (strcmp (gtk_action_get_name (action), action_name_map[i].action) == 0) {
      data = kp_entries_get_entry (action_name_map[i].human_name);
      break;
    }
  g_return_if_fail (data != NULL);

  kp_view_model_get_date (GET_VIEW (), &date);
  gtk_widget_show (data->new_dialog (&date, NULL));
}


/* Go -> Today */
static void
navi_today_action (GtkAction *action)
{
  KPDate date;
  kp_date_set_time (&date, time (NULL));
  kp_view_model_set_date (GET_VIEW (), &date);
}


/* Go -> Next Month
 * Go -> Next Week
 * Go -> Next Day
 * Go -> Prev Month
 * Go -> Prev Week
 * Go -> Prev Day */
struct {
  const gchar   *item;
  gint           mon;
  gint           day;
} navi_items[] = {
  { "NextMonthAction",  1,  0 },
  { "NextWeekAction",   0,  7 },
  { "NextDayAction",    0,  1 },
  { "PrevMonthAction",  1,  0 },
  { "PrevWeekAction",   0,  7 },
  { "PrevDayAction",    0,  1 }
};

#define KP_DATE_FROM_GDATE(kpdate,gdate) \
  kpdate.d = g_date_get_day (gdate); \
  kpdate.m = g_date_get_month (gdate); \
  kpdate.y = g_date_get_year (gdate); \

static void
navi_action (GtkAction *action)
{
  GDate *date;
  KPDate dmy;
  guint i;
  
  kp_view_model_get_date (GET_VIEW (), &dmy);
  date = g_date_new_dmy (dmy.d, dmy.m, dmy.y);
  
  for (i=0; i < G_N_ELEMENTS (navi_items); i++)
    if (strcmp (navi_items[i].item, gtk_action_get_name (action)) == 0) {

      if (strncmp (navi_items[i].item, "Next", 4) == 0) {
        g_date_add_months (date, navi_items[i].mon);
        g_date_add_days (date, navi_items[i].day);
      } 
      else {
        g_date_subtract_months (date, navi_items[i].mon);
        g_date_subtract_days (date, navi_items[i].day);
      }
    }

  KP_DATE_FROM_GDATE (dmy, date);
  kp_view_model_set_date (GET_VIEW (), &dmy);
  g_date_free (date);
}


/* Tools -> Manage Results */
static void
manage_results_action (void)
{
  gtk_widget_show (kp_results_manager_new ()); 
}


/* Tools -> Generate Html Statistics */
static void
generate_html_action (void)
{
  gtk_widget_show (kp_html_output_dialog_new ());
}


/* Tools -> Preferences */
static void
preferences_action (void) 
{
  GtkWidget *dialog = kp_preferences_dialog_new ();
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


/* Help -> Contents */
static void
help_contents_action (void)
{
#if HAS_LIBGNOME == 1
  GError *err = NULL;
  gnome_help_display ("kipina.xml", NULL, &err);

  if (err) {
    kp_gui_err_message (parent_window, "Can't display help: %s", err->message);
    g_error_free (err);
  }
#else
  g_warning ("Kipin\303\244 must be compiled with libgnome to be "
             "able to display help!");
#endif
}


#define COPYRIGHT   "Copyright \xc2\xa9 2001-2006 Ville Kangas"

/* Help -> About */
static void
about_action (void)
{
  gchar *authors[2] = { "Ville Kangas", NULL };
  gchar *version_str;
  GtkWidget *dialog;

  dialog = gtk_about_dialog_new ();

  if (KIPINA_DEV_RELEASE) {
    version_str = g_strdup_printf (_("[ Dev snapshot, date: %s %s ]"),
                                    KIPINA_VERSION_DATE,
                                    KIPINA_VERSION_TIME);
    gtk_about_dialog_set_comments (GTK_ABOUT_DIALOG (dialog), version_str);
    g_free (version_str);
  }
   
  g_object_set (dialog,
                "name", "Kipin\303\244",
                "version", KIPINA_VERSION,
                "copyright", COPYRIGHT,
                "authors", authors,
                "logo_icon_name", "kipina",
                NULL);
  
  g_signal_connect_swapped (dialog, "destroy",
                            G_CALLBACK (gtk_widget_destroy), dialog);
  g_signal_connect_swapped (dialog, "response", 
                            G_CALLBACK (gtk_widget_destroy), dialog);
  gtk_widget_show (dialog);
}


static GtkActionEntry entries[] = 
{
  /* FILE */
  { "FileMenuAction",   NULL, N_("_File"),  NULL, NULL, NULL },
  { "NewAction", GTK_STOCK_NEW, N_("_New"), "<control>N",
   N_("New Log"), G_CALLBACK (new_action) },
 
  { "OpenAction", GTK_STOCK_OPEN, N_("_Open"), "<control>O",
   N_("Open Log"), G_CALLBACK (open_action) },
    
  { "SaveAction", GTK_STOCK_SAVE, N_("_Save"), "<control>S",
   N_("Save the log"), G_CALLBACK (save_action) },

  { "SaveAsAction", GTK_STOCK_SAVE_AS, N_("Save _As"), "<control>A",
   N_("Save the Log As..."), G_CALLBACK (save_as_action) },

  { "ImportMenuAction", NULL, N_("_Import"), NULL, NULL, NULL },
  
  { "QuitAction", GTK_STOCK_QUIT, N_("_Quit"), "<control>Q",    
   N_("Quit"),  G_CALLBACK (quit_action) },
 
  /* VIEW */
  { "ViewMenuAction",   NULL, N_("_View"),  NULL, NULL, NULL },

  { "MetaInfoAction", GTK_STOCK_DIALOG_INFO, N_("_Log Meta Information"), NULL,
   N_("View Log Meta Information"), G_CALLBACK (metainfo_action) },
  
  /* ENTRY */
  { "EntryMenuAction",  NULL, N_("_Entry"), NULL, NULL, NULL },

  { "EditAction", GTK_STOCK_EDIT, N_("Edit"), NULL,
   N_("Edit Entry"), G_CALLBACK (edit_action) },

  { "RemoveAction", GTK_STOCK_REMOVE, N_("Remove"), NULL,
   N_("Remove Entry"), G_CALLBACK (remove_action) },

  { "PropertiesAction", GTK_STOCK_PROPERTIES, N_("Properties"), NULL,
   N_("Show Properties"), G_CALLBACK (properties_action) },
  
  /* INSERT */
  { "InsertMenuAction", NULL, N_("_Insert"), NULL, NULL, NULL },

  { "WorkoutAction", GTK_STOCK_ADD, N_("Workout"), NULL,
   N_("Add Workout"), G_CALLBACK (add_entry_action) },
 
  { "SplitWorkoutAction", GTK_STOCK_ADD, N_("Split Workout"), NULL,
   N_("Add Split Workout"), G_CALLBACK (add_entry_action) },
  
  { "CommentAction", GTK_STOCK_ADD, N_("Comment"), NULL,
   N_("Add Comment"), G_CALLBACK (add_entry_action) },
  
  { "ResultsAction", GTK_STOCK_ADD, N_("Results"), NULL,
   N_("Add Results"), G_CALLBACK (add_entry_action) },
  
  /* GO */
  { "GoMenuAction", NULL, N_("_Go"), NULL, NULL, NULL },

  { "TodayAction", GTK_STOCK_JUMP_TO, N_("Today"), NULL,
   N_("Go Today"), G_CALLBACK (navi_today_action) },
  
  { "NextMonthAction", GTK_STOCK_GO_FORWARD, N_("Next Month"), NULL,
   N_("Next Month"), G_CALLBACK (navi_action) },
  
  { "NextWeekAction", GTK_STOCK_GO_FORWARD, N_("Next Week"), NULL,
   N_("Next Week"), G_CALLBACK (navi_action) },
  
  { "NextDayAction", GTK_STOCK_GO_FORWARD, N_("Next Day"), NULL,
   N_("Tomorrow"), G_CALLBACK (navi_action) },
  
  { "PrevMonthAction", GTK_STOCK_GO_BACK, N_("Prev Month"), NULL,
   N_("Previous Month"), G_CALLBACK (navi_action) },
  
  { "PrevWeekAction", GTK_STOCK_GO_BACK, N_("Prev Week"), NULL,
   N_("Previous Month"), G_CALLBACK (navi_action) },

  { "PrevDayAction", GTK_STOCK_GO_BACK, N_("Prev Day"), NULL,
   N_("Previous Day"), G_CALLBACK (navi_action) },
  
  /* TOOLS */
  { "ToolsMenuAction",  NULL, N_("_Tools"), NULL, NULL, NULL },

  { "ManageResultsAction", NULL, N_("_Manage Results..."), NULL,
   N_("Manage Results"), G_CALLBACK (manage_results_action) },

  { "GenHtmlAction", NULL, N_("_Generate HTML Statistics..."), NULL,
   N_("Generate Html"), G_CALLBACK (generate_html_action) },

  { "PreferencesAction", GTK_STOCK_PREFERENCES, N_("_Preferences"), NULL,
   N_("Show Preferences"), G_CALLBACK (preferences_action) },
  
  /* HELP */
  { "HelpMenuAction",   NULL, "_Help",  NULL, NULL, NULL },

  { "HelpContentsAction", GTK_STOCK_HELP, N_("_Contents..."), NULL,
   N_("Show Help Contents"), G_CALLBACK (help_contents_action) },

  { "AboutAction", GTK_STOCK_ABOUT, N_("_About"), NULL, 
   N_("Show About Dialog"), G_CALLBACK (about_action) }
};


static GtkToggleActionEntry toggle_entries[] =
{
  { "StatusbarAction",  NULL, N_("_Statusbar"), NULL, 
   N_("View Statusbar"), G_CALLBACK (statusbar_action), FALSE },
  { "ToolbarAction",  NULL, N_("_Toolbar"), NULL, 
   N_("View Toolbar"), G_CALLBACK (toolbar_action), FALSE },
  { "SidebarAction",  NULL, N_("_Sidebar"), NULL, 
   N_("View Sidebar"), G_CALLBACK (sidebar_action), FALSE }
};


static void
menu_add_widget (GtkUIManager *ui, GtkWidget *widget, GtkContainer *container)
{
  gtk_box_pack_start (GTK_BOX (container), widget, FALSE, FALSE, 0);
  gtk_widget_show (widget);
}


void
kp_menu_bar_set_toggle_states (void)
{
  GtkAction *a_statusbar;
  GtkAction *a_sidebar;
  GtkAction *a_toolbar;
  GConfClient *client;

  a_statusbar = gtk_action_group_get_action (action_group, "StatusbarAction");
  a_sidebar = gtk_action_group_get_action (action_group, "SidebarAction");
  a_toolbar = gtk_action_group_get_action (action_group, "ToolbarAction");

  g_return_if_fail (GTK_IS_TOGGLE_ACTION (a_statusbar));
  g_return_if_fail (GTK_IS_TOGGLE_ACTION (a_sidebar));
  g_return_if_fail (GTK_IS_TOGGLE_ACTION (a_toolbar));

  client = gconf_client_get_default ();
  
  if (gconf_client_get_bool (client, KP_CONF_SHOW_STATUSBAR, NULL))
    gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (a_statusbar), TRUE);
  if (gconf_client_get_bool (client, KP_CONF_SHOW_SIDEBAR, NULL))
    gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (a_sidebar), TRUE);
  if (gconf_client_get_bool (client, KP_CONF_SHOW_TOOLBAR, NULL))
    gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (a_toolbar), TRUE);
}



GtkWidget *
kp_menu_bar_new (GtkWidget **import, GtkWidget **toolbar)
{
  GtkWidget *menu_box;
  GtkUIManager *menu_manager;
  GError *error;
  gchar *file;

  parent_window = GTK_WINDOW (kp_main_window_get_window ());
  
  menu_box = gtk_vbox_new (FALSE, 0);
  action_group = gtk_action_group_new ("TestActions");
  menu_manager = gtk_ui_manager_new ();
    
  gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);    
  gtk_action_group_add_actions (action_group, entries, 
                                G_N_ELEMENTS (entries), NULL);
  gtk_action_group_add_toggle_actions (action_group, toggle_entries, 
                                       G_N_ELEMENTS (toggle_entries), 0);
  
  gtk_ui_manager_insert_action_group (menu_manager, action_group, 0);
  
  /* Read in the UI from our XML file */
  error = NULL;
  file = g_strdup_printf ("%s/%s", KIPINA_DATA_DIR, "menubar.xml");
  gtk_ui_manager_add_ui_from_file (menu_manager, file, &error);
  g_free (file);
    
  if (error) {
    g_message ("building menus failed: %s", error->message);
    g_error_free (error);
    exit (EXIT_FAILURE);
  }

  g_signal_connect (menu_manager, "add_widget", G_CALLBACK (menu_add_widget), 
                    menu_box);
  
  gtk_widget_show (menu_box);
  gtk_ui_manager_ensure_update (menu_manager);

  if (toolbar)
    *toolbar = gtk_ui_manager_get_widget (menu_manager, "/ui/toolbar");

  if (import) {
    *import = gtk_ui_manager_get_widget (menu_manager, "/ui/menubar/FileMenu/ImportMenu");
    gtk_widget_hide (GTK_WIDGET (*import));
  }
    
  return menu_box;
}
