/**
 * Cache control
 * @copyright Copyright 2002 Shaun Jackman
 * @author Shaun Jackman <sjackman@debian.org
 */
static const char* rcsid __attribute__((unused)) =
"$Id: cache.cpp,v 1.3 2004/04/07 17:01:18 tsirc Exp $";


// kionjb
#include "njb.h"

// sqlite
#include <sqlite.h>

// kde
#include <kdebug.h>
#include <klocale.h>
#include <kstddirs.h>


/* ------------------------------------------------------------------------ */
bool
kio_njbProtocol::trackById( Track& track, unsigned id)
{
	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table_printf( m_db,
			"SELECT * FROM tracks WHERE id == %d",
			&result, &nrow, &ncolumn, &errmsg,
			id);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return false;
	}
	if( !nrow)
		return false;
	// skip header
	result += ncolumn;
	// build track
	track = Track( result);
	return true;
}


/* ------------------------------------------------------------------------ */
bool
kio_njbProtocol::trackByFilename( Track& track, const QString& filename)
{
	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table_printf( m_db,
			"SELECT * FROM tracks WHERE filename == '%q'",
			&result, &nrow, &ncolumn, &errmsg,
			filename.latin1());
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return false;
	}
	if( !nrow)
		return false;
	// skip header
	result += ncolumn;
	// build track
	track = Track( result);
	return true;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::cacheOpen( void)
{
	QString dbpath = locateLocal( "data", "kionjb/cache.db");
	kdDebug( 7182) << __func__ << ": " << dbpath << endl;

	char* errmsg;
	m_db = sqlite_open( dbpath, 0, &errmsg);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		abort();
		return -1;
	}
	return 0;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::cacheNew( void)
{
	kdDebug( 7182) << __func__ << endl;

	const char* pCreateCommands[] = {
		"create table tracks("
			"id       integer primary key,"
			"size     integer,"
			"duration integer,"
			"tracknum integer,"
			"genre    char(30),"
			"artist   char(30),"
			"album    char(30),"
			"title    char(30),"
			"codec    char(3),"
			"filename varchar(255)"
			")",
		"create index tracks_filename on tracks( filename)",
		"create table playlists( id integer primary key, name unique)",
		"create table playlisttracks( playlist, number, track)",
		"create table etc( key, value)",
		"INSERT INTO etc VALUES( 'counter', 0)",
		NULL
	};

	const char** pCommand;
	for( pCommand = pCreateCommands; *pCommand; pCommand++) {
		char* errmsg;
		sqlite_exec( m_db, *pCommand,
				NULL, NULL, &errmsg);
		if( errmsg) {
			kdDebug( 7182) << *pCommand << endl;
			kdDebug( 7182) << errmsg << endl;
			warning( errmsg);
			free( errmsg);
			return -1;
		}
	}

	kdDebug( 7182) << __func__ << ": done" << endl;
	return 0;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::cacheClose( void)
{
	kdDebug( 7182) << __func__ << endl;

	if( m_db)
		sqlite_close( m_db);
	m_db = NULL;
	return 0;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::cacheDel( const Track& track)
{
	kdDebug( 7182) << __func__ << endl;

	char* errmsg;
	sqlite_exec_printf( m_db,
			"DELETE FROM tracks WHERE id == %d",
			NULL, NULL, &errmsg,
			track.id);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	return 0;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::cacheTrack( const Track& track)
{
	char* errmsg;
	sqlite_exec_printf( m_db,
			"INSERT INTO tracks VALUES( "
			"%d, "   // id
			"%d, "   // size
			"%d, "   // duration
			"%d, "   // tracknum
			"'%q', " // genre
			"'%q', " // artist
			"'%q', " // album
			"'%q', " // title
			"'%q', " // codec
			"'%q') ",// filename
			NULL, NULL, &errmsg,
			track.id,
			track.size,
			track.duration,
			track.tracknum,
			track.genre.latin1(),
			track.artist.latin1(),
			track.album.latin1(),
			track.title.latin1(),
			track.codec.latin1(),
			track.filename.latin1());
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	return 0;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::cachePlaylists( void)
{
	kdDebug( 7182) << "cachePlaylists\n";

	char* errmsg;
	sqlite_exec( m_db, "DELETE FROM playlists", NULL, NULL, &errmsg);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}
	sqlite_exec( m_db, "DELETE FROM playlisttracks", NULL, NULL, &errmsg);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	int playlists = 0;
	NJB_Reset_Get_Playlist( m_njb);
	while( playlist_t* pl = NJB_Get_Playlist( m_njb) ) {
		infoMessage( i18n( "Downloading playlist %1...").arg( ++playlists));
		sqlite_exec_printf( m_db,
				"INSERT INTO playlists VALUES( %d, '%q')",
				NULL, NULL, &errmsg,
				pl->plid, pl->name);
		int tracks = 0;
		playlist_reset_gettrack( pl);
		while( playlist_track_t* pltrack = playlist_gettrack( pl)) {
			Track track;
			QString filename;
			if( trackById( track, pltrack->trackid))
				filename = track.filename;
			else
				filename = pltrack->trackid;
			sqlite_exec_printf( m_db,
					"INSERT INTO playlisttracks VALUES( '%q', %d, '%q')",
					NULL, NULL, &errmsg,
					pl->name, ++tracks, filename.latin1());
		}
		playlist_destroy( pl);
	}

	kdDebug( 7182) << "cachePlaylists: cached\n";
	return 0;
}


/* ------------------------------------------------------------------------ */
/** Transfer track info from the njb to the local db */
int
kio_njbProtocol::cacheTracks( void)
{
	kdDebug( 7182) << __func__ << endl;

	char* errmsg;
	sqlite_exec( m_db, "DELETE FROM tracks", NULL, NULL, &errmsg);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	int i = 0;
	NJB_Reset_Get_Track_Tag( m_njb);
	while( songid_t* song = NJB_Get_Track_Tag( m_njb)) {
		infoMessage( i18n( "Downloading track %1...").arg( i++));

		Track track = Track( song);
		songid_destroy( song);

		if( cacheTrack( track)) {
			warning( track.filename);
			return -1;
		}
	}

	kdDebug( 7182) << __func__ << ": cached" << endl;
	return 0;
}


/* ------------------------------------------------------------------------ */
/** Transfer info from the njb to the local db */
int
kio_njbProtocol::cacheLibrary( bool force)
{
	kdDebug( 7182) << __func__ << endl;

	if( !force && m_libcount)
		return 0;

	int status = connect();
	if( status)
		return status;

	char* errmsg;
	char** result;
	int nrow, ncolumn;
	sqlite_get_table( m_db,
			"SELECT value FROM etc WHERE key == 'counter'",
			&result, &nrow, &ncolumn, &errmsg);
	if( errmsg) {
		sqlite_free_table( result);
		kdDebug( 7182) << errmsg << endl;
		free( errmsg);
		warning( "Creating new cache...");
		status = cacheNew();
		if( status)
			return status;
	} else {
		unsigned counter = 0;
		if( nrow)
			counter = QString( result[ 1]).toUInt();
		sqlite_free_table( result);
		kdDebug( 7182) << "counter = " << counter << endl;
		if( counter == m_njb->libcount) {
			m_libcount = m_njb->libcount;
			kdDebug( 7182) << "cacheLibrary: pre-cached "
				<< m_libcount << endl;
			return 0;
		}
	}

	cacheTracks();
	cachePlaylists();

	sqlite_exec_printf( m_db,
			"UPDATE etc SET value=%d WHERE key='counter'",
			NULL, NULL, &errmsg,
			m_njb->libcount);
	if( errmsg) {
		kdDebug( 7182) << __func__ << ": " << errmsg << endl;
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	m_libcount = m_njb->libcount;
	kdDebug( 7182) << __func__ << ": m_libcount = " << m_libcount << endl;
	return 0;
}
