/***************************************************************************
 *   Copyright (C) 2007 by Ken Werner                                      *
 *   ken.werner@web.de                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "nvidiathermalsrc.h"
#include <qstringlist.h>
#include <qregexp.h>
#include <klocale.h>
#include <kdebug.h>

#ifndef HAVE_NVCONTROL
    #include <kprocio.h>
#else
    #include <qpaintdevice.h> // for the Device* pointer
    
    // include the NVCtrl include stuff
    #include <X11/Xlib.h>
    #include <fixx11h.h> // needed for Qt, to include X11 header
extern "C" {
    #include <NVCtrlLib.h>
}
#endif

#ifndef HAVE_NVCONTROL
NVidiaThermalSrc::NVidiaThermalSrc(QWidget* inParent, const QString& inID, const QString& inName):
		LabelSource(inParent), mProcess(0) {
#else
NVidiaThermalSrc::NVidiaThermalSrc(QWidget* inParent, const QString& inID, const QString& inName, unsigned int attrib):
        LabelSource(inParent), mAttrib(attrib) {
#endif
	mID = inID;
    mName = inName;
	mDescription = i18n("This source is provided by the nVidia GPU card driver tools");

	mRefreshTimer = new QTimer(this, "default refresh handler" );
	// connect the timer
	connect(mRefreshTimer, SIGNAL(timeout()), this, SLOT(fetchValue()));
	connect(this, SIGNAL(enabledChanged(bool, Source*)), this, SLOT(enable(bool)));
}

NVidiaThermalSrc::~NVidiaThermalSrc(){
#ifndef HAVE_NVCONTROL
	delete mProcess;
#endif
}

std::list<Source*>NVidiaThermalSrc::createInstances(QWidget* inParent){
	std::list<Source*> list;
#ifndef HAVE_NVCONTROL 
	// see if the path contains nvidia-settings. if yes, 
	// execute it and look for the query type "GPUCoreTemp"
	// and "GPUAmbientTemp" and create two instances for each
	// of them
	// start nvidia-settings, if available and wait for it to exit.
	KProcIO proc;
    proc << "nvidia-settings"
	     << "-n" // don't load config
	     << "-q" << "GPUCoreTemp"
	     << "-q" << "GPUAmbientTemp";
	if(!proc.start(KProcess::Block))
		return list;
	
	// now see what it printed...
    QString ln;
    QString pout;
    while(proc.readln(ln) != -1)
        pout+= ln + '\n';

	if(pout.contains("Attribute 'GPUCoreTemp'"))
		list.push_back(new NVidiaThermalSrc(inParent, "GPUCoreTemp", "NVidiaCore"));
	if(pout.contains("Attribute 'GPUAmbientTemp'"))
		list.push_back(new NVidiaThermalSrc(inParent, "GPUAmbientTemp", "NVidiaAmbient"));
#else
    int evt_base = 0, err_base = 0, temp;
    Display * dpy = QPaintDevice::x11AppDisplay();
    
    // do we have the XNVCtrl extension loaded?
    if(!XNVCTRLQueryExtension(dpy, &evt_base, &err_base))
        return list;
    
    // core temp support?
    if(XNVCTRLQueryAttribute (dpy, 0, 0, 
       NV_CTRL_GPU_CORE_TEMPERATURE, &temp)
      )
         list.push_back(new NVidiaThermalSrc(inParent, "GPUCoreTemp", "NVidiaCore",
                        NV_CTRL_GPU_CORE_TEMPERATURE));
    
    // ambient temp support?
    if(XNVCTRLQueryAttribute (dpy, 0, 0, 
       NV_CTRL_AMBIENT_TEMPERATURE, &temp)
      )
         list.push_back(new NVidiaThermalSrc(inParent, "GPUAmbientTemp", "NVidiaAmbient",
                        NV_CTRL_AMBIENT_TEMPERATURE));
#endif
	return list;
}

void NVidiaThermalSrc::enable(bool inEnable){
	if(inEnable && !mRefreshTimer->isActive()){ // start the timer
		fetchValue();
		mRefreshTimer->start(3000);
	}else if(!inEnable && mRefreshTimer->isActive()) // stops the timer
		mRefreshTimer->stop();
}


void NVidiaThermalSrc::evaluateStdout(){
#ifndef HAVE_NVCONTROL
		QString val = i18n("n/a");
        
        // now see what it printed...
        QString ln;
        QString pout;
        while(mProcess->readln(ln) != -1)
            pout+= ln + '\n';
		QRegExp regexp("Attribute\\s'" + mID + "'.*(\\d+)\\.");
		if(regexp.search(pout) != -1)
		    val = formatTemperature(regexp.cap(1));
		mValue = val;
		emit valueUpdated(mValue); // calls updateLabel(mValue) of LabelSource
        
        // delete the object, to be ready for a new run
        delete mProcess;
        mProcess = 0;
#endif
}

#ifndef HAVE_NVCONTROL
void NVidiaThermalSrc::createProcess() {
    mProcess = new KProcIO;
    connect(mProcess, SIGNAL(processExited(KProcess*)), this, SLOT(evaluateStdout()));
    *mProcess << "nvidia-settings" << "-n"
                << "-q" << mID;
}
#endif

QString NVidiaThermalSrc::fetchValue(){
#ifndef HAVE_NVCONTROL
	if(!mProcess) {
        createProcess();
		if(!mProcess->start()) {
			mValue = "n/a";
            delete mProcess;
            mProcess = 0;
        }
    }
	return getValue();
#else
    int temp;
    Display * dpy = QPaintDevice::x11AppDisplay();
    if(XNVCTRLQueryAttribute (dpy, 0, 0, mAttrib, &temp))
        mValue = formatTemperature(QString::number(temp));
    else
        mValue = "n/a";
    emit valueUpdated(mValue); // calls updateLabel(mValue) of LabelSource
    return mValue;
#endif
}

