/***************************************************************************
 *   Copyright (C) 2007-2009 by Marcel Hasler                              *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This file is part of KGmailNotifier.                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/

#include "kgmailnotifier.h"
#include "configuration.h"

#include <KAction>
#include <KGlobal>
#include <KHelpMenu>
#include <KLocale>
#include <KMenu>
#include <KMessageBox>
#include <KProcess>
#include <KStandardDirs>
#include <KToolInvocation>
#include <kwallet.h>

#include <QMenu>
#include <QTextStream>

static QTextStream cout(stdout, QIODevice::WriteOnly);

#include <cstdlib>
using namespace std;

// ------------------------------------------------------------------------------------------------------------------- //
//    Construction & Destruction                                                                                       //
// ------------------------------------------------------------------------------------------------------------------- //

KGmailNotifier::KGmailNotifier()
    : KUniqueApplication(),
      m_configurationDialog(0)
{
    cout << endl;
    cout << "Welcome to KGmailNotifier." << endl;
    cout << "(C) 2007-2009 Marcel Hasler (mahasler@gmail.com)" << endl << endl;
    cout << "This software is released under the terms of the GNU General Public License Version 3." << endl << endl;
    cout << "Enjoy!" << endl << endl;

    // Cannot continue without the wallet
    if (!KWallet::Wallet::isEnabled())
    {
        KMessageBox::error(0, i18n("KWallet isn't enabled. Please make sure it is running and restart the application."),
                           i18n("Couldn't connect to KWallet"));
        std::exit(-1);
    }

    this->setQuitOnLastWindowClosed(false);

    // Set up popup window
    m_popupWindow = new PopupWindow();
    connect(m_popupWindow, SIGNAL(urlClicked()), this, SLOT(slotLaunchBrowser()));

    // Set up timer for checking mail
    m_checkMailTimer = new QTimer();
    m_checkMailTimer->setSingleShot(true);
    m_checkMailTimer->setInterval(Configuration::config()->checkInterval()*1000);
    connect(m_checkMailTimer, SIGNAL(timeout()), this, SLOT(slotCheckNewMail()));

    // Set up notification tools
    m_audioPlayer = new AudioPlayer();
    m_ledHandler = new LEDHandler();

    // Set up tray icon
    m_redIcon = KStandardDirs::locate("appdata", "pics/gmailRed.png");
    m_blueIcon = KStandardDirs::locate("appdata", "pics/gmailBlue.png");
    m_alertIcon = KStandardDirs::locate("appdata", "pics/gmailAlert.png");

    m_actionCollection = new KActionCollection(this);

    m_trayIcon = new TrayIcon(0);
    m_trayIcon->setIcon(m_redIcon, false, 0);

    // Add tray icon menu items
    KAction *action;

    // Check mail now
    action = m_actionCollection->addAction("check", this, SLOT(slotCheckNewMail()));
    action->setText(i18n("Check mail now"));
    action->setIcon(KIconLoader::global()->loadIcon("go-bottom", KIconLoader::NoGroup));
    m_trayIcon->contextMenu()->addAction(action);

    // Go to inbox
    action = m_actionCollection->addAction("inbox", this, SLOT(slotLaunchBrowser()));
    action->setText(i18n("Go to inbox"));
    action->setIcon(KIconLoader::global()->loadIcon("applications-internet", KIconLoader::NoGroup));
    m_trayIcon->contextMenu()->addAction(action);

    m_trayIcon->contextMenu()->addSeparator();

    // Configure
    action = m_actionCollection->addAction("configure", this, SLOT(slotShowConfigWindow()));
    action->setText(i18n("Configure"));
    action->setIcon(KIconLoader::global()->loadIcon("configure", KIconLoader::NoGroup));
    m_trayIcon->contextMenu()->addAction(action);

    // Add standard KDE help menu
    KHelpMenu *helpMenu = new KHelpMenu(0, KGlobal::mainComponent().aboutData(), false, 0);
    helpMenu->menu()->setIcon(KIconLoader::global()->loadIcon("help-browser", KIconLoader::NoGroup));
    m_trayIcon->contextMenu()->addMenu(helpMenu->menu());

    m_trayIcon->contextMenu()->addSeparator();

    // Silent/normal notification
    m_silentNotification = Configuration::config()->silentNotification();
    action = m_actionCollection->addAction("silent", this, SLOT(slotSilentNotificationSelected()));
    if (m_silentNotification)
    {
        action->setText(i18n("Normal notification"));
        action->setIcon(KIconLoader::global()->loadIcon("audio-volume-high", KIconLoader::NoGroup));
    }
    else
    {
        action->setText(i18n("Silent notification"));
        action->setIcon(KIconLoader::global()->loadIcon("audio-volume-muted", KIconLoader::NoGroup));
    }
    m_trayIcon->contextMenu()->addAction(action);

    // Done setting up tray icon, now show it
    m_trayIcon->show();
    connect(m_trayIcon, SIGNAL(clicked()), this, SLOT(slotTogglePopupWindow()));

    // Set up account
    QString username = Configuration::config()->username();
    QString password = Configuration::config()->password();
    if (username.isEmpty() || password.isEmpty())
        this->slotShowConfigWindow();

    m_account = new Account(username, password);
    connect(m_account, SIGNAL(connectionSuccessful()), this, SLOT(slotProcessMailInfo()));
    connect(m_account, SIGNAL(connectionError(const QString)), this, SLOT(slotProcessConnectionError(const QString)));

    // Done setting up. Time to check the mail.
    this->slotCheckNewMail();
}

// ------------------------------------------------------------------------------------------------------------------- //

KGmailNotifier::~KGmailNotifier()
{
    if (m_trayIcon)
        delete m_trayIcon;

    if (m_popupWindow)
        delete m_popupWindow;

    if (m_account)
        delete m_account;

    if (m_checkMailTimer)
        delete m_checkMailTimer;

    if (m_audioPlayer)
        delete m_audioPlayer;

    if (m_ledHandler)
        delete m_ledHandler;
}


// ------------------------------------------------------------------------------------------------------------------- //
//    Private slots                                                                                                    //
// ------------------------------------------------------------------------------------------------------------------- //

void KGmailNotifier::slotSilentNotificationSelected()
{
    m_silentNotification = !m_silentNotification;
    if (m_silentNotification)
    {
        m_actionCollection->action("silent")->setText(i18n("Normal notification"));
        m_actionCollection->action("silent")->setIcon(KIconLoader::global()->loadIcon("audio-volume-high", KIconLoader::NoGroup));
    }
    else
    {
        m_actionCollection->action("silent")->setText(i18n("Silent notification"));
        m_actionCollection->action("silent")->setIcon(KIconLoader::global()->loadIcon("audio-volume-muted", KIconLoader::NoGroup));
    }
    
    Configuration::config()->setSilentNotification(m_silentNotification);

    // Save configuration
    Configuration::config()->writeConfig();
}

// ------------------------------------------------------------------------------------------------------------------- //

void KGmailNotifier::slotTogglePopupWindow()
{
    m_popupWindow->toggle();
}

// ------------------------------------------------------------------------------------------------------------------- //

void KGmailNotifier::slotCheckNewMail()
{
    m_checkMailTimer->stop();
    m_trayIcon->setToolTip(i18n("Connecting to server..."));
    m_account->checkMail(Configuration::config()->checkInterval() - 1);
}

// ------------------------------------------------------------------------------------------------------------------- //

void KGmailNotifier::slotProcessMailInfo()
{
    unsigned int msgCount = m_account->messageCount();

    if (msgCount > 0)
    {
        m_trayIcon->setIcon(m_blueIcon, Configuration::config()->numberOnSystray(), msgCount);
        m_trayIcon->setToolTip(i18np("1 new message", "%1 new messages", msgCount));
        m_popupWindow->setMessageCount(msgCount);
        m_popupWindow->setSender(m_account->lastSender());
        m_popupWindow->setSubject(m_account->lastSubject());

        if (m_account->hasNewMail())
        {
            if (!m_silentNotification)
            {
                m_audioPlayer->play();

                if (Configuration::config()->showPopup())
                    this->slotTogglePopupWindow();
            }

            // Turn on LED
            m_ledHandler->setLedOn(true);
        }
    }

    else
    {
        m_trayIcon->setIcon(m_redIcon, false, msgCount);
        m_trayIcon->setToolTip(i18n("No new messages"));
        m_popupWindow->setMessageCount(0);

        // Turn off LED
        m_ledHandler->setLedOn(false);
    }

    // Restart timer
    m_checkMailTimer->start();
}

// ------------------------------------------------------------------------------------------------------------------- //

void KGmailNotifier::slotProcessConnectionError(const QString errorType)
{
    QString errorMessage;

    if (errorType == "timeout")
        errorMessage = i18n("Connection timeout!");
    else
        errorMessage = i18n("Connection failed!");

    m_trayIcon->setIcon(m_alertIcon, false, 0);
    m_trayIcon->setToolTip(errorMessage);

    m_popupWindow->setConnectionFailed();

    // Restart timer
    m_checkMailTimer->start();
}

// ------------------------------------------------------------------------------------------------------------------- //

void KGmailNotifier::slotLaunchBrowser()
{
    if (Configuration::config()->useDefaultBrowser())
        KToolInvocation::invokeBrowser("https://mail.google.com/mail");
    else // Custom browser
    {
        KProcess browserProc;
        browserProc << Configuration::config()->browserPath();
        browserProc << "https://mail.google.com/mail";
        browserProc.startDetached();
    }
}

// ------------------------------------------------------------------------------------------------------------------- //

void KGmailNotifier::slotShowConfigWindow()
{
    if (m_configurationDialog != 0)
        return;

    m_configurationDialog = new ConfigurationDialog(Configuration::config());

    connect(m_configurationDialog, SIGNAL(settingsChanged(const QString&)), this, SLOT(slotApplyConfiguration()));
    connect(m_configurationDialog, SIGNAL(finished()), this, SLOT(slotDeleteConfigWindow()));

    m_configurationDialog->show();
}

// ------------------------------------------------------------------------------------------------------------------- //

void KGmailNotifier::slotDeleteConfigWindow()
{
    m_configurationDialog->delayedDestruct();
    m_configurationDialog = 0;
}

// ------------------------------------------------------------------------------------------------------------------- //

void KGmailNotifier::slotApplyConfiguration()
{
    // Apply what's necessary when configuration has changed
    m_popupWindow->updateConfiguration();
    m_ledHandler->updateConfiguration();
    m_audioPlayer->updateConfiguration();
    m_account->setCredentials(Configuration::config()->username(), Configuration::config()->password());
    m_checkMailTimer->setInterval(Configuration::config()->checkInterval()*1000);
}

// ------------------------------------------------------------------------------------------------------------------- //

#include "kgmailnotifier.moc"
