/*
    collectionbuilder.cpp - Build a collection of ROMs

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "collectionbuilder.h"

// Kamefu includes
#include "systemmanager.h"
#include "rommetadatafactory.h"
#include "rommetadata.h"
#include "collectionaccess.h"

// Qt includes
#include <qvaluelist.h>
#include <qvaluestack.h>

// KDE includes
#include <kdirlister.h>
#include <kdebug.h>

namespace Kamefu
{

class CollectionBuilder::Private
{
public:
	Private() 
	 : dirLister(0L), recursive(true)
	{}
	~Private()
	{
		delete dirLister;
	}

	KDirLister *dirLister;
	QValueStack<KURL> directoriesStack;
	QValueList<KURL> directoryList;

	bool recursive;
};

CollectionBuilder::CollectionBuilder(QObject *parent, const char *name)
 : QObject(parent, name), d(new Private())
{
}


CollectionBuilder::~CollectionBuilder()
{
	kdDebug() << k_funcinfo << endl;
	delete d;
}


void CollectionBuilder::setDirectoryList(const KURL::List &directoryList)
{
	// Erase the old list.
	d->directoriesStack.clear();
	d->directoryList.clear();

	// Add the directories to the URL stack.
	KURL::List::const_iterator it;
	KURL::List::const_iterator end = directoryList.end();
	for(it = directoryList.begin(); it != end; ++it)
	{
		d->directoriesStack.push(*it);
		d->directoryList.append(*it);
	}
}

void CollectionBuilder::startBuild(bool recursive)
{
	// Clear current Dir Lister.
	if(d->dirLister)
	{
		disconnect(d->dirLister, SIGNAL(completed(const KURL &)));
		disconnect(d->dirLister, SIGNAL(newItems(const KFileItemList &)));
		disconnect(d->dirLister, SIGNAL(completed()));
		disconnect(this, SIGNAL(singleRomListed( const Kamefu::RomMetaInformation& )));
		d->dirLister->deleteLater();
		d->dirLister = 0L;
	}

	d->recursive = recursive;
	d->dirLister = new KDirLister();
	QStringList mimeTypeList(Kamefu::SystemManager::self()->getRegistredMimeTypes());

	// Add the mimetype for directories (for recursing into sub-directories)
	mimeTypeList.append( QString::fromUtf8("inode/directory") );
	// Add mimetype for ZIP files.
	mimeTypeList.append( QString::fromUtf8("application/x-zip") );

	d->dirLister->setMimeFilter(mimeTypeList);
	connect(d->dirLister, SIGNAL(completed(const KURL &)), this, SLOT(slotDirectoryFinished()));
	connect(d->dirLister, SIGNAL(newItems(const KFileItemList &)), this, SLOT(slotNewFiles(const KFileItemList& )));
	connect(d->dirLister, SIGNAL(completed()), this, SLOT(slotListingComplete()));
	
	// Insert listed rom in database.
	connect(this, SIGNAL(singleRomListed( const Kamefu::RomMetaInformation& )), CollectionAccess::self(), SLOT(insertNewRom( const Kamefu::RomMetaInformation& )));
	
	if( !d->directoriesStack.isEmpty() )
		d->dirLister->openURL(d->directoriesStack.pop(), true);
}

void CollectionBuilder::cancelBuild()
{
	
}

void CollectionBuilder::slotNewFiles(const KFileItemList &list)
{
	KFileItem *item;
	QPtrListIterator<KFileItem> it( list );

	emit beginListDirectory( list.count() );


	while ( (item = it.current()) != 0 ) 
	{
		// Do not try to extract metainformation from Directory.
		if(item->isDir())
		{
			// If we recurse into sub-directories, add to the stack.
			if(d->recursive)
			{
				kdDebug() << k_funcinfo << "Adding a new directory to the directories stack." << endl;
				d->directoriesStack.push(item->url());
				d->directoryList.append(item->url());
			}
		}
		++it;
	}
}

void CollectionBuilder::slotInsertItems(const KFileItemList &list)
{
	KFileItem *item;
	QPtrListIterator<KFileItem> it( list );
	while ( (item = it.current()) != 0 ) 
	{
		// Do not try to extract metainformation from Directory.
		if( !item->isDir() )
		{
			kdDebug() << k_funcinfo << "Listing " << item->url().prettyURL() << endl;
			
			// Do not extract the meta-information if the rom is already in database.
			if( !CollectionAccess::self()->isRomUrlInDatabase( item->url().path() ) )
			{
				RomMetaData *metaDataExtractor = RomMetaDataFactory::self()->createRomMetaData(item->url().path(), item->mimetype());
				RomMetaInformation romMetaInfo = metaDataExtractor->extractMetaInformation(item);
				
				// Satefy check because a zip file can contain no valid ROM image.
				if( romMetaInfo.isValid() )
				{
					emit singleRomListed(romMetaInfo);
				}
			}
		}

		emit itemListed();
		++it;
	}
}

void CollectionBuilder::slotListingComplete()
{
	kdDebug() << k_funcinfo << "BEGIN ROM TRANSACTION..." << endl;
	CollectionAccess::self()->beginRomTransaction();

	QValueList<KURL>::const_iterator it, itEnd = d->directoryList.constEnd();
	for(it = d->directoryList.constBegin(); it != itEnd; ++it)
	{
		KFileItemList fileList = d->dirLister->itemsForDir(*it);
		slotInsertItems(fileList);
	}

	kdDebug() << k_funcinfo << "END ROM TRANSACTION !" << endl;
	CollectionAccess::self()->endRomTransaction();

	emit collectionBuildFinished();
}

void CollectionBuilder::slotDirectoryFinished()
{
	kdDebug() << k_funcinfo << endl;
	// Start another scanning if the directories stack is not empty
	if( !d->directoriesStack.isEmpty() )
	{
		kdDebug() << k_funcinfo << "Starting another directory." << endl;
		d->dirLister->openURL(d->directoriesStack.pop(), true);
	}
}

void CollectionBuilder::slotBuildCanceled()
{
	
}

}

#include "collectionbuilder.moc"
