#include <bognor/br-queue.h>
#include <nbtk/nbtk.h>

#include "hrn.h"
#include "hrn-toolbar.h"
#include "hrn-sidebar.h"
#include "hrn-pin-manager.h"
#include "hrn-ui-controls.h"
#include "hrn-source-manager.h"
#include "hrn-queue.h"
#include "hrn-queue-controls.h"
#include "hrn-marshal.h"

enum {
    PROP_0,
    PROP_PIN_MANAGER,
    PROP_QUEUE,
    PROP_SOURCE_MANAGER,
};

enum {
    QUIT_REQUEST,
    SOURCE_CHANGED,
    FILTER_CHANGED,
    QUERY_CHANGED,
    PINNED,
    LAST_SIGNAL
};

struct _HrnUiControlsPrivate {
    HrnPinManager *pin_manager;
    HrnSourceManager *source_manager;

    NbtkWidget *toolbar;
    HrnZoomLevel current_zoom;

    HrnSidebar *sidebar;

    ClutterActor *lowlight;

    BrQueue *local_queue;
    ClutterActor *queues;
    HrnQueue *queue;
    HrnQueueControls *queue_controls;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_UI_CONTROLS, HrnUiControlsPrivate))
G_DEFINE_TYPE (HrnUiControls, hrn_ui_controls, CLUTTER_TYPE_GROUP);

static guint32 signals[LAST_SIGNAL] = {0, };

static void
hrn_ui_controls_finalize (GObject *object)
{
    G_OBJECT_CLASS (hrn_ui_controls_parent_class)->finalize (object);
}

static void
hrn_ui_controls_dispose (GObject *object)
{
    HrnUiControls *self = (HrnUiControls *) object;
    HrnUiControlsPrivate *priv = self->priv;

    if (priv->pin_manager) {
        g_object_unref (priv->pin_manager);
        priv->pin_manager = NULL;
    }

    if (priv->local_queue) {
        g_object_unref (priv->local_queue);
        priv->local_queue = NULL;
    }

    G_OBJECT_CLASS (hrn_ui_controls_parent_class)->dispose (object);
}

static void
playing_changed_cb (BrQueue       *queue,
                    gboolean       playing,
                    HrnUiControls *controls)
{
    HrnUiControlsPrivate *priv = controls->priv;

    hrn_queue_controls_set_playing (priv->queue_controls, playing);
}

static void
get_playing_reply (BrQueue *queue,
                   gboolean playing,
                   GError  *error,
                   gpointer data)
{
    HrnUiControls *controls = (HrnUiControls *) data;

    if (error != NULL) {
        g_warning ("Error getting initial play state: %s", error->message);
        return;
    }

    playing_changed_cb (queue, playing, controls);
}

static void
source_ready_cb (HrnSourceManager *manager,
                 HrnUiControls    *controls)
{
    HrnUiControlsPrivate *priv = controls->priv;

    br_queue_get_playing (priv->local_queue, get_playing_reply, controls);
}

static void
hrn_ui_controls_set_property (GObject      *object,
                              guint         prop_id,
                              const GValue *value,
                              GParamSpec   *pspec)
{
    HrnUiControls *self = (HrnUiControls *) object;
    HrnUiControlsPrivate *priv = self->priv;

    switch (prop_id) {
    case PROP_PIN_MANAGER:
        priv->pin_manager = g_value_dup_object (value);
        break;

    case PROP_QUEUE:
        priv->local_queue = g_value_dup_object (value);
        g_signal_connect (priv->local_queue, "playing-changed",
                          G_CALLBACK (playing_changed_cb), self);
        break;

    case PROP_SOURCE_MANAGER:
        priv->source_manager = g_value_dup_object (value);
        if (hrn_source_manager_is_ready (priv->source_manager)) {
            source_ready_cb (priv->source_manager, self);
        } else {
            g_signal_connect (priv->source_manager, "ready",
                              G_CALLBACK (source_ready_cb), self);
        }

        break;

    default:
        break;
    }
}

static void
hrn_ui_controls_get_property (GObject    *object,
                              guint       prop_id,
                              GValue     *value,
                              GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
quit_requested_cb (HrnToolbar    *toolbar,
                   HrnUiControls *uic)
{
    g_signal_emit (uic, signals[QUIT_REQUEST], 0);
}

static void
query_changed_cb (HrnToolbar    *toolbar,
                  const char    *query,
                  HrnUiControls *controls)
{
    g_signal_emit (controls, signals[QUERY_CHANGED], 0, query);
}

static void
set_source (HrnUiControls *controls,
            HrnSource     *source)
{
    g_signal_emit (controls, signals[SOURCE_CHANGED], 0, source);
}

static void
sidebar_source_changed_cb (HrnSidebar    *sidebar,
                           HrnSource     *source,
                           int            filter,
                           HrnPin        *pin,
                           HrnUiControls *controls)
{
    HrnUiControlsPrivate *priv = controls->priv;

    g_signal_emit (controls, signals[FILTER_CHANGED], 0, filter);
    set_source (controls, source);

    if (pin == NULL) {
        hrn_toolbar_set_query ((HrnToolbar *) priv->toolbar, "");
        hrn_pin_manager_select_pin (priv->pin_manager, NULL);
        return;
    }

    hrn_toolbar_set_query ((HrnToolbar *) priv->toolbar, pin->query);
    hrn_pin_manager_select_pin (priv->pin_manager, pin);
}

static void
request_play_changed_cb (HrnQueueControls *controls,
                         gboolean          playing,
                         HrnUiControls    *ui)
{
    HrnUiControlsPrivate *priv = ui->priv;

    if (playing) {
        br_queue_play (priv->local_queue);
    } else {
        br_queue_stop (priv->local_queue);
    }
}

static void
request_next_cb (HrnQueueControls *controls,
                 HrnUiControls    *ui)
{
    HrnUiControlsPrivate *priv = ui->priv;

    br_queue_next (priv->local_queue);
}

static void
pinned_cb (HrnToolbar    *toolbar,
           gboolean       pinned,
           HrnUiControls *ui)
{
    g_signal_emit (ui, signals[PINNED], 0, pinned);
}

static GObject *
hrn_ui_controls_constructor (GType                  type,
                             guint                  n_params,
                             GObjectConstructParam *params)
{
    GObject *object;
    HrnUiControls *controls;
    HrnUiControlsPrivate *priv;
    ClutterColor  black = { 0x00, 0x00, 0x00, 0xff };

    object = G_OBJECT_CLASS (hrn_ui_controls_parent_class)->constructor
        (type, n_params, params);

    controls = HRN_UI_CONTROLS (object);
    priv = controls->priv;

    priv->toolbar = hrn_toolbar_new (priv->pin_manager);
    clutter_container_add_actor ((ClutterContainer *) controls,
                                 (ClutterActor *) priv->toolbar);
    priv->current_zoom = hrn_toolbar_get_zoom ((HrnToolbar *) priv->toolbar);
    g_signal_connect (priv->toolbar, "query-changed",
                      G_CALLBACK (query_changed_cb), controls);
    g_signal_connect (priv->toolbar, "quit-requested",
                      G_CALLBACK (quit_requested_cb), controls);
    g_signal_connect (priv->toolbar, "pinned",
                      G_CALLBACK (pinned_cb), controls);

    priv->sidebar = hrn_sidebar_new (priv->pin_manager);
    clutter_container_add_actor ((ClutterContainer *) controls,
                                 (ClutterActor *) priv->sidebar);
    g_signal_connect (priv->sidebar, "source-changed",
                      G_CALLBACK (sidebar_source_changed_cb), controls);

    priv->lowlight = clutter_rectangle_new_with_color (&black);
    clutter_container_add_actor ((ClutterContainer *) controls, priv->lowlight);
    clutter_actor_hide (priv->lowlight);
    clutter_actor_set_opacity (priv->lowlight, 0x00);

    /* Should be a custom actor */
    priv->queues = clutter_group_new ();
    clutter_container_add_actor ((ClutterContainer *) controls, priv->queues);

    master_queue = priv->queue = hrn_queue_new (priv->local_queue,
                                                priv->source_manager);
    clutter_container_add_actor ((ClutterContainer *) priv->queues,
                                 (ClutterActor *) priv->queue);

    priv->queue_controls = g_object_new (HRN_TYPE_QUEUE_CONTROLS, NULL);
    g_signal_connect (priv->queue_controls, "request-playing-changed",
                      G_CALLBACK (request_play_changed_cb), controls);
    g_signal_connect (priv->queue_controls, "request-next",
                      G_CALLBACK (request_next_cb), controls);
    clutter_container_add_actor ((ClutterContainer *) priv->queues,
                                 (ClutterActor *) priv->queue_controls);
    clutter_actor_set_position ((ClutterActor *) priv->queue_controls, 0, 74);
    clutter_actor_set_size ((ClutterActor *) priv->queue_controls, 200, 52);

    return object;
}

static void
hrn_ui_controls_allocate (ClutterActor           *self,
                          const ClutterActorBox  *box,
                          ClutterAllocationFlags  flags)
{
    HrnUiControls *controls = (HrnUiControls *) self;
    HrnUiControlsPrivate *priv = controls->priv;
    float width, height;

    width = box->x2 - box->x1;
    height = box->y2 - box->y1;

    clutter_actor_set_y (CLUTTER_ACTOR (priv->sidebar), TOOLBAR_HEIGHT);
    clutter_actor_set_position (CLUTTER_ACTOR (priv->queues),
                                0, height - 126);

    clutter_actor_set_size (CLUTTER_ACTOR (priv->toolbar), width,
                            TOOLBAR_HEIGHT);
    clutter_actor_set_size (CLUTTER_ACTOR (priv->sidebar), 200, height);

    clutter_actor_set_size (priv->lowlight, width, height);

    CLUTTER_ACTOR_CLASS (hrn_ui_controls_parent_class)->allocate
        (self, box, flags);
}

static void
hrn_ui_controls_class_init (HrnUiControlsClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    ClutterActorClass *a_class = (ClutterActorClass *) klass;

    o_class->dispose = hrn_ui_controls_dispose;
    o_class->finalize = hrn_ui_controls_finalize;
    o_class->set_property = hrn_ui_controls_set_property;
    o_class->get_property = hrn_ui_controls_get_property;
    o_class->constructor = hrn_ui_controls_constructor;

    a_class->allocate = hrn_ui_controls_allocate;

    g_type_class_add_private (klass, sizeof (HrnUiControlsPrivate));

    g_object_class_install_property (o_class, PROP_PIN_MANAGER,
                                     g_param_spec_object ("pin-manager", "", "",
                                                          HRN_TYPE_PIN_MANAGER,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_CONSTRUCT_ONLY |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_QUEUE,
                                     g_param_spec_object ("local-queue", "", "",
                                                          BR_TYPE_QUEUE,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_CONSTRUCT_ONLY |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_SOURCE_MANAGER,
                                     g_param_spec_object ("source-manager", "", "",
                                                          HRN_TYPE_SOURCE_MANAGER,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_CONSTRUCT_ONLY |
                                                          G_PARAM_STATIC_STRINGS));

    signals[QUIT_REQUEST] = g_signal_new ("quit-requested",
                                          G_TYPE_FROM_CLASS (klass),
                                          G_SIGNAL_RUN_FIRST |
                                          G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                          g_cclosure_marshal_VOID__VOID,
                                          G_TYPE_NONE, 0);
    signals[SOURCE_CHANGED] = g_signal_new ("source-changed",
                                            G_TYPE_FROM_CLASS (klass),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                            g_cclosure_marshal_VOID__POINTER,
                                            G_TYPE_NONE, 1, G_TYPE_POINTER);
    signals[FILTER_CHANGED] = g_signal_new ("filter-changed",
                                            G_TYPE_FROM_CLASS (klass),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                            g_cclosure_marshal_VOID__INT,
                                            G_TYPE_NONE, 1, G_TYPE_INT);
    signals[QUERY_CHANGED] = g_signal_new ("query-changed",
                                           G_TYPE_FROM_CLASS (klass),
                                           G_SIGNAL_RUN_FIRST |
                                           G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                           g_cclosure_marshal_VOID__STRING,
                                           G_TYPE_NONE, 1, G_TYPE_STRING);
    signals[PINNED] = g_signal_new ("pinned",
                                    G_TYPE_FROM_CLASS (klass),
                                    G_SIGNAL_RUN_FIRST |
                                    G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                    g_cclosure_marshal_VOID__BOOLEAN,
                                    G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
}

static void
hrn_ui_controls_init (HrnUiControls *self)
{
    HrnUiControlsPrivate *priv;

    self->priv = GET_PRIVATE (self);
    priv = self->priv;
}

HrnToolbar *
hrn_ui_controls_get_toolbar (HrnUiControls *controls)
{
    HrnUiControlsPrivate *priv = controls->priv;

    return (HrnToolbar *) priv->toolbar;
}

HrnSidebar *
hrn_ui_controls_get_sidebar (HrnUiControls *controls)
{
    HrnUiControlsPrivate *priv = controls->priv;

    return priv->sidebar;
}

HrnZoomLevel
hrn_ui_controls_get_zoom (HrnUiControls *controls)
{
    HrnUiControlsPrivate *priv = controls->priv;

    return priv->current_zoom;
}

void
hrn_ui_controls_set_zoom (HrnUiControls *controls,
                          HrnZoomLevel   zoom_level)
{
    HrnUiControlsPrivate *priv = controls->priv;

    if (zoom_level == priv->current_zoom) {
        return;
    }

    hrn_toolbar_set_zoom ((HrnToolbar *) priv->toolbar,
                          ((double) zoom_level) / 4.0);
}

void
hrn_ui_controls_hide (HrnUiControls *controls)
{
    clutter_actor_animate ((ClutterActor *) controls, CLUTTER_LINEAR,
                           HRN_TO_THEATRE_DURATION,
                           "opacity", 0x00,
                           NULL);
}

void
hrn_ui_controls_show (HrnUiControls *controls)
{
    HrnUiControlsPrivate *priv = controls->priv;

    clutter_actor_show ((ClutterActor *) controls);

    clutter_actor_animate ((ClutterActor *) controls, CLUTTER_LINEAR,
                           HRN_FROM_THEATRE_DURATION,
                           "opacity", 0xFF,
                           NULL);

    hrn_toolbar_focused ((HrnToolbar *) priv->toolbar);
}

void
hrn_ui_controls_set_query (HrnUiControls *controls,
                           const char    *query)
{
    HrnUiControlsPrivate *priv = controls->priv;

    hrn_toolbar_set_query ((HrnToolbar *) priv->toolbar, query);
}

void
hrn_ui_controls_set_enabled (HrnUiControls *controls,
                             gboolean       enabled)
{
    HrnUiControlsPrivate *priv = controls->priv;

    if (enabled == FALSE) {
        clutter_actor_show (priv->lowlight);
        clutter_actor_animate (priv->lowlight, CLUTTER_LINEAR,
                               500, "opacity", 0x5c,
                               NULL);
    } else {
        clutter_actor_animate (priv->lowlight, CLUTTER_LINEAR, 500,
                               "opacity", 0x00,
                               "signal-swapped::completed", clutter_actor_hide,
                               priv->lowlight,
                               NULL);
    }
}

void
hrn_ui_controls_show_queue (HrnUiControls *controls,
                            gboolean       shown)
{
    HrnUiControlsPrivate *priv = controls->priv;

    if (shown) {
        clutter_actor_show ((ClutterActor *) priv->queue);
    } else {
        clutter_actor_hide ((ClutterActor *) priv->queue);
    }
}
