#include <bognor/br-queue.h>

#include "hrn.h"

#include "hrn-state-manager.h"
#include "hrn-cluster-node.h"
#include "hrn-theatre-ui.h"

enum {
    PROP_0,
    PROP_STATE_MANAGER,
    PROP_QUEUE,
};

enum {
    LEAVE_THEATRE,
    LAST_SIGNAL
};

struct _HrnTheatreUiPrivate {
    BrQueue *local_queue;
    HrnStateManager *state_manager;

    gboolean playing;
    gboolean playing_when_focused;
    gboolean audio_was_playing;

    HrnTheatre *theatre;
    HrnControls *controls;
    guint32 play_change_id;
    guint32 position_change_id;

    NbtkWidget *leave_button;

    guint32 controls_hide_handler_id;
    gboolean in_controls;

    HrnClusterNode *current_node;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_THEATRE_UI, HrnTheatreUiPrivate))
G_DEFINE_TYPE (HrnTheatreUi, hrn_theatre_ui, CLUTTER_TYPE_GROUP);
static guint32 signals[LAST_SIGNAL] = {0, };

static void
hrn_theatre_ui_finalize (GObject *object)
{
    G_OBJECT_CLASS (hrn_theatre_ui_parent_class)->finalize (object);
}

static void
hrn_theatre_ui_dispose (GObject *object)
{
    G_OBJECT_CLASS (hrn_theatre_ui_parent_class)->dispose (object);
}

static void
hrn_theatre_ui_set_property (GObject      *object,
                             guint         prop_id,
                             const GValue *value,
                             GParamSpec   *pspec)
{
    HrnTheatreUi *self = (HrnTheatreUi *) object;
    HrnTheatreUiPrivate *priv = self->priv;

    switch (prop_id) {
    case PROP_STATE_MANAGER:
        priv->state_manager = g_value_dup_object (value);
        break;

    case PROP_QUEUE:
        priv->local_queue = g_value_dup_object (value);
        break;

    default:
        break;
    }
}

static void
hrn_theatre_ui_get_property (GObject    *object,
                             guint       prop_id,
                             GValue     *value,
                             GParamSpec *pspec)
{
}

static void
hrn_theatre_ui_allocate (ClutterActor          *actor,
                         const ClutterActorBox *box,
                         ClutterAllocationFlags flags)
{
    HrnTheatreUi *ui = (HrnTheatreUi *) actor;
    HrnTheatreUiPrivate *priv = ui->priv;
    float width, height;

    CLUTTER_ACTOR_CLASS (hrn_theatre_ui_parent_class)->allocate
        (actor, box, flags);

    width = box->x2 - box->x1;
    height = box->y2 - box->y1;

    clutter_actor_set_position ((ClutterActor *) priv->controls,
                                0, height - 60);
    clutter_actor_set_size ((ClutterActor *) priv->controls, width, 60);
    clutter_actor_set_size ((ClutterActor *) priv->theatre, width, height);
}

static gboolean
fade_out (ClutterAnimation *animation,
          gpointer          userdata)
{
    HrnTheatreUi *ui = (HrnTheatreUi *) userdata;
    HrnTheatreUiPrivate *priv = ui->priv;

    clutter_actor_hide ((ClutterActor *) priv->controls);
    clutter_actor_hide ((ClutterActor *) priv->leave_button);

    /* We don't want to hide the cursor if we changed the view */
    if (clutter_actor_get_paint_visibility ((ClutterActor *) ui) == FALSE) {
        return FALSE;
    }

    clutter_stage_hide_cursor (CLUTTER_STAGE (hrn_window_get_stage (window)));
    return FALSE;
}

static gboolean
controls_hide_handler (gpointer userdata)
{
    HrnTheatreUi *ui = (HrnTheatreUi *) userdata;
    HrnTheatreUiPrivate *priv = ui->priv;

    if (priv->in_controls ||
        clutter_actor_get_paint_visibility ((ClutterActor *) priv->controls) == FALSE) {
        priv->controls_hide_handler_id = 0;
        return FALSE;
    }

    clutter_actor_animate ((ClutterActor *) priv->leave_button,
                           CLUTTER_LINEAR, CONTROLS_OUT_DURATION,
                           "opacity", 0x00,
                           NULL);
    clutter_actor_animate ((ClutterActor *) priv->controls,
                           CLUTTER_LINEAR, CONTROLS_OUT_DURATION,
                           "opacity", 0x00,
                           "signal::completed", fade_out, ui,
                           NULL);

    priv->controls_hide_handler_id = 0;
    return FALSE;
}

static void
wake_controls (HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    if (priv->controls_hide_handler_id) {
        g_source_remove (priv->controls_hide_handler_id);
        priv->controls_hide_handler_id = 0;
    }

    clutter_actor_show ((ClutterActor *) priv->controls);
    clutter_actor_show ((ClutterActor *) priv->leave_button);

    clutter_stage_show_cursor (CLUTTER_STAGE (hrn_window_get_stage (window)));

    clutter_actor_animate ((ClutterActor *) priv->controls,
                           CLUTTER_LINEAR, CONTROLS_IN_DURATION,
                           "opacity", 255,
                           NULL);
    clutter_actor_animate ((ClutterActor *) priv->leave_button,
                           CLUTTER_LINEAR, CONTROLS_IN_DURATION,
                           "opacity", 255,
                           NULL);
    priv->controls_hide_handler_id =
        g_timeout_add (HRN_CONTROLS_TIMEOUT, controls_hide_handler, ui);
}

static gboolean
controls_entered_cb (ClutterActor         *actor,
                     ClutterCrossingEvent *event,
                     HrnTheatreUi         *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    if (priv->controls_hide_handler_id > 0) {
        g_source_remove (priv->controls_hide_handler_id);
        priv->controls_hide_handler_id = 0;
    }

    priv->in_controls = TRUE;
    return FALSE;
}

static gboolean
controls_left_cb (ClutterActor         *actor,
                  ClutterCrossingEvent *event,
                  HrnTheatreUi         *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    priv->in_controls = FALSE;
    wake_controls (ui);

    return FALSE;
}

static void
leave_theatre (HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    if (priv->audio_was_playing) {
        br_queue_play (priv->local_queue);
        priv->audio_was_playing = FALSE;
    }

    clutter_stage_show_cursor (CLUTTER_STAGE (hrn_window_get_stage (window)));

    priv->playing = FALSE;
    hrn_theatre_set_playing (priv->theatre, FALSE);
    hrn_theatre_show_uri (priv->theatre, NULL, NULL);

    g_signal_handler_block (priv->controls, priv->play_change_id);
    hrn_controls_set_playing (priv->controls, priv->playing);
    g_signal_handler_unblock (priv->controls, priv->play_change_id);

    g_signal_emit (ui, signals[LEAVE_THEATRE], 0);
}

static void
leave_theatre_request (HrnControls  *controls,
                       HrnTheatreUi *ui)
{
    leave_theatre (ui);
}

static char *
get_uri_basename (const char *uri)
{
    char *unesc, *basename;

    basename = g_path_get_basename (uri);
    unesc = g_uri_unescape_string (basename, NULL);
    g_free (basename);

    return unesc;
}

static char *
get_title (BklItem *item)
{
    BklItemType type;
    const char *title;

    type = bkl_item_get_item_type (item);
    if (type == BKL_ITEM_TYPE_IMAGE) {
        title = bkl_item_image_get_title ((BklItemImage *) item);
        if (title == NULL) {
            const char *uri = bkl_item_get_uri (item);

            return get_uri_basename (uri);
        }

        return g_strdup (title);
    } else if (type == BKL_ITEM_TYPE_VIDEO) {
        const char *uri = bkl_item_get_uri (item);

        return get_uri_basename (uri);
    }

    return NULL;
}

static void
get_playing_cb (BrQueue      *queue,
                gboolean      playing,
                GError       *error,
                HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    if (error != NULL) {
        g_warning ("Error getting playing state: %s", error->message);
    } else {
        priv->audio_was_playing = playing;
    }

    br_queue_stop (queue);
}

static void
show_leaf_node (HrnTheatreUi   *ui,
                HrnClusterNode *leaf)
{
    HrnTheatreUiPrivate *priv = ui->priv;
    char *title;
    BklItem *item;

    priv->current_node = leaf;
    /* FIXME: Should add a weak ref to this */

    if (leaf->type == HRN_CLUSTER_NODE_TYPE_IMAGE) {
        PictureCluster *picture = (PictureCluster *) leaf->data;

        item = picture->item;
        hrn_controls_show_progress (priv->controls, FALSE);
    } else if (leaf->type == HRN_CLUSTER_NODE_TYPE_VIDEO) {
        VideoCluster *video = (VideoCluster *) leaf->data;

        item = video->item;
        hrn_controls_show_progress (priv->controls, TRUE);

        br_queue_get_playing (priv->local_queue,
                              (BrQueueGetPlayingCallback) get_playing_cb, ui);
        br_queue_stop (priv->local_queue);
    } else {
        return;
    }

    hrn_theatre_show_item (priv->theatre, item);

    title = get_title (item);
    hrn_controls_set_title (priv->controls, title);
    g_free (title);

    /* FIXME: We need to start it playing here again because clutter-helix
       stops the video player when changing URI */
    if (priv->playing) {
        hrn_theatre_set_playing (priv->theatre, TRUE);
    }
}

static void
request_next_cb (HrnControls  *controls,
                 HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;
    HrnClusterNode *leaf;

    leaf = priv->current_node;
    if (leaf == NULL) {
        leave_theatre (ui);
        return;
    }

    leaf = hrn_cluster_node_get_next_leaf (leaf);
    if (leaf == NULL) {
        leave_theatre (ui);
        return;
    }

    while (leaf->hidden == TRUE) {
        leaf = hrn_cluster_node_get_next_leaf (leaf);
        if (leaf == NULL) {
            leave_theatre (ui);
            return;
        }
    }

    show_leaf_node (ui, leaf);
    if (priv->playing) {
        hrn_theatre_set_playing (priv->theatre, TRUE);
    }
}

static void
request_previous_cb (HrnControls  *controls,
                     HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;
    HrnClusterNode *leaf;

    leaf = priv->current_node;
    if (leaf == NULL) {
        leave_theatre (ui);
        return;
    }

    leaf = hrn_cluster_node_get_previous_leaf (leaf);
    if (leaf == NULL) {
        leave_theatre (ui);
        return;
    }

    while (leaf->hidden == TRUE) {
        leaf = hrn_cluster_node_get_previous_leaf (leaf);
        if (leaf == NULL) {
            leave_theatre (ui);
            return;
        }
    }

    show_leaf_node (ui, leaf);
    if (priv->playing) {
        hrn_theatre_set_playing (priv->theatre, TRUE);
    }
}

static void
request_playing_change_cb (HrnControls  *controls,
                           gboolean      playing,
                           HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    hrn_theatre_set_playing (priv->theatre, playing);
    priv->playing = playing;
}

static void
theatre_finished_cb (HrnTheatre *theatre,
                     HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;
    HrnClusterNode *leaf = NULL;

    if (priv->current_node &&
        priv->current_node->type != HRN_CLUSTER_NODE_TYPE_VIDEO) {
        leaf = hrn_cluster_node_get_next_leaf (priv->current_node);
    }

    if (leaf == NULL) {
        leave_theatre (ui);
        return;
    }

    while (leaf->hidden == TRUE) {
        leaf = hrn_cluster_node_get_next_leaf (leaf);
        if (leaf == NULL) {
            leave_theatre (ui);
            return;
        }
    }

    show_leaf_node (ui, leaf);
    if (priv->playing) {
        hrn_theatre_set_playing (priv->theatre, TRUE);
    }
}

static gboolean
theatre_key_press_cb (ClutterActor *actor,
                      ClutterEvent *event,
                      HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    switch (clutter_event_get_key_symbol ((ClutterEvent *) event)) {
    case CLUTTER_space:
        priv->playing = !(priv->playing);

        hrn_theatre_set_playing (priv->theatre, priv->playing);
        g_signal_handler_block (priv->controls, priv->play_change_id);
        hrn_controls_set_playing (priv->controls, priv->playing);
        g_signal_handler_unblock (priv->controls, priv->play_change_id);
        break;

    case CLUTTER_Right:
        request_next_cb (priv->controls, ui);
        break;

    case CLUTTER_Left:
        request_previous_cb (priv->controls, ui);
        break;

    case CLUTTER_Escape:
        leave_theatre (ui);
        break;

    default:
        break;
    }

    return FALSE;
}

static void
position_changed_cb (HrnControls  *controls,
                     double        position,
                     HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    /* Don't need to block here because the theatre doesn't emit its
       signal when calling manually */
    hrn_theatre_set_position (priv->theatre, position);
}

static void
theatre_position_changed_cb (HrnTheatre   *theatre,
                             double        position,
                             HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    g_signal_handler_block (priv->controls, priv->position_change_id);
    hrn_controls_set_position (priv->controls, position);
    g_signal_handler_unblock (priv->controls, priv->position_change_id);
}

static void
theatre_duration_changed_cb (HrnTheatre   *theatre,
                             double        duration,
                             HrnTheatreUi *ui)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    hrn_controls_set_duration (priv->controls, (guint64) duration);
}

static void
leave_clicked_cb (NbtkButton   *button,
                  HrnTheatreUi *ui)
{
    leave_theatre (ui);
}

static GObject *
hrn_theatre_ui_constructor (GType                  type,
                            guint                  n_params,
                            GObjectConstructParam *params)
{
    GObject *object;
    HrnTheatreUi *ui;
    HrnTheatreUiPrivate *priv;

    object = G_OBJECT_CLASS (hrn_theatre_ui_parent_class)->constructor
        (type, n_params, params);
    ui = (HrnTheatreUi *) object;
    priv = ui->priv;

    priv->theatre = hrn_theatre_new (priv->state_manager);
    /* hrn_theatre = (ClutterActor *) priv->theatre; */
    g_signal_connect (priv->theatre, "finished",
                      G_CALLBACK (theatre_finished_cb), ui);
    g_signal_connect (priv->theatre, "position-changed",
                      G_CALLBACK (theatre_position_changed_cb), ui);
    g_signal_connect (priv->theatre, "duration-changed",
                      G_CALLBACK (theatre_duration_changed_cb), ui);
    clutter_container_add_actor (CLUTTER_CONTAINER (ui),
                                 CLUTTER_ACTOR (priv->theatre));

    /* Connect to this here to trap the Space keypress because the theatre
       doesn't know (or care) whether its playing or not */
    g_signal_connect (priv->theatre, "key-press-event",
                      G_CALLBACK (theatre_key_press_cb), ui);

    priv->leave_button = nbtk_button_new ();
    clutter_actor_set_name ((ClutterActor *) priv->leave_button,
                            "back-button");
    clutter_actor_set_position (CLUTTER_ACTOR (priv->leave_button), 8, 8);
    g_object_set (G_OBJECT (priv->leave_button),
                  "transition-duration", 100,
                  NULL);

    clutter_container_add_actor (CLUTTER_CONTAINER (ui),
                                 CLUTTER_ACTOR (priv->leave_button));
    g_signal_connect (priv->leave_button, "clicked",
                      G_CALLBACK (leave_clicked_cb), ui);

    priv->controls = hrn_controls_new ();
    g_signal_connect (priv->controls, "enter-event",
                      G_CALLBACK (controls_entered_cb), ui);
    g_signal_connect (priv->controls, "leave-event",
                      G_CALLBACK (controls_left_cb), ui);
    g_signal_connect (priv->controls, "leave-theatre",
                      G_CALLBACK (leave_theatre_request), ui);
    g_signal_connect (priv->controls, "request-next",
                      G_CALLBACK (request_next_cb), ui);
    g_signal_connect (priv->controls, "request-previous",
                      G_CALLBACK (request_previous_cb), ui);
    priv->play_change_id = g_signal_connect (priv->controls,
                                             "request-playing-changed",
                                             G_CALLBACK (request_playing_change_cb), ui);
    priv->position_change_id = g_signal_connect (priv->controls,
                                                 "position-changed",
                                                 G_CALLBACK (position_changed_cb), ui);

    clutter_container_add_actor (CLUTTER_CONTAINER (ui),
                                 CLUTTER_ACTOR (priv->controls));

    clutter_actor_hide (CLUTTER_ACTOR (priv->theatre));
    clutter_actor_hide (CLUTTER_ACTOR (priv->controls));

    return object;
}

static gboolean
hrn_theatre_ui_capture_event (ClutterActor *actor,
                              ClutterEvent *event)
{
    switch (event->any.type) {
    case CLUTTER_LEAVE:
    /* Fall through */

    case CLUTTER_KEY_PRESS:
    case CLUTTER_KEY_RELEASE:

#if 0
        /* This should be tracked somewhere else */
        if (event->key.keyval == CLUTTER_q &&
            clutter_event_get_state (event) & CLUTTER_CONTROL_MASK) {
            hrn_quit ();
        }
#endif

        /* some keybindings should not wake the controls */
        if (event->key.keyval == CLUTTER_Shift_L ||
            event->key.keyval == CLUTTER_Shift_R ||
            event->key.keyval == CLUTTER_space) {
            return FALSE;
        }
        /* Fall through */

    case CLUTTER_MOTION:
    /* Fall through */

    case CLUTTER_BUTTON_PRESS:
    case CLUTTER_BUTTON_RELEASE:
    case CLUTTER_SCROLL:
        wake_controls ((HrnTheatreUi *) actor);
        break;

    default:
        break;
    }
  return FALSE;
}

static void
hrn_theatre_ui_class_init (HrnTheatreUiClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    ClutterActorClass *a_class = (ClutterActorClass *) klass;

    o_class->dispose = hrn_theatre_ui_dispose;
    o_class->finalize = hrn_theatre_ui_finalize;
    o_class->set_property = hrn_theatre_ui_set_property;
    o_class->get_property = hrn_theatre_ui_get_property;
    o_class->constructor = hrn_theatre_ui_constructor;

    a_class->allocate = hrn_theatre_ui_allocate;
    a_class->captured_event = hrn_theatre_ui_capture_event;

    g_type_class_add_private (klass, sizeof (HrnTheatreUiPrivate));

    g_object_class_install_property (o_class, PROP_STATE_MANAGER,
                                     g_param_spec_object ("state-manager",
                                                          "", "",
                                                          HRN_TYPE_STATE_MANAGER,
                                                          G_PARAM_CONSTRUCT_ONLY |
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_QUEUE,
                                     g_param_spec_object ("local-queue",
                                                          "", "",
                                                          BR_TYPE_QUEUE,
                                                          G_PARAM_CONSTRUCT_ONLY |
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));
    signals[LEAVE_THEATRE] = g_signal_new ("leave-theatre",
                                           G_TYPE_FROM_CLASS (o_class),
                                           G_SIGNAL_RUN_FIRST |
                                           G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                           g_cclosure_marshal_VOID__VOID,
                                           G_TYPE_NONE, 0);
}

static void
hrn_theatre_ui_init (HrnTheatreUi *self)
{
    HrnTheatreUiPrivate *priv = GET_PRIVATE (self);

    self->priv = priv;

    clutter_actor_set_reactive ((ClutterActor *) self, TRUE);
    priv->playing = FALSE;
    priv->playing_when_focused = TRUE;
}

struct _completed_data {
    HrnTheatreUi *ui;
    HrnTheatreUiFinishedCb callback;
    gpointer userdata;
};

static void
show_finished (ClutterAnimation       *animation,
               struct _completed_data *closure)
{
    closure->callback (closure->ui, TRUE, closure->userdata);
    g_slice_free (struct _completed_data, closure);
}

void
hrn_theatre_ui_show (HrnTheatreUi          *ui,
                     HrnTheatreUiFinishedCb callback,
                     gpointer               userdata)
{
  HrnTheatreUiPrivate *priv = ui->priv;
  struct _completed_data *closure = g_slice_new0 (struct _completed_data);

  closure->ui = ui;
  closure->callback = callback;
  closure->userdata = userdata;

  clutter_stage_set_key_focus (CLUTTER_STAGE (hrn_window_get_stage (window)),
                               (ClutterActor *) priv->theatre);

  clutter_actor_show ((ClutterActor *) priv->theatre);
  clutter_actor_show ((ClutterActor *) priv->controls);

  /* FIXME: Emit a signal */
  clutter_actor_animate ((ClutterActor *) ui,
                         CLUTTER_EASE_IN_OUT_CUBIC,
                         HRN_TO_THEATRE_DURATION,
                         "opacity", 0xff,
                         "signal::completed", show_finished, closure,
                         NULL);


  wake_controls (ui);   /* wake it so that it can do away .. */
}

static void
hide_finished (ClutterAnimation       *animation,
               struct _completed_data *closure)
{
    if (closure->callback) {
        closure->callback (closure->ui, FALSE, closure->userdata);
    }
    g_slice_free (struct _completed_data, closure);
}

void
hrn_theatre_ui_hide (HrnTheatreUi          *ui,
                     HrnTheatreUiFinishedCb callback,
                     gpointer               userdata)
{
  struct _completed_data *closure = g_slice_new0 (struct _completed_data);

  g_print ("Hiding theatre ui\n");
  closure->ui = ui;
  closure->callback = callback;
  closure->userdata = userdata;

  /* FIXME: Emit a signal */
  clutter_actor_animate ((ClutterActor *) ui,
                         CLUTTER_EASE_IN_OUT_CUBIC,
                         HRN_FROM_THEATRE_DURATION,
                         "opacity", 0x00,
                         "signal::completed", hide_finished, closure,
                         NULL);
}

void
hrn_theatre_ui_show_node (HrnTheatreUi   *ui,
                          HrnClusterNode *node)
{
    HrnClusterNode *leaf;

    leaf = hrn_cluster_node_get_first_leaf (node);
    if (leaf == NULL) {
        leave_theatre (ui);
        return;
    }

    while (leaf->hidden) {
        leaf = hrn_cluster_node_get_next_leaf (leaf);

        if (leaf == NULL) {
            leave_theatre (ui);
            return;
        }
    }
    show_leaf_node (ui, leaf);
}

void
hrn_theatre_ui_set_playing (HrnTheatreUi *ui,
                            gboolean      playing)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    priv->playing = playing;
    hrn_theatre_set_playing (priv->theatre, priv->playing);

    g_signal_handler_block (priv->controls, priv->play_change_id);
    hrn_controls_set_playing (priv->controls, priv->playing);
    g_signal_handler_unblock (priv->controls, priv->play_change_id);
}

void
hrn_theatre_ui_show_uri (HrnTheatreUi *ui,
                         const char   *uri,
                         const char   *mimetype)
{
    HrnTheatreUiPrivate *priv = ui->priv;
    char *title;

    priv->current_node = NULL;

    if (g_str_has_prefix (mimetype, "image/")) {
        hrn_controls_show_progress (priv->controls, FALSE);
    } else if (g_str_has_prefix (mimetype, "video/")) {
        hrn_controls_show_progress (priv->controls, TRUE);
    } else {
        return;
    }

    hrn_theatre_show_uri (priv->theatre, uri, mimetype);

    title = g_path_get_basename (uri);
    hrn_controls_set_title (priv->controls, title);
    g_free (title);
}

void
hrn_theatre_ui_set_focused (HrnTheatreUi *ui,
                            gboolean      focused)
{
    HrnTheatreUiPrivate *priv = ui->priv;

    if (focused) {
        hrn_theatre_ui_set_playing (ui, priv->playing_when_focused);
    } else {
        priv->playing_when_focused = priv->playing;
        hrn_theatre_ui_set_playing (ui, FALSE);
    }
}
