#include "hrn.h"
#include "hrn-texture-cache.h"
#include "hrn-square-clone.h"

enum {
    PROP_0,
};

struct _HrnTextureCachePrivate {
    GHashTable *cache;
};

typedef struct _HrnCacheEntry {
    ClutterActor *original;
    int clone_count;
} HrnCacheEntry;

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_TEXTURE_CACHE, HrnTextureCachePrivate))
G_DEFINE_TYPE (HrnTextureCache, hrn_texture_cache, G_TYPE_OBJECT);

static HrnTextureCache *__cache_singleton = NULL;
static ClutterActor *default_audio = NULL;
static ClutterActor *default_image = NULL;
static ClutterActor *default_video = NULL;

static void
hrn_texture_cache_finalize (GObject *object)
{
    HrnTextureCache *self = (HrnTextureCache *) object;
    HrnTextureCachePrivate *priv = self->priv;

    g_hash_table_destroy (priv->cache);
    __cache_singleton = NULL;

    G_OBJECT_CLASS (hrn_texture_cache_parent_class)->finalize (object);
}

static void
hrn_texture_cache_dispose (GObject *object)
{
    G_OBJECT_CLASS (hrn_texture_cache_parent_class)->dispose (object);
}

static void
hrn_texture_cache_class_init (HrnTextureCacheClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;

    o_class->dispose = hrn_texture_cache_dispose;
    o_class->finalize = hrn_texture_cache_finalize;

    g_type_class_add_private (klass, sizeof (HrnTextureCachePrivate));
}

static void
free_cache_entry (gpointer data)
{
    HrnCacheEntry *entry = (HrnCacheEntry *) data;

    g_slice_free (HrnCacheEntry, entry);
}

static void
hrn_texture_cache_init (HrnTextureCache *self)
{
    HrnTextureCachePrivate *priv = GET_PRIVATE (self);
    self->priv = priv;

    priv->cache = g_hash_table_new_full (g_str_hash, g_str_equal,
                                         g_free, free_cache_entry);
}

HrnTextureCache *
hrn_texture_cache_get_default (void)
{
    if (G_UNLIKELY (__cache_singleton == NULL)) {
        __cache_singleton = g_object_new (HRN_TYPE_TEXTURE_CACHE, NULL);
    }

    return __cache_singleton;
}

static void
on_clone_finalized (gpointer data,
                    GObject *where_the_object_was)
{
    HrnCacheEntry *entry = (HrnCacheEntry *) data;

    entry->clone_count--;

    if (entry->clone_count == 0) {
        /* FIXME: Track/destroy original? */
    }
}

ClutterActor *
hrn_texture_cache_get_texture (HrnTextureCache *cache,
                               const char      *uri)
{
    HrnTextureCachePrivate *priv = cache->priv;
    ClutterActor *clone;
    HrnCacheEntry *entry;

    g_return_val_if_fail (uri != NULL, NULL);

    entry = g_hash_table_lookup (priv->cache, uri);
    if (entry == NULL) {
        ClutterActor *original;

        original = clutter_texture_new ();
        clutter_texture_set_filter_quality ((ClutterTexture *) original,
                                            CLUTTER_TEXTURE_QUALITY_MEDIUM);
        clutter_texture_set_load_async ((ClutterTexture *) original, TRUE);

        if (strstr (uri, "://")) {
            char *path;
            path = g_filename_from_uri (uri, NULL, NULL);
            clutter_texture_set_from_file ((ClutterTexture *) original,
                                           path, NULL);
            g_free (path);
        } else {
            clutter_texture_set_from_file ((ClutterTexture *) original,
                                           uri, NULL);
        }

        entry = g_slice_new (HrnCacheEntry);
        entry->original = original;
        entry->clone_count = 0;
        g_object_ref_sink (original);

        g_hash_table_insert (priv->cache, g_strdup (uri), entry);
   }

    clone = hrn_square_clone_new ((ClutterTexture *) entry->original);
    entry->clone_count++;

    g_object_weak_ref (G_OBJECT (clone), on_clone_finalized, entry);
    return clone;
}

static ClutterActor *
create_default (const char *path)
{
    ClutterActor *texture;
    ClutterActor *stage;
    GError *error = NULL;

    stage = hrn_window_get_stage (window);

    texture = clutter_texture_new_from_file (path, &error);
    if (error != NULL) {
        g_warning ("Error loading %s: %s", path, error->message);
        g_error_free (error);
    } else {
        clutter_container_add_actor ((ClutterContainer *) stage, texture);
        clutter_actor_hide (texture);
    }

    return texture;
}

ClutterActor *
hrn_texture_cache_get_default_texture (HrnTextureCache *cache,
                                       BklItemType      type)
{
    switch (type) {
    case BKL_ITEM_TYPE_AUDIO:
        if (G_UNLIKELY (default_audio == NULL)) {
            default_audio = create_default (THEMEDIR "/hrn-generic-album.png");
        }

        return clutter_clone_new (default_audio);

    case BKL_ITEM_TYPE_IMAGE:
        if (G_UNLIKELY (default_image == NULL)) {
            default_image = create_default (THEMEDIR "/hrn-generic-image.png");
        }

        return clutter_clone_new (default_image);

    case BKL_ITEM_TYPE_VIDEO:
        if (G_UNLIKELY (default_video == NULL)) {
            default_video = create_default (THEMEDIR "/hrn-generic-video.png");
        }

        return clutter_clone_new (default_video);

    default:
        break;
    }

    return NULL;
}
