
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>

#define BLOCKSIZE	8192

#define ARRAYSIZE(x) (sizeof (x) / sizeof ((x)[0]))

static __inline__ int printable (int c)
{
   return ((c >= 32 && c <= 126) ||
		   (c >= 174 && c <= 223) ||
		   (c >= 242 && c <= 243) ||
		   (c >= 252 && c <= 253));
}

static void hex (char *buf,uint32_t value,uint32_t digits)
{
   uint32_t tmp;

   while (digits--)
	 {
		tmp = (value >> (digits << 2)) & 15;
		tmp += tmp < 10 ? '0' : ('a' - 10);
		*buf++ = tmp;
	 }

   *buf = '\0';
}

static void dump16 (uintptr_t addr,const uint8_t *s,uintptr_t offset,size_t len)
{
   if (len)
	 {
		static char buf[81];
		size_t i,n = 8;

		hex (buf,addr,8);
		buf[n++] = ' ';
		buf[n++] = '-';
		buf[n++] = ' ';

		for (i = 0; i < len; i++)
		  {
			 if (i && !(i & 3))
			   buf[n++] = ' ';

			 hex (buf + n,s[i],2);
			 n += 2;
			 buf[n++] = ' ';
		  }

		for ( ; i < 16; i++)
		  {
			 if (i && !(i & 3))
			   buf[n++] = ' ';

			 buf[n++] = ' ';
			 buf[n++] = ' ';
			 buf[n++] = ' ';
		  }

		buf[n++] = ' ';

		for (i = 0; i < len; i++)
		  buf[n++] = printable (s[i]) ? s[i] : '.';

		buf[n++] = '\n';

		write (STDOUT_FILENO,buf,n);
	 }
}

static void err (const char *msg[],size_t n)
{
   size_t i;

   for (i = 0; i < n; i++)
	 if (msg[i] != NULL)
	   write (STDERR_FILENO,msg[i],strlen (msg[i]));
}

static int hexdump (const char *filename,int fd)
{
   static uint32_t offset = 0;
   static uint8_t buf[BLOCKSIZE];
   ssize_t size;
   const uint8_t *s;
   size_t i;

   for (;;)
	 {
		if ((size = read (fd,buf,BLOCKSIZE)) < 0)
		  {
			 static const char *msg[] =
			   {
				  "read on file ",
				  NULL,
				  " failed: ",
				  NULL,
				  "\n"
			   };

			 msg[1] = filename;
			 msg[3] = strerror (errno);

			 err (msg,ARRAYSIZE (msg));
			 return (-1);
		  }

		if (!size)
		  return (0);

		s = buf;

		for (i = 0; i < size >> 4; i++, s += 16)
		  {
			 dump16 (offset,s,(uintptr_t) s - (uintptr_t) buf,16);
			 offset += 16;
		  }

		dump16 (offset,s,(uintptr_t) s - (uintptr_t) buf,size & 15);
		offset += size & 15;
	 }
}

int main (int argc,char *argv[])
{
   int i,fd;

   if (argc >= 2)
	 for (i = 1; i < argc; i++)
	   {
		  if ((fd = open (argv[i],O_RDONLY)) < 0)
			{
			   static const char *msg[] =
				 {
					"failed to open ",
					NULL,
					" for reading\n"
				 };

			   msg[1] = argv[i];
			   err (msg,ARRAYSIZE (msg));
			   exit (EXIT_FAILURE);
			}

		  if (hexdump (argv[i],fd))
			{
			   close (fd);
			   exit (EXIT_FAILURE);
			}

		  close (fd);
	   }
   else if (hexdump ("(stdin)",STDIN_FILENO))
	 exit (EXIT_FAILURE);

   exit (EXIT_SUCCESS);
}

