{-# LANGUAGE CPP, ForeignFunctionInterface, PatternGuards #-}
-----------------------------------------------------------------------------
-- |
-- Module      :  Text.CSL.Input.Bibutils
-- Copyright   :  (C) 2008 Andrea Rossato
-- License     :  BSD3
--
-- Maintainer  :  andrea.rossato@unitn.it
-- Stability   :  unstable
-- Portability :  unportable
--
-----------------------------------------------------------------------------

module Text.CSL.Input.Bibutils
    ( readBiblioFile
    ) where

import Data.Char
import System.FilePath ( takeExtension )
import Text.CSL.Reference
import Text.CSL.Input.Json
import Text.CSL.Input.MODS

#ifdef USE_BIBUTILS
import Control.Exception ( bracket )
import Control.Monad.Trans ( liftIO )
import System.FilePath ( (</>), (<.>) )
import System.IO.Error ( isAlreadyExistsError )
import System.Directory
import Text.Bibutils
#endif

-- | Read a file with a bibliographic database. The database format
-- is recognized by the file extension.
--
-- Supported formats are: @json@, @mods@, @bibtex@, @biblatex@, @ris@,
-- @endnote@, @endnotexml@, @isi@, @medline@, and @copac@.
readBiblioFile :: FilePath -> IO [Reference]
#ifdef USE_BIBUTILS
readBiblioFile f
    = case getExt f of
        ".mods"    -> readBiblioFile' f mods_in
        ".bib"     -> readBiblioFile' f bibtex_in
        ".bbx"     -> readBiblioFile' f biblatex_in
        ".ris"     -> readBiblioFile' f ris_in
        ".enl"     -> readBiblioFile' f endnote_in
        ".xml"     -> readBiblioFile' f endnotexml_in
        ".wos"     -> readBiblioFile' f isi_in
        ".medline" -> readBiblioFile' f medline_in
        ".copac"   -> readBiblioFile' f copac_in
        ".json"    -> readJsonInput f
        _          -> error $ "citeproc: the format of the bibliographic database could not be recognized\n" ++
                              "using the file extension."

#else
readBiblioFile f
    | ".mods" <- getExt f = readModsCollectionFile f
    | ".json" <- getExt f = readJsonInput f
    | otherwise           = error $ "citeproc: Bibliography format not supported.\n" ++
                                    "citeproc-hs was not compiled with bibutils support."
#endif

getExt :: String -> String
getExt = takeExtension . map toLower

#ifdef USE_BIBUTILS
readBiblioFile' :: FilePath -> BiblioIn -> IO [Reference]
readBiblioFile' fin bin
    | bin == mods_in = readModsCollectionFile fin
    | otherwise      = withTempDir "citeproc" $ \tdir -> do
                          let tfile = tdir </> "bibutils-tmp"
                          param <- bibl_initparams bin mods_out "hs-bibutils"
                          bibl  <- bibl_init
                          unsetBOM        param
                          setNoSplitTitle param
                          setCharsetIn    param bibl_charset_unicode
                          setCharsetOut   param bibl_charset_unicode
                          _ <- bibl_read  param bibl fin
                          _ <- bibl_write param bibl tfile
                          bibl_free bibl
                          bibl_freeparams param
                          refs <- readModsCollectionFile tfile
                          return $! refs

-- | Perform a function in a temporary directory and clean up.
withTempDir :: FilePath -> (FilePath -> IO a) -> IO a
withTempDir baseName = bracket (createTempDir 0 baseName)
  (removeDirectoryRecursive)

-- | Create a temporary directory with a unique name.
createTempDir :: Integer -> FilePath -> IO FilePath
createTempDir num baseName = do
  sysTempDir <- getTemporaryDirectory
  let dirName = sysTempDir </> baseName <.> show num
  liftIO $ catch (createDirectory dirName >> return dirName) $
      \e -> if isAlreadyExistsError e
               then createTempDir (num + 1) baseName
               else ioError e
#endif
