/* gt_x.c - X display driver
 *
 * Copyright (C) 1997, 1998, 1999 Free Software Foundation
 * Copyright (C) 1995, 1996 Eric M. Ludlam
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 * 
 * This code is widget set independed X Intrinsics code.  Widget sets that
 * are not based on Xt will not be able to use this.
 * 
 * $Log: gt_x.c,v $
 * Revision 1.30  1999/08/26 11:58:27  zappo
 * X_read tracks the eightbit flag.
 *
 * Revision 1.29  1998/04/26 15:06:41  zappo
 * Moved all the menu code into a new file.
 *
 * Revision 1.28  1998/01/10 12:44:30  zappo
 * Fixed "hangup" menu item.
 *
 * Revision 1.27  1997/12/14 19:17:41  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.26  1997/10/25 03:07:56  zappo
 * Fixed closing of display (XtCloseDisplay exited on me!)
 *
 * Revision 1.25  1997/10/25 02:45:30  zappo
 * Removed old athena header files.  Fixed a menu item's callback data.
 * Adjusted X_xread to not clobber existing data if it's there.
 *
 * Revision 1.24  1997/10/23 01:56:15  zappo
 * Added a default paste action that pastes over the net for all Xt based
 * interfaces.
 *
 * Revision 1.23  1997/10/22 11:13:52  zappo
 * Realization of popup changed to be more logical.
 *
 * Revision 1.22  1997/10/18 03:00:45  zappo
 * Added XtNinput to params of top level shell.  Apparently SGI has other
 * ideas than making our shell InputOutput.
 * Fixed Help menu callback.
 *
 * Revision 1.21  1997/10/14 02:13:52  zappo
 * Added the arbitrary message menu item
 *
 * Revision 1.20  1997/10/07 00:10:18  zappo
 * Added new tool menu for the way-cool options
 *
 * Revision 1.19  1997/06/26 23:30:29  zappo
 * Ifdefed out some types of X keys that don't exist everywhere
 *
 * Revision 1.18  1997/03/21 12:10:36  zappo
 * Removed font resizing of the toplevel shell for the widgt specifics.
 *
 * Revision 1.17  1997/03/12 02:28:36  zappo
 * Added new X_iconify fn which is bound to the suspend method. (C-z)
 *
 * Revision 1.16  1997/03/12 01:30:13  zappo
 * Fixed one ANSI to std prototype
 *
 * Revision 1.15  1997/02/23 03:19:11  zappo
 * API change on GT_clean_dev.  Added smarts in callback to remove
 * cancel prompts if active.
 *
 * Revision 1.14  1997/02/20 04:50:27  zappo
 * Added new pixmap icon for systems with Xpm installed.
 *
 * Revision 1.13  1997/02/01 14:28:21  zappo
 * Updated some menu help strings
 *
 * Revision 1.12  1997/01/28  03:22:17  zappo
 * Removed repetative and incorrect prototypes to delwin and xread
 *
 * Revision 1.11  1997/01/26  15:30:59  zappo
 * Changed the Shell name for etalk to use new version number thing.
 *
 * Revision 1.10  1996/07/28  18:03:25  zappo
 * Added a menu item for encryption methods available.
 *
 * Revision 1.9  1996/03/02  03:20:38  zappo
 * Updated so the initial window size always is the same based on the
 * desired font, not pixel size
 *
 * Revision 1.8  1996/02/01  02:33:18  zappo
 * Updated to handle more types of keysyms
 *
 * Revision 1.7  1996/01/20  19:08:51  zappo
 * Updated display code to be cleaner by using a Methods structure.
 *
 * Revision 1.6  1995/12/10  00:17:05  zappo
 * Updated some menu items
 *
 * Revision 1.5  1995/11/21  03:45:29  zappo
 * Updated menus, and also store X device into Ctxt->tty so we can limit
 * access when doing yes_or_no queries.
 *
 * Revision 1.4  1995/09/29  08:40:53  zappo
 * Upgraded width of window, added X_bell
 *
 * Revision 1.3  1995/09/22  14:01:08  zappo
 * Moved menu information into this file
 *
 * Revision 1.2  1995/09/20  23:16:26  zappo
 * Added the functionality.
 *
 * Revision 1.1  1995/09/14  11:33:43  zappo
 * Initial revision
 *
 * Revision 1.1  1995/08/25  16:46:23  zappo
 * Initial revision
 *
 * Tokens: ::Header:: gtalkc.h
 */

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

#include "gtalklib.h"
#include "gtalkc.h"
#include "gtalkx.h"

#include "gnuicon.xbm"
#include "gnuicon.mask"

/* Use X pixmap library if available */
#if HAVE_LIBXPM == 1
#include <X11/xpm.h>
#include "gnuicon.xpm"
#endif

static struct Xcontext XCtxt;	/* X related context information */

static unsigned char Xinputbuffer[100]; /* This is used to fake an IO stream
					   for the UI command parser */

static Widget Popup = NULL;

/* The time of the last mouse event we get. */
Time mouse_time;

static void no_action_here() {};

static void X_close();
static char X_readch();
#ifdef PROTOTYPES
static void X_iconify(struct TalkContext *Ctxt);
static void *X_popup(int width, int height);
static void X_manage_popup(void *win);
static void X_bell(struct TalkContext *Ctxt);
static void X_puttext(void *win, char *text, int len);
static void XW_paste_talk(Widget parent, XEvent *e, String *items, Cardinal *n);
#else
static void X_iconify();
static void *X_popup();
static void X_manage_popup();
static void X_bell();
static void X_puttext();
static void XW_paste_talk();
#endif

static struct IfaceMethods XMethods =
{
  X_close,
  X_bell,
  X_iconify,
  /* Management */
  XW_fix_user_windows,
  XW_clearwin,
  XW_winwidth,
  XW_setlabel,
  /* Editing */
  XW_delchar,
  XW_delline,
  XW_delword,
  XW_fillcolumn,
  X_puttext,
  no_action_here,
  no_action_here,
  /* IO type stuff */
  X_xread,
  X_readch,
  X_popup,
  X_manage_popup,
  X_delwin,
  XW_set_echo_label
};


/*
 * Function: X_init
 *
 *   Attemt to initializes an X connection.  If it fails, return FAIL,
 * otherwise, generate the devices we need to multiplex with X, the
 * application context, and all that other good stuff.
 *
 * Returns:     int  - 
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   9/16/95    Created
 */
struct IfaceMethods *X_init(Ctxt)
     struct TalkContext *Ctxt;
{
  int fake_argc = 0;
  Pixmap icon, mask;
#if HAVE_LIBXPM == 1
  XpmAttributes xpma;
#endif
  static XtActionsRec act[] =
  { { "paste-with-net", (XtActionProc)XW_paste_talk } };

  XCtxt.Ctxt = Ctxt;

  XtToolkitInitialize();	/* get things rolling */

  XCtxt.app_context = XtCreateApplicationContext(); /* get a context */

  XtAppAddActions(XCtxt.app_context, act, sizeof(act)/sizeof(XtActionsRec));

  /* Open the display, and read the database... */
  XCtxt.display = XtOpenDisplay(XCtxt.app_context,
				NULL, /* name of display */
				"GNU talk",
				"Gtalk",
				NULL, 0, /* resource options */
				&fake_argc, NULL);

  /* Lets link in with our IO device routines */
  if(!XCtxt.display)
    return NULL;

  /* Load in the GC we will use in our application */
  XCtxt.gc = DefaultGC(XCtxt.display, DefaultScreen(XCtxt.display));

  XCtxt.X = GTL_x(XCtxt.display);
  Ctxt->tty = XCtxt.X;		/* we need the user input here! */

  /* Setup management hooks so our new device does stuff */
  XCtxt.X->readme  = X_xread;
  XCtxt.X->timeout = 0;
  XCtxt.X->timefn  = NULL;
  XCtxt.X->timemsg = NULL;

#if HAVE_LIBXPM == 1
  icon = None;

  xpma.closeness = 4000;	/* from manual */
  xpma.valuemask = XpmCloseness;

  XpmCreatePixmapFromData(XCtxt.display,
			  XDefaultRootWindow(XCtxt.display),
			  gnuicon_xpm,
			  &icon, NULL,
			  &xpma);

  if(icon == None) {
#endif
    icon = XCreateBitmapFromData(XCtxt.display,
				 XDefaultRootWindow(XCtxt.display),
				 gnuicon_bits, gnuicon_width, gnuicon_height);
#if HAVE_LIBXPM == 1
  }
#endif

  /* Now load in the mask */
  mask = XCreateBitmapFromData(XCtxt.display,
			       XDefaultRootWindow(XCtxt.display),
			       gnuicon_mask_bits, gnuicon_mask_width,
			       gnuicon_mask_height);


  /* Ok, now make a top-level shell */
  XCtxt.topLevel = XtVaAppCreateShell(GTALKX_,
				      "Gtalk",
				      applicationShellWidgetClass,
				      XCtxt.display,
				      XtNallowShellResize, True,
				      /*XtNheight, 200,*/
				      /* Arbitrary width for 80 chars
					 w/ fixed */
				      /*XtNwidth, 300,*/
				      /* don't check, a null means 
					 no icon anyway */
				      XtNiconPixmap, icon,
				      XtNiconMask, mask,
				      /*SGI defaults to something that does
				       * not let the window get focus. */
				      XtNinput, InputOutput,
				      NULL
				      );

  /* build the widget structure... */
  XW_build_mainwindow(&XCtxt);  

  /* Put it all up */
  XtRealizeWidget (XCtxt.topLevel);

  /* Run the read loop for a little bit here.  This will flush our Q's
     and start the realization process */
  X_xread(Ctxt, XCtxt.X);

  return &XMethods;
}


/*
 * Function: X_iconify
 *
 *   A locally defined function which will iconify the etalk window.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   3/11/97    Created
 */
static void X_iconify(Ctxt)
     struct TalkContext *Ctxt;
{
  /* Iconify whatever we have */
  XIconifyWindow(XCtxt.display, XtWindow(XCtxt.topLevel), 
		 DefaultScreen(XCtxt.display));
}

/*
 * Function: X_popup
 *
 *   Intermediate step to create a popup.  May be different depending
 * on the toolkit used.
 *
 * Returns:     Nothing
 * Parameters:  width  - Number of width
 *              height - Number of height
 * History:
 * zappo   9/18/95    Created
 */
static void *X_popup(width, height)
     int width, height;
{
  Popup = XW_popup(&XCtxt, width, height);

  return (void *)Popup;
}

/*
 * Function: X_manage_popup
 *
 *   This is called when a popup has been filled up. It will make it
 * actually appear, and give the text widget a chance to resize itself.
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   9/20/95    Created
 */
static void X_manage_popup(win)
     void *win;
{
  XtManageChild((Widget)win);
  XtRealizeWidget((Widget)win);
}

/*
 * Function: X_delwin
 *
 *   Deletes a widget hierarchy starting with WIN.  For popups, this
 * would be the shell widget.  For talk windows, this would be the
 * panes holding the text and label widgets.
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   9/19/95    Created
 */
void X_delwin(win)
     void *win;
{
  if(win == (void *)Popup)
    {
      XW_kill_popup(&XCtxt);
      Popup = NULL;
    }

  XtUnmanageChild((Widget)win);
  XtDestroyWidget((Widget)win);
}

/*
 * Function: X_readch
 *
 *   Reads from the Xinput buffer created from the X event reader.
 * This is used to simulate a stream environment for the text reading
 * functions.
 *
 * Returns:     unsigned char  - 
 * Parameters:  None
 *
 * History:
 * zappo   9/18/95    Created
 */
static char X_readch()
{
  unsigned char r;
  int i;

  r = Xinputbuffer[0];

  /* Copy everything backwards... This will usually be a 1 character
     string, so lets just see what happens in terms of performance. */
  for(i = 0; Xinputbuffer[i]; Xinputbuffer[i] = Xinputbuffer[i+1], i++);

  return r;
}


/*
 * Function: X_bell
 *
 *   Rings the bell belonging to an X display.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   9/24/95    Created
 */
static void X_bell(Ctxt)
     struct TalkContext *Ctxt;
{
  XBell(XCtxt.display, 100);
}

/*
 * Function: X_close
 *
 *   Cleanly closes the X display
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   9/16/95    Created
 */
static void X_close()
{
  XCloseDisplay(XCtxt.display); /* close the X display nicely */

  XCtxt.X->fd = 1; /* to prevent clean_dev from erroring out */
  GT_clean_dev(XCtxt.X, XCtxt.Ctxt->cleanup);
}

static void X_do_selection(event)
     XSelectionEvent *event;
{
  /* We Called ConvertSelection and this is the generated event.
   * there is only ONE reason, and that is to paste some goodies
   * into our buffer!  Lets do it!
   */
  if(event->property != None)
    {
      unsigned char *selection;
      Atom ret_type;
      int actual_format_return;
      long size, bytes_left;

      XGetWindowProperty(event->display, event->requestor, event->property,
			 0, 0, False, XA_STRING, &ret_type, 
			 &actual_format_return, &size, &bytes_left,
			 &selection);

      /* Do it twice.  The first time just to get the size. ;) */
      XGetWindowProperty(event->display, event->requestor, event->property,
			 0, bytes_left, False, XA_STRING, &ret_type, 
			 &actual_format_return, &size, &bytes_left,
			 &selection);

      /* Cast to int is ok size it won't be more than 1024 */
      DISP_local_string(XCtxt.Ctxt, selection,
			(int)size/(actual_format_return/8));
      /* Is this part true? */
      XFree(selection);
    }
  /* else the request was refused. */
}

/*
 * Function: X_xread
 *
 *   Locally defined function which reads from the X fid and does
 * useful things with it.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              dev  - Pointer to device
 * History:
 * zappo   9/17/95    Created
 */
void X_xread(Ctxt, dev)
     struct TalkContext *Ctxt;	/* NOT USED */
     struct InputDevice *dev;
{
  XEvent event;

  /* Our IO looping is costly, handle X events in big chunks! In
     addition, the pending fn flushes all our X io, as a result, we
     needn't worry about things not being completed nicely. */
  while(XtAppPending(XCtxt.app_context))
    {
      XtAppNextEvent(XCtxt.app_context, &event);
  
      /* We could hang a callback on every text widget, but I feel
	 like cheating since I have this handy location where I can
	 grab all keyboard IO. */
      switch(event.type)
	{
	case KeyPress:
	  /* The following bit of genius was taken from the emacs
	     source tree: xterm.c */
	  {
	    XComposeStatus compose_status;
	    KeySym keysym;
	    int modifiers;
	    int nbytes;
	    char *bufferpos = Xinputbuffer+strlen(Xinputbuffer);

	    modifiers = event.xkey.state;

	    event.xkey.state &= ~ControlMask;
	    nbytes = XLookupString (&event.xkey, bufferpos,
				    80, &keysym, &compose_status);
	    bufferpos[nbytes] = 0; /* terminate it just in case */

	    switch(keysym)
	      {
		/* Read some of the keysyms which might not directly 
		 * translate themselves, and force a translation.  The
		 * previous command *might* handle some of these, but
		 * I just want to make sure there are no mistakes
		 */
	      case XK_BackSpace:
	      case XK_Delete:
#ifdef XK_KP_Delete
	      case XK_KP_Delete:
#endif
		sprintf(bufferpos, "%c", Ctxt->editkeys[0]);
		break;
	      case XK_Clear:
		sprintf(bufferpos, "%c", Ctxt->editkeys[1]);
		break;
		/* Here are some keysyms with no normal mapping.  Translate
		 * them into the keycodes used in etalk to perform some 
		 * useful functions.
		 */
	      case XK_Home:
#ifdef XK_KP_Home
	      case XK_KP_Home:
#endif
		/* M-f = Fix the windows */
		sprintf(bufferpos, "%c", 'f' | 1<<7);
		break;
		/* Let everything else just flow through */
	      case XK_End:
#ifdef XK_KP_End
	      case XK_KP_End:
#endif
		/* M-h = hangup on somebody */
		sprintf(bufferpos, "%c", 'h' | 1<<7);
		break;
	      case XK_Begin:
#ifdef XK_KP_Begin
	      case XK_KP_Begin:
#endif
		/* M-c = call on somebody */
		sprintf(bufferpos, "%c", 'c' | 1<<7);
		break;
	      case XK_Cancel:
	      case XK_Break:
		/* C-g = Abort key in most instances */
		sprintf(bufferpos, "%c", 7);
		break;
	      case XK_Help:
		/* M-H = GUI help */
		sprintf(bufferpos, "%c", 'h' | 1<<7);
		break;
		
	      default:
		if(modifiers & ControlMask)
		  {
		    /* If it's upper case, downshift it! */
		    if((bufferpos[0] > 'A') && (bufferpos[0] < 'Z'))
		      bufferpos[0] -= 'A' - 'a';
		    
		    if(bufferpos[0] == ' ')
		      bufferpos[0] = 0;
		    else
		      /* Now turn it into a control character */
		      /* Ctrl-a = 1, so subtract 'a' and + 1 */
		      bufferpos[0] -= ('a' - 1);
		  }

		/* Now lets turn it into a META command */
		if((modifiers & Mod1Mask) || (modifiers & Mod2Mask))
		  {
		    bufferpos[2] = 0;
		    bufferpos[1] = bufferpos[0];
		    bufferpos[0] = 27;  /* META prefix */
		  }
	      }
#ifdef X_DUMP
	    printf("Read in [%c] total %d chars\n", bufferpos[0], nbytes);
#endif
	    /* this is ok because I terminated the string by hand */
	    while(strlen(Xinputbuffer) > 0)
	      {
		/* CRs are interpreted this way.. */
		if(Xinputbuffer[0] == 13)
		  Xinputbuffer[0] = 10;
		DISP_input(XCtxt.Ctxt, NULL);
	      }
	  }
	  break;
	case SelectionNotify:
	  X_do_selection(&event.xselection);
	  break;	  
	case ButtonPress:
	case ButtonRelease:
	  mouse_time = event.xbutton.time;
	  /* Store the time, and then fall through to dispatching 
	   * this event. */
	default:
	  XtDispatchEvent(&event);
	  break;
	}

    }
}


/*
 * Function: X_puttext
 *
 *   Places LEN characters of TEXT into the window found at WIN.  WIN
 * is void to allow non-X files a hook into the X realm.
 *
 * Returns:     Nothing
 * Parameters:  win  - EmptyPointer to win
 *              text - Pointer toCharacter of text
 *              len  - length of text
 * History:
 * zappo   9/19/95    Created
 */
static void X_puttext(win, text, len)
     void *win;
     char *text;
     int   len;
{
  if(win == Popup)
    {
      /* Add text specifically to the popup text widget... */
      XW_popup_text(text, len);
    }
  else
    {
      XW_add_text(win, text, len);
    }
}

/*
 * Function: *_callback
 *
 *   Locally defined functions which are the callbacks bound to menu items,
 * and key presses found in the widget version of the etalk interface
 *
 * Returns:     static void  - 
 * Parameters:  w         - Widget w
 *              XCtxt     - Context
 *              call_data - 
 * History:
 * zappo   9/17/95    Created
 */
void simulate_key_callback(w, key, call_data)
     Widget w;
     char  *key;
     XtPointer call_data;
{
  /* This prevents spurious outputs */
  if(Popup == NULL) DISP_reset_any_prompts(XCtxt.Ctxt);

  strcpy(Xinputbuffer+strlen(Xinputbuffer), key);

  while(strlen(Xinputbuffer) > 0) /* run new input */
    {
      DISP_suppress_feedback(XCtxt.Ctxt, 1);
      DISP_input(XCtxt.Ctxt, NULL);
      DISP_suppress_feedback(XCtxt.Ctxt, 0);
    }
}     

/*
 * Function: XW_paste_talk
 *
 *   Called by translation table when the user clicks Btn2.  It can be
 * a Down event (prime the paste) or an Up event (paste X selection
 * down the talk line)
 *
 * Returns:     Nothing
 * Parameters:  parent - Widget parent
 *              e      - Pointer to an event
 *              items  - Pointer to passed in items
 *              n      - Pointer to number of items
 * History:
 * zappo   2/1/97     Created
 * zappo   2/1/97     Added parameters parent, e, items, n
 */
static void XW_paste_talk(parent, e, items, n)
     Widget parent;
     XEvent *e;
     String *items;
     Cardinal *n;
{
  static Widget down_click = NULL;

  switch(e->type)
    {
    case ButtonPress:
      /* This is the lock-down event */
      down_click = parent;
      break;
    case ButtonRelease:
      /* This is the up-event, which actually does the pasting. */
      if(down_click == parent)
	{
	  /* Get the selection, and then insert it using DISP calls */
	  Window owner;
	  static Atom my_atom = 0;

	  owner = XGetSelectionOwner(XCtxt.display, XA_PRIMARY);

	  if(owner == None)
	    {
	      DISP_message(NULL, "No selection owner.");
	    }
	  else
	    {
	      /* Now send the selection event... */
	      if(my_atom == 0)
		{
		  my_atom = XInternAtom(XCtxt.display, "_ETALK_TMP_", False);
		}
	      XConvertSelection(XCtxt.display, XA_PRIMARY, XA_STRING, my_atom,
				XtWindow(XCtxt.topLevel), mouse_time);
	      /* We will eventually get a response.  If not, who really cares
	       * at this point...
	       */
	    }
	}
      else
	{
	  /* do nothing. */
	}
      down_click = NULL;
      break;
    }
}
