//*****************************************************************************
//                                Component.cpp                               *
//                               ---------------                              *
// Started     : 14/05/2004                                                   *
// Last Update : 15/04/2010                                                   *
// Copyright   : (C) 2004 by M.S.Waters                                       *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "netlist/Component.hpp"

//*****************************************************************************
// Finish the definition of the ArrayComponent type. The following expands into
// some C++ code and so should only be compiled once (ie. don't put this into a
// header file rather put it in a source file or there will be linker errors).

#include <wx/arrimpl.cpp>
WX_DEFINE_OBJARRAY( ArrayComponent );

//*****************************************************************************
// Constructor.

Component::Component( void ) : wxString( )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

Component::~Component( )
{
}

//*****************************************************************************
// Get the component ports.

void  Component::GetPorts( void )
{
  // Not yet implemented ???
}

//*****************************************************************************
// Test the values against each other.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  Component::bValidate( void )
{
  m_osErrMsg.Empty( );

  if( IsEmpty( ) )
    bSetErrMsg( wxT("component is empty") );
  else if( m_eType<eCPNT_FST || m_eType>eCPNT_LST )
    bSetErrMsg( wxT("component type is invalid") );
  else if( m_osName.IsEmpty( ) )
    bSetErrMsg( wxT("component has no name") );
  else if( m_osaNodes.GetCount( ) < 2 )
    bSetErrMsg( wxT("component has less than two nodes") );
  else if( m_osValue.IsEmpty( ) )
    bSetErrMsg( wxT("component has no value") );

  return( bIsValid( ) );
}

//*****************************************************************************
// Attempt to set the objects error message string attribute.
//
// Argument List :
//   rosErrMsg - A reference to a string containing the new error message
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  Component::bSetErrMsg( const wxString & rosErrMsg )
{
  if( ! bIsValid( ) ) return( FALSE );

  m_osErrMsg = rosErrMsg;

  return( TRUE );
}

//*****************************************************************************
// Compare algorithm for sorting Component objects.
//
// Argument List :
//   ppo1 - The address of a pointer to the first  Component object to compare
//   ppo2 - The address of a pointer to the second Component object to compare
//
// Return Values :
//   >0 - If *(*ppo1) is greater than *(*ppo2)
//   =0 - If *(*ppo1) is equal to     *(*ppo2)
//   <0 - If *(*ppo1) is less    than *(*ppo2)

int  Component::iCompare( Component ** ppo1, Component ** ppo2 )
{
  return( iStrCmp( (const wxString &) **ppo1, (const wxString &) **ppo2 ) );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  Component::bClear( void )
{
  wxString::Empty( );

  m_osErrMsg = wxT("Invalid");

  m_osName  .Clear( );
  m_osaNodes.Clear( );
  m_osValue .Clear( );

  m_eType = eCPNT_NONE;
  m_osaPorts.Clear( );

  return( TRUE );
}

//*****************************************************************************
// Parse a component definition string extracted from a netlist.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  Component::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  size_t             szt1;

  // Clear the object attributes
  os1 = (wxString &) *this;
  bClear( );
  wxString::assign( os1 );
  m_osErrMsg.Empty( );

  // Tokenize the command string
  if( IsEmpty( ) )               return( bValidate( ) );
  ostk1.SetString( os1 );

  // Extract the component name
  bSetName( ostk1.GetNextToken( ) );

  // Extract the nodes the component is connected to
  if( ! ostk1.HasMoreTokens( ) ) return( bValidate( ) );
  bSetNodes( ostk1.GetString( ) );
  for( szt1=0; szt1<m_osaNodes.GetCount( ); szt1++ )
    ostk1.GetNextToken( );

  // Extract the component value
  if( ! ostk1.HasMoreTokens( ) ) return( bValidate( ) );
  bSetValue( ostk1.GetString( ) );

  return( bValidate( ) );
}

//*****************************************************************************
// Format the command string.
//
// Eg. : C1 6 2 100n
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  Component::bFormat( void )
{
  wxString  osCmd;
  size_t    sz1;

  // Format the value part of the component definition
  bFormatValue( );

  // Set the name
  osCmd = m_osName;

  // Append the nodes
  for( sz1=0; sz1<m_osaNodes.GetCount( ); sz1++ )
    osCmd << wxT(' ') << m_osaNodes.Item( sz1 );

  // Append the value
  osCmd << wxT(' ') << m_osValue;

  wxString::assign( osCmd );

  return( bValidate( ) );
}

//*****************************************************************************
// Set the object attributes via a component definition string.
//
// Argument List :
//   rosCmd - A reference to a string containing a component definition
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  Component::bSetString( const wxString & rosCmd )
{
  // Clear the component attributes
  bClear( );

  // Set the underlying string to the raw component definition
  wxString::assign( rosTrim( rosCmd ) );

  // Parse the component definition
  bParse( );

  return( bIsValid( ) );
}

//*****************************************************************************
// Set the component name.
//
// Argument List :
//   rosName - A reference to a string containing a compoent name
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  Component::bSetName( const wxString & rosName )
{
  wxStringTokenizer  ostk1;

  // Tokenize the name string
  ostk1.SetString( rosName );
  if( ostk1.CountTokens( ) <=0 ) return( FALSE );

  // Extract the component name
  m_osName = ostk1.GetNextToken( );

  // Determine the component type and units type
  m_eType = eGetCpntType( m_osName );

  return( TRUE );
}

//*****************************************************************************
// Set the component nodes.
//
// Argument List :
//   rosNodes - A reference to a string containing component nodes eg. "2,1".
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  Component::bSetNodes( const wxString & rosNodes )
{
  wxStringTokenizer  ostk1;
  size_t             szNodeCnt;
  size_t             sz1;

  // Tokenize the name string
  ostk1.SetString( rosNodes, wxT(" \t\r\n,") );
  if( ostk1.CountTokens( ) <=0 )         return( FALSE );

  // Determine how many nodes to look for based on the component type
  switch( m_eType )
  {
    case eCPNT_SCJN   :
    case eCPNT_CAP    :
    case eCPNT_DIODE  :
    case eCPNT_VCVS   :
    case eCPNT_CCCS   :
    case eCPNT_VCCS   :
    case eCPNT_CCVS   :
    case eCPNT_ICS    :
    case eCPNT_CIND   :
    case eCPNT_IND    :
    case eCPNT_RES    :
    case eCPNT_IVS    :
    case eCPNT_ADM    : szNodeCnt = 2;                         break;

    case eCPNT_JFET   :
    case eCPNT_MOS    :
    case eCPNT_BJT    :
    case eCPNT_VCSW   :
    case eCPNT_CCSW   : szNodeCnt = 3;                        break;

    case eCPNT_TLINE  : szNodeCnt = 3;                        break;

    case eCPNT_LOGIC  : szNodeCnt = ostk1.CountTokens( ) - 2; break;

    case eCPNT_SUBCKT : szNodeCnt = ostk1.CountTokens( ) - 1; break;

    default           : szNodeCnt = 0;                        break;
  }
  if( szNodeCnt > ostk1.CountTokens( ) ) return( FALSE );

  // Extract the nodes the component is connected to
  m_osaNodes.Empty( );
  if( szNodeCnt > 0 )
  {
    for( sz1=0; sz1<szNodeCnt; sz1++ )
      m_osaNodes.Add( ostk1.GetNextToken( ) );
  }
  else
  {
    for( sz1=0; sz1<ostk1.CountTokens( ); sz1++ )
    {
      if( isalpha( ostk1.GetString( ).GetChar( 0 ) ) ) break;
      m_osaNodes.Add( ostk1.GetNextToken( ) );
    }
  }

  // Determine the device ports
  GetPorts( );

  return( TRUE );
}

//*****************************************************************************
// Set the component value.
//
// Argument List :
//   rosName - A reference to a string containing a compoent value
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  Component::bSetValue( const wxString & rosValue )
{
  wxString  os1;

  // Check if the string is empty
  os1 = rosTrim( rosValue );
  if( os1.IsEmpty( ) ) return( FALSE );

  // Extract the component value
  m_osValue = os1;

  return( bParseValue( ) );
}

//*****************************************************************************
// Get the component definition string.
//
// Return Values :
//   A reference to the component definition string

const wxString & Component::rosGetString( void )
{
  bFormat( );

  return( (wxString &) *this );
}

//*****************************************************************************
// Get the component nodes as a string separated by a comma.
//
// Return Values :
//   A reference to the component nodes string

const wxString & Component::rosGetNodes( void )
{
  static  wxString  osNodes;
  size_t  sz1;

  osNodes.Empty( );

  if( ! m_osaNodes.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_osaNodes.GetCount( ); sz1++ )
    {
      if( ! osNodes.IsEmpty( ) ) osNodes << wxT(",");
      osNodes << m_osaNodes.Item( sz1 );
    }
  }

  return( osNodes );
}

//*****************************************************************************
// Get the value as a mantissa and exponent if possible.

const wxString & Component::rosGetNumValue( void )
{
  static  wxString   osNumValue;
  wxStringTokenizer  ostk1;
  wxString           os1;

  // Clear the local static variable
  osNumValue.Empty( );

  // Tokenize the value string
  ostk1 = m_osValue;

  // Attempt to parse a field in the value string as a numeric quantity
  while( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( );
    osNumValue = ConvertType::rosGetNum( os1 );
    if( ! osNumValue.IsEmpty( ) ) break;
  }

  return( osNumValue );
}

//*****************************************************************************
// Get the component type.
//
// Argument List :
//   rosName - A component name
//
// Return Values :
//   An enumerate type indicating the type of the component

eCpntType  Component::eGetCpntType( const wxString & rosName )
{
  // Determine the component type from the first character of the name
  switch( toupper( rosName.GetChar( 0 ) ) )
  {
    case wxT('B') : return( eCPNT_SCJN   );
    case wxT('C') : return( eCPNT_CAP    );
    case wxT('D') : return( eCPNT_DIODE  );
    case wxT('E') : return( eCPNT_VCVS   );
    case wxT('F') : return( eCPNT_CCCS   );
    case wxT('G') : return( eCPNT_VCCS   );
    case wxT('H') : return( eCPNT_CCVS   );
    case wxT('I') : return( eCPNT_ICS    );
    case wxT('J') : return( eCPNT_JFET   );
    case wxT('K') : return( eCPNT_CIND   );
    case wxT('L') : return( eCPNT_IND    );
    case wxT('M') : return( eCPNT_MOS    );
    case wxT('Q') : return( eCPNT_BJT    );
    case wxT('R') : return( eCPNT_RES    );
    case wxT('S') : return( eCPNT_VCSW   );
    case wxT('T') : return( eCPNT_TLINE  );
    case wxT('U') : return( eCPNT_LOGIC  );
    case wxT('V') : return( eCPNT_IVS    );
    case wxT('W') : return( eCPNT_CCSW   );
    case wxT('X') : return( eCPNT_SUBCKT );
    case wxT('Y') : return( eCPNT_ADM    );

    default       : return( eCPNT_NONE   );
  }
}

//*****************************************************************************
// Get the units type (where appropriate).
//
// Argument List :
//   rosName - A component name
//
// Return Values :
//   An enumerate type indicating the type of units associated with this
//   component

eUnitsType  Component::eGetUnitsType( const wxString & rosName )
{
  // Determine the units type from the first character of the name
  switch( toupper( rosName.GetChar( 0 ) ) )
  {
    case wxT('B') : return( eUNITS_SCLR );
    case wxT('C') : return( eUNITS_CAP  );
    case wxT('D') : return( eUNITS_NONE );
    case wxT('E') : return( eUNITS_SCLR );
    case wxT('F') : return( eUNITS_SCLR );
    case wxT('G') : return( eUNITS_SCLR );
    case wxT('H') : return( eUNITS_SCLR );
    case wxT('I') : return( eUNITS_CURR );
    case wxT('J') : return( eUNITS_NONE );
    case wxT('K') : return( eUNITS_IND  );
    case wxT('L') : return( eUNITS_IND  );
    case wxT('M') : return( eUNITS_NONE );
    case wxT('Q') : return( eUNITS_NONE );
    case wxT('R') : return( eUNITS_RES  );
    case wxT('S') : return( eUNITS_NONE );
    case wxT('T') : return( eUNITS_NONE );
    case wxT('U') : return( eUNITS_NONE );
    case wxT('V') : return( eUNITS_VOLT );
    case wxT('W') : return( eUNITS_NONE );
    case wxT('X') : return( eUNITS_NONE );
    case wxT('Y') : return( eUNITS_RES  );

    default       : return( eUNITS_NONE );
  }
}

//*****************************************************************************
// Trim a component definition string by replacing all field separtors with a
// single space character and removing any line terminators.
//
// Argument List :
//   rosDefn - A reference to a string containing a component definition
//
// Return Values :
//   A reference to a string containing a trimmed copy of the component defn

const wxString & Component::rosTrim( const wxString & rosDefn )
{
  static  wxString   osDefn;
  wxStringTokenizer  ostk1;

  // Tokenize the component definition
  ostk1.SetString( rosDefn, wxT(" \t\r\n"), wxTOKEN_STRTOK );

  // Reconstruct the component definition
  osDefn = ostk1.GetNextToken( );
  while( ostk1.HasMoreTokens( ) )
    osDefn << wxT(' ') << ostk1.GetNextToken( );

  return( osDefn );
}

//*****************************************************************************
// Copy the contents of a wxString object.
//
// Argument List :
//   rosCmd - A reference to a string containing a component definition
//
// Return Values :
//   A reference to this object

wxString & Component::operator = ( const wxString & rosCmd )
{
  bSetString( rosCmd );

  return( *((wxString *) this) );
}

//*****************************************************************************
// Copy the contents of another Component object.
//
// Argument List :
//   roCpnt - A reference to a Component object
//
// Return Values :
//   A reference to this object

Component & Component::operator = ( const Component & roCpnt )
{
  (wxString &) *this = (wxString &) roCpnt;

  m_eType    = roCpnt.m_eType;

  m_osName   = roCpnt.m_osName;
  m_osaNodes = roCpnt.m_osaNodes;
  m_osValue  = roCpnt.m_osValue;

  m_osaPorts = roCpnt.m_osaPorts;

  m_osErrMsg = roCpnt.m_osErrMsg;

  return( *this );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  Component::Print( const wxString & rosPrefix )
{
  wxString  osPrefix;
  size_t    sz1;

  osPrefix = rosPrefix + wxT( "Component::" );

  cout << osPrefix  .mb_str( ) << "wxString::"
       << this     ->mb_str( ) << '\n';

  cout << osPrefix  .mb_str( ) << "m_osErrMsg       : "
       << m_osErrMsg.mb_str( ) << '\n';

  cout << osPrefix  .mb_str( ) << "m_eType          : ";
  switch( m_eType )
  {
    case eCPNT_SCJN   : cout << "eCPNT_SCJN";   break;
    case eCPNT_ADM    : cout << "eCPNT_ADM";    break;
    case eCPNT_BJT    : cout << "eCPNT_BJT";    break;
    case eCPNT_CAP    : cout << "eCPNT_CAP";    break;
    case eCPNT_CCCS   : cout << "eCPNT_CCCS";   break;
    case eCPNT_CCSW   : cout << "eCPNT_CCSW";   break;
    case eCPNT_CCVS   : cout << "eCPNT_CCVS";   break;
    case eCPNT_CIND   : cout << "eCPNT_CIND";   break;
    case eCPNT_DIODE  : cout << "eCPNT_DIODE";  break;
    case eCPNT_ICS    : cout << "eCPNT_ICS";    break;
    case eCPNT_IND    : cout << "eCPNT_IND";    break;
    case eCPNT_IVS    : cout << "eCPNT_IVS";    break;
    case eCPNT_JFET   : cout << "eCPNT_JFET";   break;
    case eCPNT_LOGIC  : cout << "eCPNT_LOGIC";  break;
    case eCPNT_MOS    : cout << "eCPNT_MOS";    break;
    case eCPNT_RES    : cout << "eCPNT_RES";    break;
    case eCPNT_SUBCKT : cout << "eCPNT_SUBCKT"; break;
    case eCPNT_TLINE  : cout << "eCPNT_TLINE";  break;
    case eCPNT_VCCS   : cout << "eCPNT_VCCS";   break;
    case eCPNT_VCSW   : cout << "eCPNT_VCSW";   break;
    case eCPNT_VCVS   : cout << "eCPNT_VCVS";   break;
    case eCPNT_NONE   : cout << "eCPNT_NONE";   break;

    default           : cout << "Invalid";
  }
  cout << '\n';

  cout << osPrefix  .mb_str( ) << "m_osName         : "
       << m_osName  .mb_str( ) << '\n';

  cout << osPrefix  .mb_str( ) << "m_osaNodes[ ]    : ";
  for( sz1=0; sz1<m_osaNodes.GetCount( ); sz1++ )
  {
    if( sz1 > 0 ) cout << ", ";
    cout << m_osaNodes.Item( sz1 ).mb_str( );
  }
  cout << '\n';

  cout << osPrefix  .mb_str( ) << "m_osaPorts[ ]    : ";
  for( sz1=0; sz1<m_osaPorts.GetCount( ); sz1++ )
  {
    if( sz1 > 0 ) cout << " ";
    cout << m_osaPorts.Item( sz1 ).mb_str( );
  }
  cout << '\n';

  cout << osPrefix  .mb_str( ) << "m_osValue        : "
       << m_osValue .mb_str( ) << '\n';

  cout << osPrefix  .mb_str( ) << "eGetUnitsType( ) : ";
  switch( eGetUnitsType( m_osName ) )
  {
    case eUNITS_CAP   : cout << "eUNITS_CAP";   break;
    case eUNITS_IND   : cout << "eUNITS_IND";   break;
    case eUNITS_RES   : cout << "eUNITS_RES";   break;
    case eUNITS_VOLT  : cout << "eUNITS_VOLT";  break;
    case eUNITS_CURR  : cout << "eUNITS_CURR";  break;
    case eUNITS_TIME  : cout << "eUNITS_TIME";  break;
    case eUNITS_FREQ  : cout << "eUNITS_FREQ";  break;
    case eUNITS_PHASE : cout << "eUNITS_PHASE"; break;
    case eUNITS_TEMP  : cout << "eUNITS_TEMP";  break;
    case eUNITS_SCLR  : cout << "eUNITS_SCLR";  break;
    case eUNITS_NONE  : cout << "eUNITS_NONE";  break;

    default           : cout << "Invalid";      break;
  }
  cout << '\n';
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_COMPONENT

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCpnt;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )                    { Usage( argv[0] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1.at( 0 ) == wxT('-') )
    {
      if( os1.at( 1 ) == wxT('h') ) { Usage( argv[0] ); exit( EXIT_SUCCESS ); }
      else                          { Usage( argv[0] ); exit( EXIT_FAILURE ); }
    }
  }

  // Display the utility banner
  cout << "\n  Component Structure Test Utility"
       << "\n     Version 1.07 (08/05/2009)\n";

  // Create a GNU-CAP OP command object
  Component  tCpnt;

  // Use the following command example to check the formatter and the parser :
  osCpnt = wxT("Vin	 2	0	 AC	1V"); // This contains TABs and SPACEs
  cout << "\nCpnt defn to test : "
       << osCpnt.mb_str( ) << "\n\n";

  // ******************** Test the formatter ********************
  cout << "Test the component formatter :";

  // Set things up for a formatter test
  tCpnt.bClear( );
  tCpnt.bSetName ( wxT("Vin")   );
  tCpnt.bSetNodes( wxT("2,0")   );
  tCpnt.bSetValue( wxT("AC 1V") );

  // Run the formatter
  cout << "\n  Run formatter  : ";
  if( tCpnt.bFormat( ) )
       cout << "OK";
  else cout << "FAULT (" << tCpnt.rosGetErrMsg( ).mb_str( ) << ')';

  // Test the formatter output
  cout << "\n  Test format    : ";
  if( (wxString) tCpnt == Component::rosTrim( osCpnt ) )
    cout << "OK\n" << "  Component defn : "
         << tCpnt.rosGetString( ).mb_str( ) << '\n';
  else
  { // There's been an error so print the component contents
    cout << "FAULT\n";
    cout << "  tCpnt Contents : \n";
    tCpnt.Print( wxT("    ") );
  }

  cout << "\n";

  // ********************* Test the parser *********************
  cout << "Test the component parser :";

  // Set things up for a parser test
  tCpnt.bClear( );
  tCpnt = osCpnt;

  // Run the parser
  cout << "\n  Run parser     : ";
  if( tCpnt.bParse( ) )
       cout << "OK";
  else cout << "FAULT (" << tCpnt.rosGetErrMsg( ).mb_str( ) << ')';

  // Test the parser output
  tCpnt.bFormat( );
  cout << "\n  Test format    : ";
  if( (wxString) tCpnt == Component::rosTrim( osCpnt ) )
    cout << "OK\n" << "  Component defn : "
         << tCpnt.rosGetString( ).mb_str( ) << '\n';
  else
  { // There's been an error so print the component contents
    cout << "FAULT\n";
    cout << "  tCpnt Contents : \n";
    tCpnt.Print( wxT("    ") );
  }

  cout << "\n";

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_COMPONENT

//*****************************************************************************
