#!/usr/bin/perl -w

#     process-chain - A complete process chain on a single image in tif/bmp 
#     format using gpiv tools resulting into estimated PIV velocities
#     or its derivatives.
#

#   Copyright (C) 2002 Gerber van der Graaf

#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2, or (at your option)
#   any later version.

#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.

#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software Foundation,
#   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#--------------------------------------------------------------------

$VERSION = q$Id: process-chain.pl,v 1.6 2006/03/04 12:37:08 gerber Exp $;
$HELP = "Processes a pipe of Gpiv-tool command's";
$USAGE = "gpiv_process-chain [-h|-help] [-n|-none] [-af] [-c|-clean] [-fik]
[-p|-print] [proc_*] [-t] [-v|-version] [-verbose] file_name

keys:
-af string:         append string to file-base name
-c:                 removes raw image data and header
-fik:               use fi-keyline for filtering gpiv parms from README
-h:                 this on-line help
-n:                 suppresses real execution
-p:                 prints process commands to stdout
-pf string          prepend string to file-base name
-proc_*             defines processes to be included in the chain (valid, 
		    scale, manipiv, flipx, flipy, revert, rot90, rot180, 
		    and vorty or nstrain or sstrain. Vorty, nstrain and sstrain
                    not in combination with -t hdf.
-t:                 image type or format: hdf (.gpi), dav (.IMG) or a type as 
                    defined by ImageMagic\'s convert. Default is raw binary 
                    image (.r).
-v:                 prints version
-verbose            more verbose. Without -proc_*, interrogation parameters are
                    printed to the output.
";

#----------------- Command line arguments handling ----------
##$opt_a = 0;
$opt_c = 0;
$opt_fik = 0;
$opt_h = 0;
$opt_n = 0;
$opt_p = 0;
$opt_v = 0;
$opt_verbose = 0;

$opt_proc_valid = 0;
$opt_proc_scale = 0;
$opt_proc_manipiv = 0;
$opt_proc_flipx = 0;
$opt_proc_flipy = 0;
$opt_proc_revert = 0;
$opt_proc_rot90 = 0;
$opt_proc_rot180 = 0;
$opt_proc_vorty = 0;
$opt_proc_nstrain = 0;
$opt_proc_sstrain = 0;

#$post_spatial_scale = 0.0; # [mm/px]
#$post_time_scale = .0;        # [s]
#$post_zero_offx = 0.0;       # [mm]
#$post_zero_offy = 0.0;          # [mm]

use Getopt::Long;
##$result = 
GetOptions('h|help', 'n|none', 'p|print', 'v|version', 'verbose',
'c|clean',
'af=s'  => \$append_fname,
't=s'  => \$img_format,
'pf=s'  => \$prepend_fname,
'fik',
'proc_valid',
'proc_scale',
'proc_manipiv',
'proc_flipx',
'proc_flipy',
'proc_revert',
'proc_rot90',
'proc_rot180',
'proc_vorty',
'proc_nstrain',
'proc_sstrain'
);


if ($opt_h) {
  print ("$HELP\n");
  print ("$USAGE\n");
  exit;
}
if ($opt_v) {
  print ("$VERSION\n");
  exit;
}

#if (!$img_format) {
#  $img_format = "bmp";
#}

$file_name = shift (@ARGV);

if ($#ARGV != -1) {
  printf ("\nUsage: $USAGE\n");
  exit;
}

if ($opt_proc_vorty && $opt_proc_nstrain 
   || $opt_proc_vorty && $opt_proc_sstrain 
   || $opt_proc_sstrain && $opt_proc_nstrain) {
    print ("Only one of the processes proc_vorty, proc_sstrain or proc_nstrain may be enabled\n");
    exit;
}

if (($img_format && $img_format =~ "hdf") && $opt_proc_vorty
   ||($img_format && $img_format =~ "hdf") && $opt_proc_sstrain 
   ||($img_format && $img_format =~ "hdf") && $opt_proc_nstrain) {
    print ("proc_vorty, proc_sstrain or proc_nstrain may no be enabled in combination with -t hdf\n");
    exit;
}

# ----------------- Default pars and Initial variables
if ($append_fname) {
    $out_basefile = $file_name.$append_fname;
}

if ($prepend_fname) {
    @words = split(/\//, $file_name);      #splitting line $_ up in words
#printf("nwords = $#words\n");
    for ($i=0; $i <= $#words; $i = $i+1) {
#printf("words[$i] = $words[$i]\n");
    }
    $words[$#words] = $prepend_fname.$words[$#words];
#printf("words[$#words] = $words[$#words]\n");
    $out_basefile = join("/", @words);
#$prepend_fname.$file_name; 
#printf("out_basefile = $out_basefile\n");
}

if ($append_fname || $prepend_fname) {
    if ($opt_proc_vorty) {
        $out_file = $out_basefile.".vor";
        $parfile = $out_basefile.".par";
    } elsif ($opt_proc_nstrain) {
        $out_file = $out_basefile.".nstr";
        $parfile = $out_basefile.".par";
    } elsif ($opt_proc_sstrain) {
        $out_file = $out_basefile.".sstr";
        $parfile = $out_basefile.".par";
    } else {
        $out_file = $out_basefile.".piv";
        $parfile = $out_basefile.".par";
    }
} else {
    if ($opt_proc_vorty) {
        $out_file = $file_name.".vor";
        $parfile = $file_name.".par";
    } elsif ($opt_proc_nstrain) {
        $out_file = $file_name.".nstr";
        $parfile = $file_name.".par";
    } elsif ($opt_proc_sstrain) {
        $out_file = $file_name.".sstr";
        $parfile = $file_name.".par";
    } else {
        $out_file = $file_name.".piv";
        $parfile = $file_name.".par";
    }
}


if ($img_format && $img_format =~ "hdf") {
    if ($opt_p || $opt_n) {printf ("out_file = $file_name.gpi\n");}
} elsif ($img_format && $img_format =~ "dav") {
    if ($opt_p || $opt_n) {printf ("out_file = $file_name.IMG\n");}
} else {
    if ($opt_p || $opt_n) {printf ("out_file = $out_file\n");}
}



# ----------------- gpivtools image conversion program names
$hdf2piv="hdf2piv_d";
$dav2piv="dav2piv_d";
$img2gpiv="img2gpiv_d";
$fi_keyline="fi-keyline_d";

# ----------------- Image processing
if ($img_format && $img_format =~ "hdf") {
    if ($opt_verbose) {
        @args=("$hdf2piv -e -p $file_name");
    } else {
        @args=("$hdf2piv -e $file_name");
    }
    if ($opt_p || $opt_n) {printf ("@args \n");}
    if (!$opt_n) {system (@args) == 0 || die "system @args failed: $?";}
    $opt_c = 1;

} elsif ($img_format && $img_format =~ "dav") {
    if ($opt_verbose) {
        @args=("$dav2piv -p $file_name");
    } else {
        @args=("$dav2piv $file_name");
    }
    if ($opt_p || $opt_n) {printf ("@args \n");}
    if (!$opt_n) {system (@args) == 0 || die "system @args failed: $?";}
    $opt_c = 1;

} elsif ($img_format) {
        @args=("$img2gpiv -t $img_format $file_name");
    if ($opt_p || $opt_n) {printf ("@args \n");}
    if (!$opt_n) {system (@args) == 0 || die "system @args failed: $?";}
    $opt_c = 1;
}


# ----------------- Filter parameters from README
if ($opt_fik) {
    @words = split(/\//, $file_name);      #splitting line $_ up in words
    $keyword = $words[$#words - 1];
    $keyword = $keyword.">>";
    @args=("$fi_keyline -if README -of gpivrc \"$keyword\"");
    if ($opt_p || $opt_n) {printf ("@args \n");}
    if (!$opt_n) {system (@args) == 0 || die "system @args failed: $?";}
}

# ----------------- Creating chain and execute
# $out_file
printf ("opt_verbose = $opt_verbose\n");
if ($opt_verbose
    && !$opt_proc_valid
    && !$opt_proc_scale
    && !$opt_proc_manipiv
    && !$opt_proc_flipx
    && !$opt_proc_flipy
    && !$opt_proc_revert
    && !$opt_proc_rot90
    && !$opt_proc_rot180
    && !$opt_proc_vorty
    && !$opt_proc_nstrain
    && !$opt_proc_sstrain
    ) {
    $CMD = "rr_d -p < $file_name.r ";
} else {
    $CMD = "rr_d < $file_name.r ";
}

if ($opt_proc_valid) {$CMD = $CMD." | errvec_d ";}
if ($opt_proc_scale) {$CMD = $CMD."| scale_d ";}
if ($opt_proc_manipiv) {$CMD = $CMD."| manipiv_d ";}
if ($opt_proc_flipx) {$CMD = $CMD."| flipx_d ";}
if ($opt_proc_flipy) {$CMD = $CMD."| flipy_d ";}
if ($opt_proc_revert) {$CMD = $CMD."| revert_d ";}
if ($opt_proc_rot90) {$CMD = $CMD."| rot90_d ";}
if ($opt_proc_rot180) {$CMD = $CMD."| rot180_d ";}
if ($opt_proc_vorty) {$CMD = $CMD."| vorty_d ";}
if ($opt_proc_nstrain) {$CMD = $CMD."| nstrain_d ";}
if ($opt_proc_sstrain) {$CMD = $CMD."| sstrain_d ";}

$CMD = $CMD." > $out_file";

#@args=("$eval  < $file_name.r  | $valid | $post1 | $post2 > $out_file");
@args=($CMD);

if ($opt_p || $opt_n) {printf ("@args \n");}
if (!$opt_n) {system (@args) == 0 || die "system @args failed: $?";}


# ----------------- Save the pararameters
#  @args=("cp gpivrc $parfile");
#  if ($opt_p || $opt_n) {printf ("@args \n");}
#  if (!$opt_n) {system (@args) == 0 || die "system @args failed: $?";}

if ($img_format && $img_format =~ "hdf") {
    if ($opt_verbose) {
        @args=("piv2hdf -i -p $file_name");
    } else {
        @args=("piv2hdf -i $file_name");
    }
    if ($opt_p || $opt_n) {printf ("@args \n");}
    if (!$opt_n) {system (@args) == 0 || die "system @args failed: $?";}
}


# ----------------- Cleaning up the mesh
if ($opt_c) {
    @args=("rm $file_name.r $file_name.h");
    if ($opt_p || $opt_n) {printf ("@args \n");}
    if (!$opt_n) {system (@args) == 0 || die "system @args failed: $?";}

#    @args=("rm $file_name.piv");
#    if ($opt_p || $opt_n) {printf ("@args \n");}
#    if (!$opt_n) {system (@args);}
}

#
# that's all folks!
#
