/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*----------------------------------------------------------------------
 
  gpiv - Graphic program for Particle Image Velocimetry, based on gtk/gnome
          libraries.

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of gpiv.

   Gpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

-----------------------------------------------------------------------------*/

/*
 * (callback) functions for Piv post processing window/tabulator
 * $Log: pivpost.c,v $
 * Revision 1.13  2006/01/31 14:28:12  gerber
 * version 0.3.0
 *
 * Revision 1.11  2005/02/26 09:43:30  gerber
 * parameter flags (parameter_logic) defined as gboolean
 *
 * Revision 1.10  2004/10/15 19:24:05  gerber
 * GPIV_ and Gpiv prefix to defines and structure names of libgpiv
 *
 * Revision 1.9  2004/06/14 21:19:23  gerber
 * Image depth up to 16 bits.
 * Improvement "single int" and "drag int" in Eval tab.
 * Viewer's pop-up menu.
 * Adaption for gpiv_matrix_* and gpiv_vector_*.
 * Resizing console.
 * See Changelog for further info.
 *
 * Revision 1.8  2003/07/13 14:38:18  gerber
 * changed error handling of libgpiv
 *
 * Revision 1.7  2003/07/12 21:21:16  gerber
 * changed error handling libgpiv
 *
 * Revision 1.5  2003/07/10 11:56:07  gerber
 * added man page
 *
 * Revision 1.4  2003/07/04 10:47:01  gerber
 * cleaning up
 *
 * Revision 1.3  2003/07/03 17:08:02  gerber
 * display ruler adjusted for scaled data
 *
 * Revision 1.2  2003/06/27 13:47:26  gerber
 * display ruler, line/point evaluation
 *
 * Revision 1.1.1.1  2003/06/17 17:10:52  gerber
 * Imported gpiv
 *
 */


#include "gpiv_gui.h"
#include "utils.h"
#include "pivpost.h"
#include "display.h"


/*
 * Public post-processing functions
 */

void 
exec_scale(PivPost * pivpost
           )
/*-----------------------------------------------------------------------------
 */
{

    if (display_act->gpd.scaled_piv) 
        gpiv_free_pivdata(&display_act->gpd.piv_data_scaled);

    display_act->gpd.piv_data_scaled.nx = display_act->gpd.piv_data.nx;
    display_act->gpd.piv_data_scaled.ny = display_act->gpd.piv_data.ny;
    gpiv_alloc_pivdata(&display_act->gpd.piv_data_scaled);

    if(display_act->gpd.exist_piv && !cancel_process) {
        if ((display_act->gpd.scaled_piv = 
             gpiv_post_scale (display_act->gpd.piv_data, 
                              &display_act->gpd.piv_data_scaled, 
                              display_act->img.image_par)) != 1) {
            error_gpiv(_("exec_scale: Failure calling gpiv_post_scale"));
        }
        
/*
 * Adjusting display rulers
 * BUGFIX , TODO
 */
        gtk_ruler_set_range(GTK_RULER(display_act->hruler),
                            (gfloat) image_par.z_off_x *10e2,
                            (gfloat) gpiv_par.img_width 
                            * image_par.s_scale *10e2 - 1.0
                            + (gfloat) image_par.z_off_x *10e2,
                            (gfloat) image_par.z_off_x *10e2,
                            (gfloat) gpiv_par.img_width 
                            * image_par.s_scale *10e2 - 1.0
                            + (gfloat) image_par.z_off_x *10e2);

        gtk_ruler_set_range(GTK_RULER(display_act->vruler),
                            (gfloat) image_par.z_off_y *10e2,
                            (gfloat) gpiv_par.img_height 
                            * image_par.s_scale *10e2 - 1.0
                            + (gfloat) image_par.z_off_y *10e2,
                            (gfloat) image_par.z_off_y *10e2,
                            (gfloat) gpiv_par.img_height 
                            * image_par.s_scale *10e2 - 1.0
                            + (gfloat) image_par.z_off_y *10e2);

        
        free_post_bufmems(display_act);
    } else {
        warning_gpiv(_("no PIV data"));
    }

}



void 
exec_savg(PivPost * pivpost
          )
/*-----------------------------------------------------------------------------
 */
{
    gint ndata = 0;
    display_act->gpd.averaged_piv = TRUE;

    display_act->gpd.piv_data.mean_dx = 0.0;
    display_act->gpd.piv_data.sdev_dx = 0.0; 
    display_act->gpd.piv_data.min_dx = 0.0;
    display_act->gpd.piv_data.max_dx = 0.0;

    display_act->gpd.piv_data.mean_dy = 0.0;
    display_act->gpd.piv_data.sdev_dy = 0.0; 
    display_act->gpd.piv_data.min_dy = 0.0;
    display_act->gpd.piv_data.max_dy = 0.0;


    display_act->gpd.piv_data_scaled.mean_dx = 0.0;
    display_act->gpd.piv_data_scaled.sdev_dx = 0.0; 
    display_act->gpd.piv_data_scaled.min_dx = 0.0;
    display_act->gpd.piv_data_scaled.max_dx = 0.0;

    display_act->gpd.piv_data_scaled.mean_dy = 0.0;
    display_act->gpd.piv_data_scaled.sdev_dy = 0.0; 
    display_act->gpd.piv_data_scaled.min_dy = 0.0;
    display_act->gpd.piv_data_scaled.max_dy = 0.0;



    if (display_act->gpd.exist_piv && !cancel_process) {
        exec_process = TRUE;
        piv_post_par.subtract = 0;

        if (display_act->gpd.scaled_piv) {
            gpiv_post_savg(display_act->gpd.piv_data_scaled, 
                           &display_act->gpd.piv_data_scaled, 
                           piv_post_par, 
                           &ndata);

/*             g_warning("exec_savg:: SCALED: mean_dx = %f sdev_dx = %f min_dx = %f max_dx = %f",  */
/*                    display_act->gpd.piv_data_scaled.mean_dx,  */
/*                       display_act->gpd.piv_data_scaled.sdev_dx,  */
/*                       display_act->gpd.piv_data_scaled.min_dx,   */
/*                       display_act->gpd.piv_data_scaled.max_dx); */
/*             g_warning("exec_savg:: SCALED: mean_dy = %f sdev_dy = %f min_dy = %f max_dy = %f\n",  */
/*                    display_act->gpd.piv_data_scaled.mean_dy, */
/*                       display_act->gpd.piv_data_scaled.sdev_dy,  */
/*                       display_act->gpd.piv_data_scaled.min_dy,  */
/*                       display_act->gpd.piv_data_scaled.max_dy); */
        

            gtk_spin_button_set_value (GTK_SPIN_BUTTON
                                       (pivpost->spinbutton_suavg), 
                                       (gfloat) display_act->
                                       gpd.piv_data_scaled.mean_dx);
            gtk_spin_button_set_value (GTK_SPIN_BUTTON
                                       (pivpost->spinbutton_svavg), 
                                       (gfloat) display_act->
                                       gpd.piv_data_scaled.mean_dy);

        } else {
            gpiv_post_savg(display_act->gpd.piv_data, 
                           &display_act->gpd.piv_data, 
                           piv_post_par, 
                           &ndata);
            
/*             g_warning("exec_savg:: mean_dx = %f sdev_dx = %f min_dx = %f max_dx = %f",  */
/*                       display_act->gpd.piv_data.mean_dx, display_act->gpd.piv_data.sdev_dx, display_act->gpd.piv_data.min_dx,   */
/*                       display_act->gpd.piv_data.max_dx); */
/*             g_warning("exec_savg:: mean_dy = %f sdev_dy = %f min_dy = %f max_dy = %f\n",  */
/*                       display_act->gpd.piv_data.mean_dy, display_act->gpd.piv_data.sdev_dy, display_act->gpd.piv_data.min_dy,   */
/*                       display_act->gpd.piv_data.max_dy); */
            
            
            gtk_spin_button_set_value (GTK_SPIN_BUTTON
                                       (pivpost->spinbutton_suavg), 
                                       (gfloat) display_act->
                                       gpd.piv_data.mean_dx);
            gtk_spin_button_set_value (GTK_SPIN_BUTTON
                                       (pivpost->spinbutton_svavg), 
                                       (gfloat) display_act->
                                       gpd.piv_data.mean_dy);
            
        }

        exec_process = FALSE;
    } else {
        warning_gpiv(_("no PIV data"));
    }
}



void 
exec_addavg(PivPost * pivpost
            )
/*-----------------------------------------------------------------------------
 * Adds average or offsets for displacements or velocities
 */
{
    gint i, j;
    GpivPivData out_data;

    out_data.point_x = NULL;
    out_data.point_y = NULL;
    out_data.dx = NULL;
    out_data.dy = NULL;
    out_data.snr = NULL;
    out_data.peak_no = NULL;

    if  (display_act->gpd.exist_piv && !cancel_process) {
        exec_process = TRUE;
        piv_post_par.subtract = 1;
        out_data.nx = display_act->gpd.piv_data.nx;
        out_data.ny = display_act->gpd.piv_data.ny;
        gpiv_alloc_pivdata(&out_data);

        if (display_act->gpd.scaled_piv) {

            if (display_act->gpd.averaged_piv) {
            gpiv_post_subtract_dxdy(display_act->gpd.piv_data_scaled, 
                                    &out_data, 
                                    - display_act->gpd.piv_data_scaled.mean_dx, 
                                    - display_act->gpd.piv_data_scaled.mean_dy);

/*
 * Copy parameters in Buffer structure for saving and, eventual,
 * later use
 */
            display_act->gpd.piv_post_par.z_off_dx = 
                - display_act->gpd.piv_data_scaled.mean_dx;
            display_act->gpd.piv_post_par.z_off_dy = 
                - display_act->gpd.piv_data_scaled.mean_dy;

            } else {
            gpiv_post_subtract_dxdy(display_act->gpd.piv_data_scaled, 
                                    &out_data, 
                                    - piv_post_par.z_off_dx, 
                                    - piv_post_par.z_off_dx);
            }

            for (i = 0; i < display_act->gpd.piv_data.ny; i++) {
                for (j = 0; j < display_act->gpd.piv_data.nx; j++) {
                    display_act->gpd.piv_data_scaled.dx[i][j] = 
                        out_data.dx[i][j];
                    display_act->gpd.piv_data_scaled.dy[i][j] = 
                        out_data.dy[i][j];
                }
            }


/*
 * Subtract from piv_data, only for displaying purposes
 */
            if (display_act->gpd.averaged_piv) {
                gpiv_post_subtract_dxdy(display_act->gpd.piv_data, 
                                        &out_data, 
                                        - display_act->gpd.piv_data.mean_dx, 
                                        - display_act->gpd.piv_data.mean_dy);
            } else {
/*
 * apply inverse scale factor for offsetting of partimcle image displacements
 */
                gpiv_post_subtract_dxdy(display_act->gpd.piv_data, 
                                        &out_data, 
                                        - piv_post_par.z_off_dx
                                        / image_par.s_scale 
                                        * image_par.t_scale, 
                                        - piv_post_par.z_off_dy
                                        / image_par.s_scale 
                                        * image_par.t_scale
                                        );
            }

            for (i = 0; i < display_act->gpd.piv_data.ny; i++) {
                for (j = 0; j < display_act->gpd.piv_data.nx; j++) {
                    display_act->gpd.piv_data.dx[i][j] = out_data.dx[i][j];
                    display_act->gpd.piv_data.dy[i][j] = out_data.dy[i][j];
                }
            }
            update_all_vectors(&display_act->gpd);



/*
 * Only apply offseting for non-scaled pivdata with values from entry 
 * or z_off_dx and z_off_dy
 */
        } else {
            if (display_act->gpd.averaged_piv) {
            gpiv_post_subtract_dxdy(display_act->gpd.piv_data, 
                                    &out_data, 
                                    - display_act->gpd.piv_data.mean_dx, 
                                    - display_act->gpd.piv_data.mean_dy);
/*
 * Copy parameters in Buffer structure for saving and, eventual,
 * later use
 */
            display_act->gpd.piv_post_par.z_off_dx = 
                - display_act->gpd.piv_data.mean_dx;
            display_act->gpd.piv_post_par.z_off_dy = 
                - display_act->gpd.piv_data.mean_dy;

            } else {
                piv_post_par.z_off_dx = 
                    gtk_spin_button_get_value_as_float
                    (GTK_SPIN_BUTTON(pivpost->spinbutton_suavg));
                piv_post_par.z_off_dy = 
                    gtk_spin_button_get_value_as_float
                     (GTK_SPIN_BUTTON(pivpost->spinbutton_svavg));
                gpiv_post_subtract_dxdy(display_act->gpd.piv_data, 
                                        &out_data, 
                                        - piv_post_par.z_off_dx, 
                                        - piv_post_par.z_off_dy);
            }
            
            for (i = 0; i < display_act->gpd.piv_data.ny; i++) {
                for (j = 0; j < display_act->gpd.piv_data.nx; j++) {
                    display_act->gpd.piv_data.dx[i][j] = out_data.dx[i][j];
                    display_act->gpd.piv_data.dy[i][j] = out_data.dy[i][j];
                }
            }

            update_all_vectors(&display_act->gpd);
            gpiv_free_pivdata(&out_data);
        }

        display_act->gpd.piv_post_par.z_off_dx_logic = TRUE;
        display_act->gpd.piv_post_par.z_off_dy_logic = TRUE;
        exec_process = FALSE;

    } else {
        warning_gpiv(_("no PIV data"));
    }

/*    g_warning("exec_savg:: entering"); */
/*    g_warning(_("not connected, yet")); */
}



void 
exec_subavg(PivPost * pivpost
            )
/*-----------------------------------------------------------------------------
 * Subtracts average or zero offsets for displacements or velocities
 */
{
    gint i, j;
    GpivPivData out_data;

    out_data.point_x = NULL;
    out_data.point_y = NULL;
    out_data.dx = NULL;
    out_data.dy = NULL;
    out_data.snr = NULL;
    out_data.peak_no = NULL;

    if  (display_act->gpd.exist_piv && !cancel_process) {
        exec_process = TRUE;
        piv_post_par.subtract = 1;
        out_data.nx = display_act->gpd.piv_data.nx;
        out_data.ny = display_act->gpd.piv_data.ny;
        gpiv_alloc_pivdata(&out_data);

        if (display_act->gpd.scaled_piv) {

            if (display_act->gpd.averaged_piv) {
            gpiv_post_subtract_dxdy(display_act->gpd.piv_data_scaled, 
                                    &out_data, 
                                    display_act->gpd.piv_data_scaled.mean_dx, 
                                    display_act->gpd.piv_data_scaled.mean_dy);

/*
 * Copy parameters in Buffer structure for saving and, eventual,
 * later use
 */
            display_act->gpd.piv_post_par.z_off_dx = 
                display_act->gpd.piv_data_scaled.mean_dx;
            display_act->gpd.piv_post_par.z_off_dy = 
                display_act->gpd.piv_data_scaled.mean_dy;

            } else {
            gpiv_post_subtract_dxdy(display_act->gpd.piv_data_scaled, 
                                    &out_data, 
                                    piv_post_par.z_off_dx, 
                                    piv_post_par.z_off_dx);
            }

            for (i = 0; i < display_act->gpd.piv_data.ny; i++) {
                for (j = 0; j < display_act->gpd.piv_data.nx; j++) {
                    display_act->gpd.piv_data_scaled.dx[i][j] = 
                        out_data.dx[i][j];
                    display_act->gpd.piv_data_scaled.dy[i][j] = 
                        out_data.dy[i][j];
                }
            }


/*
 * Subtract from piv_data, only for displaying purposes
 */
            if (display_act->gpd.averaged_piv) {
                gpiv_post_subtract_dxdy(display_act->gpd.piv_data, 
                                        &out_data, 
                                        display_act->gpd.piv_data.mean_dx, 
                                        display_act->gpd.piv_data.mean_dy);
            } else {
/*
 * apply inverse scale factor for offsetting of partimcle image displacements
 */
                gpiv_post_subtract_dxdy(display_act->gpd.piv_data, 
                                        &out_data, 
                                        piv_post_par.z_off_dx
                                        / image_par.s_scale 
                                        * image_par.t_scale, 
                                        piv_post_par.z_off_dy
                                        / image_par.s_scale 
                                        * image_par.t_scale
                                        );
            }

            for (i = 0; i < display_act->gpd.piv_data.ny; i++) {
                for (j = 0; j < display_act->gpd.piv_data.nx; j++) {
                    display_act->gpd.piv_data.dx[i][j] = out_data.dx[i][j];
                    display_act->gpd.piv_data.dy[i][j] = out_data.dy[i][j];
                }
            }
            update_all_vectors(&display_act->gpd);



/*
 * Only apply offseting for non-scaled pivdata with values from entry 
 * or z_off_dx and z_off_dy
 */
        } else {
            if (display_act->gpd.averaged_piv) {
            gpiv_post_subtract_dxdy(display_act->gpd.piv_data, 
                                    &out_data, 
                                    display_act->gpd.piv_data.mean_dx, 
                                    display_act->gpd.piv_data.mean_dy);
/*
 * Copy parameters in Buffer structure for saving and, eventual,
 * later use
 */
            display_act->gpd.piv_post_par.z_off_dx = 
                display_act->gpd.piv_data.mean_dx;
            display_act->gpd.piv_post_par.z_off_dy = 
                display_act->gpd.piv_data.mean_dy;

            } else {
                piv_post_par.z_off_dx = 
                    gtk_spin_button_get_value_as_float
                    (GTK_SPIN_BUTTON(pivpost->spinbutton_suavg));
                piv_post_par.z_off_dy = 
                    gtk_spin_button_get_value_as_float
                     (GTK_SPIN_BUTTON(pivpost->spinbutton_svavg));
                gpiv_post_subtract_dxdy(display_act->gpd.piv_data, 
                                        &out_data, 
                                        piv_post_par.z_off_dx, 
                                        piv_post_par.z_off_dy);
            }
            
            for (i = 0; i < display_act->gpd.piv_data.ny; i++) {
                for (j = 0; j < display_act->gpd.piv_data.nx; j++) {
                    display_act->gpd.piv_data.dx[i][j] = out_data.dx[i][j];
                    display_act->gpd.piv_data.dy[i][j] = out_data.dy[i][j];
                }
            }

            update_all_vectors(&display_act->gpd);
            gpiv_free_pivdata(&out_data);
        }

        display_act->gpd.piv_post_par.z_off_dx_logic = TRUE;
        display_act->gpd.piv_post_par.z_off_dy_logic = TRUE;
        exec_process = FALSE;

    } else {
        warning_gpiv(_("no PIV data"));
    }

/*    g_warning("exec_savg:: entering"); */
/*     g_warning(_("not connected, yet")); */
}



void 
exec_vorstra(void
             )
/*-----------------------------------------------------------------------------
 */
{
    char *err_msg = NULL;
/*      GpivScalarData out_data; */
    GtkWidget *view_scalardata_display1 = 
        gtk_object_get_data(GTK_OBJECT(display_act->mwin),
                            "view_scalardata_display1");
    GtkWidget *view_scalardata_display2 = 
        gtk_object_get_data(GTK_OBJECT(display_act->mwin),
                            "view_scalardata_display2");
    GtkWidget *view_scalardata_display3 = 
        gtk_object_get_data(GTK_OBJECT(display_act->mwin),
                            "view_scalardata_display3");


    if  (display_act->gpd.exist_piv && !cancel_process) {
        exec_process = TRUE;


/*
 * Vorticity calculation
 */
        if (piv_post_par.operator_vorstra == GPIV_VORTICITY) {
            if (display_act->gpd.exist_vor) {
/*                 destroy_all_scalars(&display_act->gpd.vor_data, GPIV_VORTICITY); */
                destroy_all_scalars(display_act, GPIV_VORTICITY);
                gpiv_free_scdata(&display_act->gpd.vor_data);
                display_act->gpd.exist_vor = FALSE;
            }
            
            display_act->gpd.vor_data.nx = display_act->gpd.piv_data.nx;
            display_act->gpd.vor_data.ny = display_act->gpd.piv_data.ny;
            gpiv_alloc_scdata(&display_act->gpd.vor_data);
            if ((err_msg = 
                 gpiv_post_vorstra(display_act->gpd.piv_data, 
                                   &display_act->gpd.vor_data, 
                                   piv_post_par))
                != NULL) error_gpiv ("%s: %s", RCSID, err_msg);
            create_all_scalars(display_act, GPIV_VORTICITY);
/*
 * Vorticity data exist but are unsaved
 * and are displayed as scalar data
 */
            display_act->gpd.exist_vor = TRUE;
            display_act->gpd.saved_vor = FALSE;
            gtk_check_menu_item_set_active
                (GTK_CHECK_MENU_ITEM(view_scalardata_display1), 
                 TRUE);

/*
 * Repete vorticity calculation process for scaled piv_data
 */
            if (display_act->gpd.scaled_piv) {
                display_act->gpd.vor_data_scaled.nx = 
                    display_act->gpd.piv_data.nx;
                display_act->gpd.vor_data_scaled.ny = 
                    display_act->gpd.piv_data.ny;
                gpiv_alloc_scdata(&display_act->gpd.vor_data_scaled);
                if ((err_msg = 
                     gpiv_post_vorstra(display_act->gpd.piv_data_scaled, 
                                       &display_act->gpd.vor_data_scaled, 
                                       piv_post_par))
                    != NULL) error_gpiv ("%s: %s", RCSID, err_msg);

            display_act->gpd.exist_vor_scaled = TRUE;
            }


/*
 * Shear strain calculation
 */
        } else if (piv_post_par.operator_vorstra == GPIV_S_STRAIN) {
            if (display_act->gpd.exist_sstrain) {
/*                 destroy_all_scalars(&display_act->gpd.sstrain_data, GPIV_S_STRAIN); */
                destroy_all_scalars(display_act, GPIV_S_STRAIN);
                gpiv_free_scdata(&display_act->gpd.sstrain_data);
                display_act->gpd.exist_sstrain = FALSE;
            }
            
            display_act->gpd.sstrain_data.nx = display_act->gpd.piv_data.nx;
            display_act->gpd.sstrain_data.ny = display_act->gpd.piv_data.ny;
            gpiv_alloc_scdata(&display_act->gpd.sstrain_data);
            if ((err_msg = 
                 gpiv_post_vorstra(display_act->gpd.piv_data, 
                                   &display_act->gpd.sstrain_data, 
                                   piv_post_par))
                != NULL) error_gpiv ("%s: %s", RCSID, err_msg);

            create_all_scalars(display_act, GPIV_S_STRAIN);
/*
 * Shear strain data exist but are unsaved
 * and are displayed as scalar data
 */
            display_act->gpd.exist_sstrain = TRUE;
            display_act->gpd.saved_sstrain = FALSE;
            gtk_check_menu_item_set_active
                (GTK_CHECK_MENU_ITEM(view_scalardata_display2), 
                 TRUE);
/*
 * Repete shear strain calculation process for scaled piv_data
 */
            if (display_act->gpd.scaled_piv) {
                display_act->gpd.sstrain_data_scaled.nx = 
                    display_act->gpd.piv_data.nx;
                display_act->gpd.sstrain_data_scaled.ny = 
                    display_act->gpd.piv_data.ny;
                gpiv_alloc_scdata(&display_act->gpd.sstrain_data_scaled);
                if ((err_msg = 
                     gpiv_post_vorstra(display_act->gpd.piv_data_scaled, 
                                       &display_act->gpd.sstrain_data_scaled, 
                                       piv_post_par))
                    != NULL) error_gpiv ("%s: %s", RCSID, err_msg);

                display_act->gpd.exist_sstrain_scaled = TRUE;
            }

/*
 * Normal strain calculation
 */
        } else if (piv_post_par.operator_vorstra == GPIV_N_STRAIN) {
            if (display_act->gpd.exist_nstrain) {
/*                 destroy_all_scalars(&display_act->gpd.nstrain_data, GPIV_N_STRAIN); */
                destroy_all_scalars(display_act, GPIV_N_STRAIN);
                gpiv_free_scdata(&display_act->gpd.nstrain_data);
                display_act->gpd.exist_nstrain = FALSE;
            }
            
            display_act->gpd.nstrain_data.nx = display_act->gpd.piv_data.nx;
            display_act->gpd.nstrain_data.ny = display_act->gpd.piv_data.ny;
            gpiv_alloc_scdata(&display_act->gpd.nstrain_data);
            if ((err_msg = 
                 gpiv_post_vorstra(display_act->gpd.piv_data, 
                                   &display_act->gpd.nstrain_data, 
                                   piv_post_par))
                != NULL) error_gpiv ("%s: %s", RCSID, err_msg);

            create_all_scalars(display_act, GPIV_N_STRAIN);
/*
 * Normal strain data exist but are unsaved
 * and are displayed as scalar data
 */
            display_act->gpd.exist_nstrain = TRUE;
            display_act->gpd.saved_nstrain = FALSE;
            gtk_check_menu_item_set_active
                (GTK_CHECK_MENU_ITEM(view_scalardata_display3), 
                 TRUE);
/*
 * Repete normal strain calculation process for scaled piv_data
 */
            if (display_act->gpd.scaled_piv) {
                display_act->gpd.nstrain_data_scaled.nx = 
                    display_act->gpd.piv_data.nx;
                display_act->gpd.nstrain_data_scaled.ny = 
                    display_act->gpd.piv_data.ny;
                gpiv_alloc_scdata(&display_act->gpd.nstrain_data_scaled);
                if ((err_msg = 
                     gpiv_post_vorstra(display_act->gpd.piv_data_scaled, 
                                       &display_act->gpd.nstrain_data_scaled, 
                                       piv_post_par))
                    != NULL) error_gpiv ("%s: %s", RCSID, err_msg);

                display_act->gpd.exist_nstrain_scaled = TRUE;
            }



       }  else {
        warning_gpiv(_("exec_vorstra: non valid operation"));
       }
     exec_process = FALSE;
 
   } else {
        warning_gpiv(_("no PIV data"));
    }

/*
 * Copy parameters in Buffer structure for saving and, eventual,
 * later use
 */

    if (display_act->gpd.exist_vor ||  display_act->gpd.exist_sstrain || 
        display_act->gpd.exist_nstrain) {

        display_act->gpd.piv_post_par.diff_type = piv_post_par.diff_type;
        display_act->gpd.piv_post_par.operator_vorstra = 
            piv_post_par.operator_vorstra;
        
        display_act->gpd.piv_post_par.diff_type_logic = TRUE;
        display_act->gpd.piv_post_par.operator_vorstra_logic = TRUE;
        
        gpiv_post_print_parameters(display_act->gpd.piv_post_par);
    }


/* 	  gpiv_write_scdata( &display_act->gpd.vor_data, c_line, nc_lines,  */
/* 			     scale, RCSID); */

}




/*
 * Piv post-processing callback functions
 */

void
on_spinbutton_post_scale(GtkSpinButton * widget, 
                         GtkWidget * entry
/* (GtkWidget *widget, gpointer data) */
                         )
/*-----------------------------------------------------------------------------
 */
{
    enum VariableType {
	COL_POS = 1,
	ROW_POS = 2,
	S_SCALE = 3,
	T_SCALE = 4
    } var_type;

    var_type = atoi(gtk_object_get_data(GTK_OBJECT(widget), "var_type"));

    if (var_type == COL_POS) {
	image_par.z_off_x = gtk_spin_button_get_value_as_float(widget);
	image_par.z_off_x_logic = TRUE;
        if (display_act != NULL) {
            display_act->img.image_par.z_off_x = image_par.z_off_x;
            display_act->img.image_par.z_off_x_logic = TRUE;
         }

    } else if (var_type == ROW_POS) {
	image_par.z_off_y = gtk_spin_button_get_value_as_float(widget);
	image_par.z_off_y_logic = TRUE;
        if (display_act != NULL) {
            display_act->img.image_par.z_off_y = image_par.z_off_y;
            display_act->img.image_par.z_off_y_logic = TRUE;
        }

    } else if (var_type == S_SCALE) {
	image_par.s_scale = gtk_spin_button_get_value_as_float(widget);
	image_par.s_scale_logic = TRUE;
        if (display_act != NULL) {
            display_act->img.image_par.s_scale = image_par.s_scale;
            display_act->img.image_par.s_scale_logic = TRUE;
        }

    } else if (var_type == T_SCALE) {
	image_par.t_scale = gtk_spin_button_get_value_as_float(widget);
        image_par.t_scale_logic = TRUE;
        if (display_act != NULL) {
            display_act->img.image_par.t_scale = image_par.t_scale;
            display_act->img.image_par.t_scale_logic = TRUE;
        }

    } else  {
	gpiv_warning(_("on_spinbutton_post_scale: should not arrive here"));
    }

}



void
on_button_post_scale_enter(GtkWidget *widget, 
                           gpointer data
                           )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Calculates scaled locations and velocities");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_post_scale(GtkWidget *widget, 
                     gpointer data
                     )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gint row, ibuf;

    cancel_process = FALSE;
    if (nbufs > 0) {
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                                 row);
            ibuf = display_act->id;
            if (display[ibuf] != NULL
                && display_act->mwin != NULL 
                && GTK_WIDGET_VISIBLE(GTK_WIDGET(display_act->mwin)) ) { 
                gdk_window_show(GTK_WIDGET(display_act->mwin)->window);
                gdk_window_raise(GTK_WIDGET(display_act->mwin)->window);
            }
            exec_scale(gpiv->pivpost);
        }
    }
}



void 
on_spinbutton_post_suavg(GtkSpinButton * widget, 
			 GtkWidget * entry
                         )
/*-----------------------------------------------------------------------------
 */
{
    piv_post_par.z_off_dx = gtk_spin_button_get_value_as_float(widget);
/*     g_warning("on_spinbutton_post_suavg:: z_off_dx = %f",  */
/*               piv_post_par.z_off_dx); */
    display_act->gpd.averaged_piv = FALSE;
}



void 
on_spinbutton_post_svavg(GtkSpinButton * widget, 
			 GtkWidget * entry
                         )
/*-----------------------------------------------------------------------------
 */
{
    piv_post_par.z_off_dy = gtk_spin_button_get_value_as_float(widget);
/*     g_warning("n_spinbutton_post_svavg:: z_off_dy = %f",  */
/*               piv_post_par.z_off_dy); */
    display_act->gpd.averaged_piv = FALSE;
}



void 
on_button_post_savg_enter(GtkWidget *widget, 
                          gpointer data
                          )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Calculates spatial average displacements");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_post_savg(GtkWidget *widget, 
                    gpointer data
                    )
/*-----------------------------------------------------------------------------
 */
{
    PivPost * post = gtk_object_get_data(GTK_OBJECT(widget), "post");
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gint row, ibuf;

    cancel_process = FALSE;
    if (nbufs > 0) {
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                                 row);
            ibuf = display_act->id;
            if (display[ibuf] != NULL
                && display_act->mwin != NULL 
                && GTK_WIDGET_VISIBLE(GTK_WIDGET(display_act->mwin)) ) { 
                gdk_window_show(GTK_WIDGET(display_act->mwin)->window);
                gdk_window_raise(GTK_WIDGET(display_act->mwin)->window);
            }
            exec_savg(post);
        }
    }
}



void 
on_button_post_subavg_enter(GtkWidget *widget, 
                            gpointer data
                            )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Subtracts mean displacements / offset values from PIV data");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_post_subavg(GtkWidget *widget, 
                      gpointer data
                      )
/*-----------------------------------------------------------------------------
 */
{
    PivPost * post = gtk_object_get_data(GTK_OBJECT(widget), "post");
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gint row, ibuf;

    cancel_process = FALSE;
    if (nbufs > 0) {
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                                 row);
            ibuf = display_act->id;
            if (display[ibuf] != NULL
                && display_act->mwin != NULL 
                && GTK_WIDGET_VISIBLE(GTK_WIDGET(display_act->mwin)) ) { 
                gdk_window_show(GTK_WIDGET(display_act->mwin)->window);
                gdk_window_raise(GTK_WIDGET(display_act->mwin)->window);
            }
            exec_subavg(post);
        }
    }
}



void 
on_button_post_addavg_enter(GtkWidget *widget, 
                            gpointer data
                            )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Adds mean displacements / offset values from PIV data");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_post_addavg(GtkWidget *widget, 
                      gpointer data
                      )
/*-----------------------------------------------------------------------------
 */
{
    PivPost * post = gtk_object_get_data(GTK_OBJECT(widget), "post");
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gint row, ibuf;

    cancel_process = FALSE;
    if (nbufs > 0) {
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                                 row);
            ibuf = display_act->id;
            if (display[ibuf] != NULL
                && display_act->mwin != NULL 
                && GTK_WIDGET_VISIBLE(GTK_WIDGET(display_act->mwin)) ) { 
                gdk_window_show(GTK_WIDGET(display_act->mwin)->window);
                gdk_window_raise(GTK_WIDGET(display_act->mwin)->window);
            }
            exec_addavg(post);
        }
    }
}



void 
on_radiobutton_post_vorstra_output_enter(GtkWidget *widget, 
                                         gpointer data
                                         )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Selects the differential quantity");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_radiobutton_post_vorstra_output(GtkWidget *widget, 
				   gpointer data
                                   )
/*-----------------------------------------------------------------------------
 */
{
    PivPost * post = gtk_object_get_data(GTK_OBJECT(widget), "post");
    piv_post_par.operator_vorstra = atoi(gtk_object_get_data
                                         (GTK_OBJECT(widget),
					"operator"));
    fprintf(stderr, "operator_vorstra = %d\n", piv_post_par.operator_vorstra);

    if (piv_post_par.operator_vorstra == GPIV_VORTICITY) {
	gtk_widget_set_sensitive(post->radiobutton_vorstra_diffscheme_4,
				 TRUE);
    } else {
        if (piv_post_par.diff_type == GPIV_CIRCULATION) {
            piv_post_par.diff_type = GPIV_CENTRAL;
            gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                        (post->radiobutton_vorstra_diffscheme_1), 
                                        TRUE);
        }
        gtk_widget_set_sensitive(post->radiobutton_vorstra_diffscheme_4,
				 FALSE);
    }
}



void 
on_radiobutton_post_vorstra_diffscheme_enter(GtkWidget *widget, 
                                             gpointer data
                                             )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Selects the type of differential scheme");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_radiobutton_post_vorstra_diffscheme(GtkWidget *widget, 
                                       gpointer data
                                       )
/*-----------------------------------------------------------------------------
 */
{
    piv_post_par.diff_type = atoi(gtk_object_get_data(GTK_OBJECT(widget),
					"diff_type"));
/*     fprintf(stderr, "diff_type = %d\n", piv_post_par.diff_type); */

}



void 
on_button_post_vorstra_enter(GtkWidget *widget, 
			     gpointer data
                             )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Calculates differential quantity");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_post_vorstra(GtkWidget *widget, 
		       gpointer data
                       )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gint row, ibuf;

    cancel_process = FALSE;
    if (nbufs > 0) {
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                                 row);
            ibuf = display_act->id;
            if (display[ibuf] != NULL 
                && display_act->mwin != NULL 
                && GTK_WIDGET_VISIBLE(GTK_WIDGET(display_act->mwin)) ) { 
                gdk_window_show(GTK_WIDGET(display_act->mwin)->window);
                gdk_window_raise(GTK_WIDGET(display_act->mwin)->window);
            }
            exec_vorstra();
        }
    }
}


