/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*----------------------------------------------------------------------
 
  gpiv - Graphic program for Particle Image Velocimetry, based on gtk/gnome
          libraries.

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of gpiv.

   Gpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

----------------------------------------------------------------------*/

/*
 * (Callback) functions for console
 * $Log: console.c,v $
 * Revision 1.21  2006/01/31 14:28:11  gerber
 * version 0.3.0
 *
 * Revision 1.19  2005/02/26 09:43:30  gerber
 * parameter flags (parameter_logic) defined as gboolean
 *
 * Revision 1.18  2005/02/26 09:17:13  gerber
 * structured of interrogate function by using gpiv_piv_isiadapt
 *
 * Revision 1.17  2005/02/12 13:09:21  gerber
 * Changing some structure and constant names for DAC
 *
 * Revision 1.16  2005/01/19 15:53:41  gerber
 * Initiation of Data Acquisition (DAC); trigerring of lasers and camera
 * by using RTAI and Realtime Linux, recording images from IEEE1394
 * (Firewire) IIDC compliant camera's
 *
 * Revision 1.15  2004/10/15 19:24:05  gerber
 * GPIV_ and Gpiv prefix to defines and structure names of libgpiv
 *
 * Revision 1.14  2004/06/14 21:19:23  gerber
 * Image depth up to 16 bits.
 * Improvement "single int" and "drag int" in Eval tab.
 * Viewer's pop-up menu.
 * Adaption for gpiv_matrix_* and gpiv_vector_*.
 * Resizing console.
 * See Changelog for further info.
 *
 * Revision 1.13  2003/09/04 13:31:54  gerber
 * init of printing (unfinished)
 *
 * Revision 1.12  2003/09/01 11:17:14  gerber
 * improved monitoring of interrogation process
 *
 * Revision 1.11  2003/08/22 15:24:52  gerber
 * interactive spatial scaling
 *
 * Revision 1.10  2003/07/31 11:43:26  gerber
 * display images in gnome canvas (HOERAreset)
 *
 * Revision 1.9  2003/07/25 15:40:23  gerber
 * removed/disabled setting of correlation in Eval tab, Correlation type in Image info tab
 *
 * Revision 1.8  2003/07/13 14:38:18  gerber
 * changed error handling of libgpiv
 *
 * Revision 1.7  2003/07/12 21:21:15  gerber
 * changed error handling libgpiv
 *
 * Revision 1.5  2003/07/10 11:56:07  gerber
 * added man page
 *
 * Revision 1.4  2003/07/06 15:29:49  gerber
 * repair message text closing buffer
 *
 * Revision 1.3  2003/07/05 13:14:57  gerber
 * drag and drop of a _series_ of filenames from NAUTILUS
 *
 * Revision 1.2  2003/06/27 13:47:26  gerber
 * display ruler, line/point evaluation
 *
 * Revision 1.1.1.1  2003/06/17 17:10:52  gerber
 * Imported gpiv
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include "support.h"

#include "gpiv_gui.h"
#include "console_interface.h"
#include "dialog_interface.h"
/* #include "console_menus.h" */
#include "console.h"
#include "display.h"
#ifndef DISABLE_DAC
#include "dac.h"
#endif /* DISABLE_DAC */
#include "piveval.h"
#include "pivvalid.h"
#include "pivpost.h"
#include "preferences.h"
#include "print.h"
#include "utils.h"



/*
 * Local functions
 */

typedef enum _Format Format;
enum _Format {
    RAW_ASCII,
    HDF,
    GPIV_DAVIS,
}; 

/*
 * Local functions
 */

static int
open_img(char *fname_in, 
         Image * img
         )
/*-----------------------------------------------------------------------------
 * Opens an image from hdf or raw-data file (<=> open_dac_img)
 */
{
    char *err_msg = NULL;
    char *ext = g_strdup(strrchr(fname_in, '.'));
/*     g_warning("open_img:: 0 fname_in=%s ext=%s", fname_in, ext); */

/*
 * parameter initializing of image
 */
    gpiv_img_parameters_logic(&img->image_par, FALSE);
/*
 * Reads hdf format
 */
    if (strcmp(ext, GPIV_EXT_GPIV) == 0) {
        if ((err_msg = 
             gpiv_img_fread_hdf5_parameters(fname_in, &img->image_par))
             != NULL) {
            warning_gpiv("%s", err_msg); 
            return -1;
        }
        if (img->image_par.nrows != gpiv_par.img_height
            || img->image_par.ncolumns != gpiv_par.img_width) {
            warning_gpiv(_("Image or header doesn't exist \n\
or image dimensions  (%dx%d) differ \n\
from parameter settings (%dx%d)")
                        , img->image_par.ncolumns, img->image_par.nrows, 
                        gpiv_par.img_width, gpiv_par.img_height
                        );
            return -1;
        }
        gpiv_img_cp_parameters(image_par, &img->image_par, FALSE, FALSE);
        if ((err_msg = gpiv_img_check_header_read(img->image_par)) != NULL) {
            warning_gpiv("%s", err_msg); 
            return -1;
        }

        img->img1 = gpiv_alloc_img(img->image_par);
        if (img->image_par.x_corr != image_par.x_corr) {
            warning_gpiv(_("cross correlation parameter from image \n\
not in agreement with GPIV setting"));
            return -1;
        }
        if (img->image_par.x_corr) 
            img->img2 = gpiv_alloc_img(img->image_par);

        if ((err_msg = 
             gpiv_fread_hdf5_image (fname_in, img->img1, 
                                    img->img2, &img->image_par)) 
            != NULL) {
            warning_gpiv ("%s", err_msg);
            return -1;
        }


/*
 * BUGFIX: Reads raw portable graymap (pgm) image format
 */
     } else if (strcmp(ext, GPIV_EXT_PGM_IMAGE) == 0) {
        gint line_nr;
        g_warning("open_img:: reads pgm format 0");

        if (!g_module_supported()) {
            g_warning("open_img:: !g_module_supported");
        } else {
            g_warning("open_img:: g_module_supported");
        }
        if ((err_msg = gpiv_img_read_pgm_header(fname_in,
                                                &img->image_par,
                                                &line_nr
                                                )) != NULL) {
            warning_gpiv ("%s", err_msg);
            return -1;
        }
        g_message("open_img:: line_nr=%d", line_nr);

        img->img1 = gpiv_alloc_img(img->image_par);
        if (img->image_par.x_corr) {
            img->img2 = gpiv_alloc_img(img->image_par);
        }

        if ((err_msg = gpiv_fread_pgm_image(fname_in,
                                            &img->img1[0], 
                                            &img->img2[0], 
                                            &img->image_par,
                                            line_nr
                                            )) != NULL) {
            warning_gpiv ("%s", err_msg);
            return -1;
        }

        g_message("open_img:: assert (img->img1[0] != NULL);");
        assert (img->img1[0] != NULL);
        g_message("open_img:: returned from assert");

        g_warning("open_img:: reads pgm format 1");
        if (img->image_par.nrows != gpiv_par.img_height
            || img->image_par.ncolumns != gpiv_par.img_width) {
            warning_gpiv(_("Image or header doesn't exist \n\
or image dimensions  (%dx%d) differ \n\
from parameter settings (%dx%d)")
                        , img->image_par.ncolumns, img->image_par.nrows, 
                        gpiv_par.img_width, gpiv_par.img_height
                        );

            return -1;
        }

        gpiv_img_cp_parameters(image_par, &img->image_par, FALSE, FALSE);
        if ((err_msg = gpiv_img_check_header_read(img->image_par)) != NULL) {
            warning_gpiv("%s", err_msg); 
            return -1;
        }


/*
 * Reads raw data format
 * Reads image header data from file.h or copies from originals
 */
     } else if (strcmp(ext, GPIV_EXT_RAW_IMAGE) == 0){
        char fname_header[GPIV_MAX_CHARS];
        char *fname_ext = display_act->fname_base;
       g_snprintf(fname_header, GPIV_MAX_CHARS, "%s%s", fname_ext, GPIV_EXT_HEADER);
        if (print_par) g_message("image header is: %s", fname_header);
        gpiv_scan_parameter( "", fname_header, &img->image_par, print_par);
        if (print_par) 
            g_message("open_img: x_corr_logic = %d \n"
                        "image_par.nrows = %d \n"
                        "image_par.ncolumns = %d",
                        img->image_par.x_corr_logic,
                        img->image_par.nrows, 
                        img->image_par.ncolumns );
        if (img->image_par.nrows != gpiv_par.img_height
            || img->image_par.ncolumns != gpiv_par.img_width) {
            warning_gpiv(_("Image or header doesn't exist \n\
or image dimensions  (%dx%d) differ \n\
from parameter settings (%dx%d)")
                        , img->image_par.ncolumns, img->image_par.nrows, 
                        gpiv_par.img_width, gpiv_par.img_height
                        );
            return -1;
        }
        gpiv_img_cp_parameters (image_par, &img->image_par, FALSE, FALSE);
        if ((err_msg = 
             gpiv_img_check_header_read(img->image_par))
            != NULL) {
            warning_gpiv ("%s", err_msg); 
            return -1;
        }
/*
 * memory allocation of matrices and vectors
 * gpiv_alloc_imgdata(img->img1, image_par.nrows, image_par.ncolumns);
 */
        img->img1 = gpiv_alloc_img(img->image_par);
        if (img->image_par.x_corr != image_par.x_corr) {
            warning_gpiv(_("cross correlation parameter from image \n\
not in agreement with GPIV setting"));
            return -1;
        }
        if (image_par.x_corr) 
        img->img2 = gpiv_alloc_img(img->image_par);


/*
 * reads image data from file in binary format
 */
        if ((err_msg = 
             gpiv_fread_image(fname_in, img->img1, img->img2, img->image_par))
             != NULL) {
            warning_gpiv ("%s", err_msg);
            return -1;
        }

/*         if (!img->img1 || !img->img2) { */
/*             warning_gpiv(_("failure reading image")); */
/*             return -1; */
/*         } */


/*
 * Reads LaVision's (tm) DaVis format (.img)
 */
    } else if (strcmp(ext, GPIV_EXT_DAVIS) == 0) {
        gpiv_img_fread_davis_parameters(fname_in, &img->image_par);
        
        img->img1 = gpiv_alloc_img(img->image_par);
        img->img2 = gpiv_alloc_img(img->image_par);
        
        gpiv_fread_davis_image(fname_in, img->img1, img->img2, 
                               &img->image_par);
    } else {
        warning_gpiv(_("open_img: should not arrive here"));
        return -1;
    }

    g_free(ext);
    display_act->img.exist_img = TRUE;
    return 0;
}



float
image_mean(guint16 **img, 
           gint ncols, 
           gint nrows)
/*-----------------------------------------------------------------------------
 */
{
    int i, j, ncount = ncols * nrows;
    float img_sum = 0.0, mean = 0.0;
    
    for (i = 0; i < nrows; i++) {
        for (j = 0; j < ncols; j++) {
            img_sum += (float) img[i][j];
        }
    }
    
    mean = img_sum / (float) ncount;
    return mean;
}



static int
open_piv(char *fname_in, 
         GpivData * gpd
         )
/*-----------------------------------------------------------------------------
 * Opens piv data file and displays
 */
{
    char *err_msg = NULL, RCSID_DATA[GPIV_MAX_CHARS];
    char *ext = g_strdup(strrchr(fname_in, '.'));
    gint var_scale = 0, scale = 0;

/*
 * parameter initializing of piv data
 */
    gpd->piv_eval_par.autokey = 1;
    gpiv_piv_parameters_logic(&gpd->piv_eval_par, FALSE);

/*
 * Reads hdf format
 */
    if (strcmp(ext, GPIV_EXT_GPIV) == 0) {
        if ((err_msg = 
             gpiv_fcount_hdf5_data(fname_in, &gpd->piv_data.ny, 
                                   &gpd->piv_data.nx))
            != NULL) {
            g_warning("%s", err_msg);
/*             warning_gpiv ("open_piv: will return -1; no PIV data"); */
            return -1;

        }

        gpiv_alloc_pivdata(&gpd->piv_data);
        if ((err_msg =
             gpiv_fread_hdf5_piv_position(fname_in, &gpd->piv_data, 
                                          RCSID_DATA, &var_scale))
            != NULL)  error_gpiv ("%s: %s", VERSION, err_msg);
        if ((err_msg = 
             gpiv_fread_hdf5_pivdata(fname_in, &gpd->piv_data, "PIV", 
                                     RCSID_DATA, &var_scale))
            != NULL) {
            warning_gpiv ("%s", err_msg);
            return -1;
        }
        gpd->exist_piv = TRUE;
        gpd->averaged_piv = FALSE;
        
        
/*
 * Reads raw data format
 */
    } else if (strcmp(ext, GPIV_EXT_RAW_IMAGE) == 0
               || strcmp(ext, GPIV_EXT_DAVIS) == 0
               ) {
        FILE *fp_in;
        gchar *dirname = g_strdup(g_dirname(fname_in)),
            *fname_base = g_strdup(g_basename(fname_in)),
            *fname_ext,
            *fname_piv;

/* BUGFIX
 * check on existance of piv data file
 */
        strtok(fname_base, ".");
        fname_ext = g_strdup(g_strconcat(dirname, G_DIR_SEPARATOR_S,
                                         fname_base, NULL));
        fname_piv = g_strdup(g_strconcat(fname_ext, GPIV_EXT_PIV, NULL));

        if((fp_in = fopen(fname_piv,"r")) != NULL) { 
            fclose(fp_in);
            if ((scale = 
                 gpiv_fcount_pivdata(fname_piv, &gpd->piv_data)) == -1) {
/*             g_warning ("open_piv: will return -1; no PIV data"); */
                return -1;
            }

            gpiv_alloc_pivdata(&gpd->piv_data);

            if ((err_msg = 
                 gpiv_fread_pivdata(fname_piv, 
                                    &gpd->piv_data, 
                                    c_line, 
                                    &nc_lines))
                != NULL) {
                warning_gpiv ("%s", err_msg);
                return -1;
            }
            gpd->exist_piv = TRUE;
            gpd->averaged_piv = FALSE;
        }

        g_free(dirname);
        g_free(fname_base); 
        g_free(fname_ext); 
        g_free(fname_piv); 


    } else {
         if (print_par) g_message("open_piv: no PIV data");
         return -1;
    }


    g_free(ext);
    return 0;
}



static void
select_action_from_fname(GpivConsole *gpiv, 
                          gchar *fname_in)
/* ----------------------------------------------------------------------------
 * checks filename extension on valid image/data name, on existence of buffer,
 * loads data
 */
{
    gint ibuf, return_val;
    gchar *ext, 
        *dirname, 
        *fname_base, 
        *fname_ext;
    gboolean exist_buf = FALSE;

    ext = g_strdup(strrchr(fname_in, '.'));
/* also added to push_list_lastfnames(fname_base); */
    gpiv_var.fname_last = g_strdup(fname_in);
    gnome_config_push_prefix("/gpiv/RuntimeVariables/");
    gnome_config_set_string("fname_last", gpiv_var.fname_last);
    gnome_config_pop_prefix();
    gnome_config_sync();

/*
 * Stripping file name
 *
 */
/*----------------------------------------------------------------------
 * With GLIB-2.0:
 * gchar*      g_path_get_basename             (const gchar *file_name);
 * gchar*      g_path_get_dirname             (const gchar *file_name);
 *             g_build_filename 
 *----------------------------------------------------------------------*/
    dirname = g_strdup(g_dirname(fname_in));
    fname_base = g_strdup(g_basename(fname_in));
    strtok(fname_base, ".");
    push_list_lastfnames(fname_base);
    fname_ext = g_strdup(g_strconcat(dirname, G_DIR_SEPARATOR_S,
                                     fname_base, NULL));
    
/*
 * reading image data from raw data format or hdf format
 */
    if (strcmp(ext, GPIV_EXT_RAW_IMAGE) == 0 
        || strcmp(ext, GPIV_EXT_PGM_IMAGE) == 0
        || strcmp(ext, GPIV_EXT_GPIV) == 0
        || strcmp(ext, GPIV_EXT_DAVIS) == 0) {
	exist_buf = FALSE;
	if (nbufs == 0) {
	    sensitive(gpiv, IMG, TRUE);
 	    sensitive(gpiv, EVAL, TRUE);
	} else {
            
/*
 * Check if file basename (and buffer/display) already exists
 * or if there is a gap in the numbering of th buffers
 */
	    for (ibuf = 0; ibuf < nbufs; ibuf++) {
                if (display[ibuf] != NULL 
                    && strcmp(fname_ext, display[ibuf]->fname_base) == 0 ) {
                    if((return_val = 
                        open_img(fname_in, &display_act->img)) 
                       != 0) {
                        return;
                    }

                    exist_buf = TRUE;
                    break;

                } else if (display[ibuf] == NULL) {
                    load_buffer(gpiv, fname_in, fname_base, fname_ext, 
                                ibuf, INSERT);
                    exist_buf = TRUE;
                    nbufs++;
                    break;
                } else {
/*                     g_warning("select_action_from_fname: should not arrive here"); */
                }
            }
        }

/*
 * Create new buffer and launches a display if file basename doesn't exist
 * and append to the list
 */
        if (!exist_buf) {
	    nbufs++;
	    ibuf = nbufs - 1;
            load_buffer(gpiv, fname_in, fname_base, fname_ext,
                        ibuf, APPEND);
            
        }

    } else {
	warning_gpiv(_("select_action_from_fname: non-valid file name"));
    }

    gtk_clist_set_selection_mode(GTK_CLIST (gpiv->clist_buf),
                                 GTK_SELECTION_EXTENDED);

    g_free(ext);
    g_free(dirname); 
    g_free(fname_base); 
    g_free(fname_ext);
/*     if (print_par)  */
/*         g_message("select_action_from_fname: fname_base = %s", display_act->fname_base); */
}



static void
exec_save(Display * disp)
/*-----------------------------------------------------------------------------
 * Store all data in display_act->fname_base.gpi in hdf5 format
 */
{
    char *err_msg = NULL;
    gint flag_vor = 0;
    char fname_out[GPIV_MAX_CHARS], fname_par[GPIV_MAX_CHARS];
    FILE *fp_par;

    if (gpiv_par.hdf) {
        if (print_par) g_message("writing HDF5 fmt");
        gpiv_io_make_fname(display_act->fname_base, GPIV_EXT_GPIV, fname_out);
        
        if ((err_msg = 
             gpiv_fcreate_hdf5 (fname_out)) 
                != NULL) error_gpiv ("%s", err_msg);

        if ((err_msg = 
             gpiv_img_fwrite_hdf5_parameters (fname_out, 
                                              &display_act->img.image_par))
            != NULL) error_gpiv ("%s", err_msg);

        if (gpiv_par.hdf_img) {
            if ((err_msg = 
                 gpiv_fwrite_hdf5_image (fname_out, display_act->img.img1, 
                                         display_act->img.img2, 
                                         &display_act->img.image_par))
                != NULL) error_gpiv ("%s", err_msg);
        }


            
        if (display_act->gpd.exist_piv) {
            if (display_act->gpd.scaled_piv) {
                if ((err_msg = 
                     gpiv_fwrite_hdf5_piv_position
                     (fname_out, 
                      &display_act->gpd.piv_data_scaled, 
                      VERSION, 
                      &display_act->gpd.scaled_piv))
                    != NULL) error_gpiv ("%s", err_msg);
                
                if ((err_msg = 
                gpiv_fwrite_hdf5_pivdata 
                    (fname_out, 
                     &display_act->gpd.piv_data_scaled,  
                     "PIV", 
                     VERSION, 
                     &display_act->gpd.scaled_piv))
                   != NULL) error_gpiv ("%s", err_msg);
             } else {
                if ((err_msg = 
                     gpiv_fwrite_hdf5_piv_position
                     (fname_out, 
                      &display_act->gpd.piv_data, 
                      VERSION, 
                      &display_act->gpd.scaled_piv))
                    != NULL) error_gpiv ("%s", err_msg);
                
                if ((err_msg = 
                     gpiv_fwrite_hdf5_pivdata 
                     (fname_out, 
                      &display_act->gpd.piv_data,  
                      "PIV", 
                      VERSION, 
                      &display_act->gpd.scaled_piv))
                    != NULL) error_gpiv ("%s", err_msg);
             }
            
            display_act->gpd.saved_piv = TRUE;
            
            if ((err_msg = 
                 gpiv_piv_fwrite_hdf5_parameters 
                 (fname_out, 
                  &display_act->gpd.piv_eval_par))
                != NULL) error_gpiv ("%s", err_msg);
        }
            

        if (display_act->gpd.exist_valid) {
            if ((err_msg = 
                 gpiv_valid_fwrite_hdf5_parameters 
                 (fname_out, &display_act->gpd.piv_valid_par))
                != NULL) error_gpiv ("%s", err_msg);
        }


        if (display_act->gpd.exist_vor) {
            if (display_act->gpd.scaled_piv ) {
                if ((err_msg = 
                     gpiv_fwrite_hdf5_scdata(fname_out, 
                                             &display_act->gpd.
                                             vor_data_scaled, 
                                             "GPIV_VORTICITY",
                                             VERSION))
                    != NULL) error_gpiv ("%s", err_msg);
            } else {
               if ((err_msg = 
                    gpiv_fwrite_hdf5_scdata(fname_out, 
                                            &display_act->gpd.vor_data, 
                                            "GPIV_VORTICITY",
                                            VERSION))
                   != NULL) error_gpiv ("%s", err_msg);
            }
            display_act->gpd.saved_vor = TRUE;
        }


        if (display_act->gpd.exist_sstrain) {
            if (display_act->gpd.scaled_piv ) {
                if ((err_msg = 
                     gpiv_fwrite_hdf5_scdata(fname_out, 
                                             &display_act->gpd.
                                             sstrain_data_scaled, 
                                             "SHEAR_STRAIN",
                                             VERSION))
                   != NULL) error_gpiv ("%s", err_msg);
            } else {
                if ((err_msg = 
                     gpiv_fwrite_hdf5_scdata(fname_out, 
                                             &display_act->gpd.sstrain_data, 
                                             "SHEAR_STRAIN",
                                             VERSION))
                    != NULL) error_gpiv ("%s", err_msg);
            }
            display_act->gpd.saved_sstrain = TRUE;
        }
        
        
        if (display_act->gpd.exist_nstrain) {
            if (display_act->gpd.scaled_piv ) {
                if ((err_msg = 
                     gpiv_fwrite_hdf5_scdata(fname_out, 
                                             &display_act->gpd.
                                             nstrain_data_scaled, 
                                             "NORMAL_STRAIN",
                                             VERSION))
                    != NULL) error_gpiv ("%s", err_msg);
             } else {
                 if ((err_msg = 
                      gpiv_fwrite_hdf5_scdata(fname_out, 
                                              &display_act->gpd.nstrain_data, 
                                              "NORMAL_STRAIN",
                                              VERSION))
                     != NULL) error_gpiv ("%s", err_msg);
             }
            display_act->gpd.saved_nstrain = TRUE;
        }
        
        
        if ((err_msg = 
             gpiv_post_fwrite_hdf5_parameters (fname_out, 
                                               &display_act->gpd.piv_post_par))
            != NULL) error_gpiv ("%s", err_msg);
        
            

/*
 * Store all data in ASCII format at different files
 */
    } else {
        if (print_par) g_message("writing ASCII fmt");
        
/*
 * Image header
 */
        gpiv_io_make_fname(display_act->fname_base, GPIV_EXT_HEADER, fname_par);
        if ((fp_par = fopen(fname_par, "w")) == NULL) {
            error_gpiv(_("Failure opening image header for output"));
        }
 /*        fprintf(fp_par, "#%s\n", VERSION);
  */       gpiv_img_fprint_header(fp_par, display_act->img.image_par);
        fclose(fp_par);


/*
 * Parameters
 */
        gpiv_io_make_fname(display_act->fname_base, GPIV_EXT_PAR, fname_par);
        if ((fp_par = fopen(fname_par, "w")) == NULL) {
            error_gpiv(_("Failure opening parameter file for output"));
        }            
        fprintf(fp_par, "#%s\n", VERSION);
        gpiv_img_fprint_header(fp_par, display_act->img.image_par);


/*
 * Piv data
 */
        if (display_act->gpd.exist_piv) {
            gpiv_io_make_fname(display_act->fname_base, GPIV_EXT_PIV, 
                               fname_out);
            
            if (display_act->gpd.scaled_piv) {
                if ((err_msg = 
                     gpiv_fwrite_pivdata(fname_out, 
                                         &display_act->gpd.piv_data_scaled, 
                                         c_line, 
                                         0, 
                                         display_act->gpd.scaled_piv, 
                                         VERSION))
                    != NULL) error_gpiv ("%s", err_msg);
            } else {
                if ((err_msg = 
                     gpiv_fwrite_pivdata(fname_out, 
                                         &display_act->gpd.piv_data, 
                                         c_line, 
                                         0, 
                                         display_act->gpd.scaled_piv, 
                                         VERSION))
                    != NULL) error_gpiv ("%s", err_msg);
            }
            
            display_act->gpd.saved_piv = TRUE;
            gpiv_piv_fprint_parameters
                (fp_par, display_act->gpd.piv_eval_par);
            if (display_act->gpd.exist_valid) {
                gpiv_valid_fprint_parameters
                    (fp_par, display_act->gpd.piv_valid_par);
            }
        }


/*
 * (Eventual) histogram
 */
        if (display_act->gpd.exist_histo) {
            gpiv_io_make_fname(display_act->fname_base, GPIV_EXT_UVHISTO, 
                               fname_out);
            gpiv_fprint_histo(fname_out, 
                              &disp->gpd.klass, 
                              c_line, 
                              0,
                              display_act->gpd.scaled_piv, 
                              VERSION);
            display_act->gpd.saved_histo = TRUE;
         }


/*
 * (Eventual) scalar data
 */
        if (display_act->gpd.exist_vor) {
            gpiv_io_make_fname(display_act->fname_base, GPIV_EXT_VOR, 
                               fname_out);

            if (display_act->gpd.exist_vor_scaled) {
                if ((err_msg = 
                     gpiv_fwrite_scdata(fname_out, 
                                        &display_act->gpd.vor_data_scaled, 
                                        c_line, 
                                        nc_lines, 
/*                                         display_act->gpd.scaled_piv,  */
                                        display_act->gpd.exist_vor_scaled,
                                        VERSION))
                    != NULL) error_gpiv ("%s", err_msg);
            } else {
                if ((err_msg = 
                     gpiv_fwrite_scdata(fname_out, 
                                        &display_act->gpd.vor_data, 
                                        c_line, 
                                        nc_lines, 
                                        display_act->gpd.scaled_piv, 
                                        VERSION))
                    != NULL) error_gpiv ("%s", err_msg);
            }
            display_act->gpd.saved_vor = TRUE;

            if (flag_vor == 0) {
                gpiv_post_print_parameters(display_act->gpd.piv_post_par);
                gpiv_post_fprint_parameters(fp_par, 
                                            display_act->gpd.piv_post_par);
                flag_vor = 1;
            }
        }
        

        if (display_act->gpd.exist_sstrain) {
            gpiv_io_make_fname(display_act->fname_base, GPIV_EXT_SSTR, 
                               fname_out);
            if (display_act->gpd.exist_sstrain_scaled) {
                if ((err_msg = 
                     gpiv_fwrite_scdata(fname_out, 
                                        &display_act->gpd.sstrain_data_scaled, 
                                        c_line, 
                                        nc_lines, 
/*                                         display_act->gpd.scaled_piv, */
                                        display_act->gpd.exist_sstrain_scaled,
                                        VERSION)) 
                    != NULL) error_gpiv ("%s", err_msg);
             } else {
                if ((err_msg = 
                     gpiv_fwrite_scdata(fname_out, 
                                        &display_act->gpd.sstrain_data, 
                                        c_line, 
                                        nc_lines, display_act->gpd.scaled_piv,
                                        VERSION)) 
                    != NULL) error_gpiv ("%s", err_msg);
            }
            display_act->gpd.saved_sstrain = TRUE;
            
            if (flag_vor == 0) {
                gpiv_post_fprint_parameters(fp_par, 
                                            display_act->gpd.piv_post_par);
                flag_vor = 1;
            }
        }
        

        if (display_act->gpd.exist_nstrain) {
            gpiv_io_make_fname(display_act->fname_base, GPIV_EXT_NSTR, 
                               fname_out);
            if (display_act->gpd.exist_nstrain_scaled) {
                if ((err_msg = 
                     gpiv_fwrite_scdata(fname_out, 
                                        &display_act->gpd.nstrain_data_scaled, 
                                        c_line, 
                                        nc_lines, 
                                        display_act->gpd.exist_nstrain_scaled, 
                                        VERSION)) 
                    != NULL) error_gpiv ("%s", err_msg);
            } else {
                if ((err_msg = 
                     gpiv_fwrite_scdata(fname_out, 
                                        &display_act->gpd.nstrain_data, 
                                        c_line, 
                                        nc_lines, 
                                        display_act->gpd.scaled_piv, 
                                        VERSION)) 
                    != NULL) error_gpiv ("%s", err_msg);
            }
            display_act->gpd.saved_nstrain = TRUE;
            
            if (flag_vor == 0) {
                gpiv_post_fprint_parameters(fp_par, 
                                            display_act->gpd.piv_post_par);
                flag_vor = 1;
            }
        }
        
        fclose(fp_par);
    }
}




static void 
file_ok_sel(GtkWidget * widget, 
            GtkFileSelection * fs)
/* ----------------------------------------------------------------------------
 * image/PIV-data file selection
 */
{
    gchar *fname_in; 
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(fs), "gpiv");


    gtk_clist_set_selection_mode(GTK_CLIST (gpiv->clist_buf),
                                 GTK_SELECTION_SINGLE);
    fname_in = g_strdup(gtk_file_selection_get_filename 
                        (GTK_FILE_SELECTION (fs)));
    select_action_from_fname(gpiv, fname_in);
    g_free(fname_in);
}



/*
 * General, program-wide public, functions (no callbacks)
 */
void
load_buffer(GpivConsole *gpiv,
            char *fname_in,
            char *fname_base,
            char *fname_ext,
            int ibuf, 
            enum ClistPut clist_put)
/*-----------------------------------------------------------------------------
 * create display and its (popup) menu, load image, piv data and puts 
 * buffername into clist
 */
{
    gint return_val = 0;
    gchar *clist_buf_txt[MAX_BUFS][2], cl_int[2];
    gboolean local_stretch_window = gpiv_par.stretch_window;

    gint new_width = 0;
    gint new_height = 0;


    display[ibuf] = create_display(fname_ext, ibuf, gpiv);
    display_act = display[ibuf];
/*BUGFIX:: disabled 25.07.2005*/
     gtk_widget_show(display_act->mwin);
/*      g_message("load_buffer:: zoom_index=%d", display_act->zoom_index); */
/*     zoom_display(display_act, display_act->zoom_index); */


/*
 * Load image data and eventually existing PIV data
 */
    g_snprintf(display_act->fname_base, GPIV_MAX_CHARS, "%s", fname_ext);
    if((return_val = open_img(fname_in, &display_act->img)) 
       != 0) {
        gtk_object_destroy(GTK_OBJECT(display[ibuf]->mwin));
        if (ibuf > 0) {
            ibuf = nbufs - 1;;
            display_act = display[ibuf];
        } else {
            display_act = NULL;
        }
        nbufs--;
        return;
    }
    open_piv(fname_in, &display_act->gpd);

/* 
 * Loading image data
 * Suppress creating intregs during load_buffer by toggle menu;
 * after displaying results into correct effect
 */
    view_toggle_intregs_flag = FALSE;
    gpiv_par.stretch_window = TRUE;
/*      g_message("load_buffer:: create_display_menu"); */
    display_act->display_menu = create_display_menu(display_act);
    gtk_widget_show(display_act->display_menu);
    gtk_signal_connect_object(GTK_OBJECT(display_act->mwin), 
                              "button_press_event",
                              GTK_SIGNAL_FUNC (on_my_popup_handler), 
                              GTK_OBJECT(display_act->display_menu));
    gpiv_par.stretch_window = local_stretch_window;

/*
 * Displaying
 */
    create_background(display_act);
    create_img(display_act);
    if (display_act->display_img1) {
        show_img1(display_act);
    } else {
        hide_img1(display_act);
    }

    if (image_par.x_corr) {
        if (display_act->display_img2) {
            show_img2(display_act);
        } else {
            hide_img2(display_act);
        }
    }

    if (display_act->display_piv) {
        create_all_vectors(&display_act->gpd);
    }

    display_act->img.img_mean = image_mean(display_act->img.img1,
                                           gpiv_par.img_width, 
                                           gpiv_par.img_height);


    if (gpiv_par.display_intregs == 1) {
        show_all_intregs(display_act);
    }
    view_toggle_intregs_flag = TRUE;

/*
 * Add buffer to clist
 */
    g_snprintf(cl_int, 2, "%d", ibuf);
    clist_buf_txt[ibuf][0] = (gchar *) cl_int;
    clist_buf_txt[ibuf][1] = fname_base;
    if (clist_put == PREPEND) {
        gtk_clist_prepend(GTK_CLIST(gpiv->clist_buf), 
                          clist_buf_txt[ibuf]);
    } else if (clist_put == INSERT) {
        gtk_clist_insert(GTK_CLIST(gpiv->clist_buf), ibuf, 
                         clist_buf_txt[ibuf]);
    } else if (clist_put == APPEND) {
        gtk_clist_append(GTK_CLIST(gpiv->clist_buf), 
                         clist_buf_txt[ibuf]);
    } else {
        error_gpiv("non-existent CLIST enumerate");
    }
    
    gtk_clist_set_row_data(GTK_CLIST(gpiv->clist_buf), ibuf, 
                           display_act);
 
/* BUGFIX:: */
/*BUGFIX::disabled 25.07.2005 */
/*     zoom_display(display_act, display_act->zoom_index); */

/*     new_width = (gint) (display_act->zoom_factor * gpiv_par.img_width */
/*                         + VIEW_HMARGE */
/*                         ); */
/*     new_height = (gint) (display_act->zoom_factor * gpiv_par.img_height */
/*                          + VIEW_VMARGE */
/*                          ); */
/*     g_message("load_buffer:::: wi=%d hi=%d", new_width, new_height); */
/*     gtk_window_resize(GTK_WINDOW (display_act->mwin), new_width, new_height); */

    gtk_widget_set_size_request (display_act->canvas, 
                                 (gint) (display_act->zoom_factor * 
                                         gpiv_par.img_width), 
                                 (gint) (display_act->zoom_factor * 
                                         gpiv_par.img_height));
    
}



void
close_buffer(GpivConsole *gpiv,
             Display *disp,
             gint *ibuf
             )
/*-----------------------------------------------------------------------------
 * closes all data of a display. Checks if they have been saved
 */
{
    char message[2 * GPIV_MAX_CHARS];
    GtkDialog *close_buffer_dialog;
    gint i;

    if (display[*ibuf] != NULL) {
        if(!disp->gpd.saved_piv
           || !disp->gpd.saved_histo
           || !disp->gpd.saved_vor
           || !disp->gpd.saved_nstrain
           || !disp->gpd.saved_sstrain) {
            g_snprintf(message, 2 * GPIV_MAX_CHARS,
                       _("There are unsaved data that will be lost.\n\
Are you sure you want to close buffer #%d ?"), 
                       *ibuf);
            close_buffer_dialog = create_close_buffer_dialog (gpiv, message);
            gtk_widget_show (GTK_WIDGET(close_buffer_dialog));
        } else {
            free_all_bufmems(disp);    
            gtk_widget_destroy(GTK_WIDGET (disp->mwin));
            display[*ibuf] = NULL;
        }
    }
}



void
point_to_existbuffer(GpivConsole * gpiv
                     )
/*-----------------------------------------------------------------------------
 * Point display_act to an existing buffer
 */
{
    gint i;

    for (i = 0; i < nbufs; i++) {
        if(display[i] != NULL) {
            gtk_clist_select_row(GTK_CLIST(gpiv->clist_buf), i, 0);
            display_act = display[i];
        }
    }
}

/*
 * Callback functions for pop-up menus
 */

gint
on_my_popup_handler(GtkWidget *widget, GdkEvent *event)
/*-----------------------------------------------------------------------------
 */
 {
   GtkMenu *menu;
   GdkEventButton *event_button;

   g_return_val_if_fail (widget != NULL, FALSE);
   g_return_val_if_fail (GTK_IS_MENU (widget), FALSE);
   g_return_val_if_fail (event != NULL, FALSE);

   /* The "widget" is the menu that was supplied when 
    * gtk_signal_connect_object was called.
    */
   menu = GTK_MENU (widget);

   if (event->type == GDK_BUTTON_PRESS)
     {
       event_button = (GdkEventButton *) event;
       if (event_button->button == 3)
         {
           gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 
                           event_button->button, event_button->time);
           return TRUE;
         }
     }

   return FALSE;
 }





/*
 * Main gpiv-gui callbacks
 */


void
delete_console(GtkWidget *widget,
               GdkEvent  *event,
               gpointer   data)
/*-----------------------------------------------------------------------------
 */
{
/*     GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv"); */
    gint ibuf;
    GtkDialog *gpiv_exit;

#ifndef DISABLE_DAC
    exec_trigger_stop();
    exec_rec_stop();
#endif /* DISABLE_DAC */

    if (nbufs > 0) {
        for (ibuf = 0; ibuf < nbufs; ibuf++) {
            if (display[ibuf] != NULL) display_act = display[ibuf];
            
            if(!display_act->gpd.saved_piv
               || !display_act->gpd.saved_histo
               || !display_act->gpd.saved_vor
               || !display_act->gpd.saved_nstrain
               || !display_act->gpd.saved_sstrain) {
                gpiv_exit = create_exit_dialog (/* gpiv->console */);
                gtk_widget_show_all (GTK_WIDGET(gpiv_exit));
            } else {
                free_all_bufmems(display_act);
                gtk_main_quit();
            }
            
        }
        
    } else {
        gtk_main_quit();
    }
}



void
on_clist_buf_rowselect(GtkWidget * clist,
		       gint row,
		       gint column, 
                       GdkEventButton * event, 
                       gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    gchar * text, labelno[GPIV_MAX_CHARS];
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(clist), "gpiv");
    GList *lis; 
    gint cnt = 0, buf_dum = 0;
        
    if (!exec_process && GTK_CLIST (clist)->selection) { 
        for (lis = GTK_CLIST (clist)->selection; lis; lis = lis->next) { 
            row = GPOINTER_TO_INT (lis->data); 
            gtk_clist_get_text(GTK_CLIST(clist), row, /* column */ 0, &text);
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                                 row); 
            
            if(cnt == 0) {
                gpiv->first_selected_row = row;
            } else {
                gpiv->last_selected_row = row;
            }
/*
 * update variables of display before leaving the focus
 */
            if (display_act->intreg.exist_int) {
                display_act->intreg.row_start_old = 0;
                display_act->intreg.row_start = piv_eval_par.row_start;
                display_act->intreg.row_end = piv_eval_par.row_end;
                display_act->intreg.col_start_old = 0;
                display_act->intreg.col_start = piv_eval_par.col_start;
                display_act->intreg.col_end = piv_eval_par.col_end;
                display_act->intreg.int_size_1 = piv_eval_par.int_size_1;
                display_act->intreg.int_size_2 = piv_eval_par.int_size_2;
                display_act->intreg.int_shift = piv_eval_par.int_shift;
                display_act->intreg.pre_shift_row = piv_eval_par.pre_shift_row;
                display_act->intreg.pre_shift_col = piv_eval_par.pre_shift_col;
            }


/* if (GTK_WIDGET_REALIZED(display_act->gpd.display)) */

            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                         row); 
/*
 * Set variables of new active display equal to parameters
*/
            if (display_act->intreg.exist_int
                && (display_act->intreg.row_start != piv_eval_par.row_start
                    || display_act->intreg.row_end != piv_eval_par.row_end
                    || display_act->intreg.col_start != piv_eval_par.col_start
                    || display_act->intreg.col_end != piv_eval_par.col_end
                    || display_act->intreg.int_size_1 != piv_eval_par.int_size_1
                    || display_act->intreg.int_size_2 != piv_eval_par.int_size_2
                    || display_act->intreg.int_shift != piv_eval_par.int_shift
                    || display_act->intreg.pre_shift_row != piv_eval_par.pre_shift_row
                    || display_act->intreg.pre_shift_col != piv_eval_par.pre_shift_col)
                ) {
                destroy_all_intregs(display_act);
                create_all_intregs(display_act);
            }

            if (display_act->mwin != NULL 
                && GTK_WIDGET_VISIBLE(GTK_WIDGET(display_act->mwin)) 
                ) { 
                gdk_window_show(GTK_WIDGET(display_act->mwin)->window);
                gdk_window_raise(GTK_WIDGET(display_act->mwin)->window);
                
/*
 * update labels in image header tab/window
 */
                g_snprintf(labelno, GPIV_MAX_CHARS,"%d", display_act->id);
                gtk_label_set_text(GTK_LABEL(gpiv->imgh->label_bufno), 
                                   (char *) labelno);
                gtk_label_set_text(GTK_LABEL(gpiv->imgh->label_name), 
                                   display_act->fname_base);
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_imgtitle),  */
/*                                    image_par.title); */
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_crdate),  */
/*                                    image_par.creation_date); */
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_location),  */
/*                                    image_par.location); */
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_author),  */
/*                                    image_par.author); */
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_software),  */
/*                                    image_par.software); */
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_source),  */
/*                                    image_par.source); */
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_usertext),  */
/*                                    image_par.usertext); */
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_warning),  */
/*                                    image_par.warning); */
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_disclaimer),  */
/*                                    image_par.disclaimer); */
/*                 gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_comment),  */
/*                                    image_par.comment); */

                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_imgtitle), 
                                   display_act->img.image_par.title);
                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_crdate), 
                                   display_act->img.image_par.creation_date);
                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_location), 
                                   display_act->img.image_par.location);
                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_author), 
                                   display_act->img.image_par.author);
                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_software), 
                                   display_act->img.image_par.software);
                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_source), 
                                   display_act->img.image_par.source);
                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_usertext), 
                                   display_act->img.image_par.usertext);
                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_warning), 
                                   display_act->img.image_par.warning);
                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_disclaimer), 
                                   display_act->img.image_par.disclaimer);
                gtk_entry_set_text(GTK_ENTRY(gpiv->imgh->entry_comment), 
                                   display_act->img.image_par.comment);
/*     gtk_widget_grab_focus (display_act); */
            }


            cnt++; 
        }

        if (cnt == 1) {
            gpiv->last_selected_row = gpiv->first_selected_row;
        }
        
        if (gpiv->last_selected_row < gpiv->first_selected_row) {
            buf_dum = gpiv->last_selected_row;
            gpiv->last_selected_row = gpiv->first_selected_row;
            gpiv->first_selected_row = buf_dum;
        }
        
    }
/*     g_message("on_clist_buf_rowselect:: nbufs = %d, first=%d last = %d", */
/*               nbufs, gpiv->first_selected_row, gpiv->last_selected_row); */

}



void  
on_clist_buf_drag_data_received  (GtkWidget          *widget,
                                  GdkDragContext     *context,
                                  gint                x,
                                  gint                y,
                                  GtkSelectionData   *data,
                                  guint               info,
                                  guint               time)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    char *fname_cnt;
    gchar *fname_in; 
    gchar **fname_v;
    gint i, count = 0;

    g_print("Got: %s\n", data->data);

    fname_v = g_strsplit(data->data, "file:", MAX_BUFS);

/*
 * Count the number of selected files
 */
    fname_cnt = strtok(data->data, ":");
    while (fname_cnt != NULL && count < MAX_BUFS) {
        fname_cnt = strtok(NULL, ":");
        count++;
    }

/*
 * load the data from selected files
 */
    for (i = 1; i <= count - 1; i++) {
        fname_in = g_strdup(fname_v[i]);
        g_strchomp (fname_in);
        select_action_from_fname(gpiv, fname_in);
        g_free(fname_in);
    }

    g_strfreev(fname_v);
}


void 
on_open_activate(GtkMenuItem * menuitem, 
                 gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(menuitem), "gpiv");
    GtkWidget *filew;
/*     GtkWidget *dialog; */

/*     dialog = gtk_file_chooser_dialog_new ("Open File", */
/*
 * Disable
 */
/*                                           GTK_WINDOW (gpiv->console), */
/*                                           NULL, */
/*                                           GTK_FILE_CHOOSER_ACTION_OPEN, */
/*                                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, */
/*                                           GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, */
/*                                           NULL); */

/*     if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) { */
/*         file_ok_sel(gpiv, dialog); */
/*     } */

/*     gtk_widget_destroy (dialog); */


    filew = gtk_file_selection_new("gpiv: open image");
    gtk_signal_connect(GTK_OBJECT(filew), "destroy",
		       (GtkSignalFunc) destroy, &filew);
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filew)->ok_button),
		       "clicked", (GtkSignalFunc) file_ok_sel,
		       GTK_OBJECT(filew));
    gtk_signal_connect_object(GTK_OBJECT
			      (GTK_FILE_SELECTION(filew)->ok_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(filew));
    gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filew)->ok_button), 
                        "gpiv",
                        gpiv);
    gtk_signal_connect_object(GTK_OBJECT
			      (GTK_FILE_SELECTION(filew)->cancel_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(filew));
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(filew),
				    gpiv_var.fname_last);
    gtk_widget_show(filew);
    gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filew)), 
                        "gpiv",
                        gpiv);

}



void 
on_save_activate(GtkMenuItem * menuitem, 
                 gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(menuitem), "gpiv");
    save_all_data(gpiv);
}


void 
on_save_as_activate_choose(GtkMenuItem * menuitem, 
                           gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(menuitem), "gpiv");
    GtkWidget *dialog;

    dialog = gtk_file_chooser_dialog_new ("Save as",
/*                                           gpiv->console, */
                                          NULL,
                                          GTK_FILE_CHOOSER_ACTION_SAVE,
                                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                          GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                                          NULL);
    
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
        file_saveas_ok_sel(gpiv, dialog);
    }
     
    gtk_widget_destroy (dialog);
}



void 
on_save_as_activate(GtkMenuItem * menuitem, 
                    gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GtkWidget *filew;
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(menuitem), "gpiv");


/* 
 * Create a new file selection widget
 * Connect the ok_button and cancel_button to file_saveas_ok_sel function
 * set user data to filew
 * Lets set a default filename
 */
    filew = gtk_file_selection_new("File selection");
    gtk_signal_connect(GTK_OBJECT(filew), "destroy",
		       (GtkSignalFunc) destroy, &filew);
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filew)->ok_button),
		       "clicked", (GtkSignalFunc) file_saveas_ok_sel,
		       GTK_OBJECT(filew));
    gtk_signal_connect_object(GTK_OBJECT
			      (GTK_FILE_SELECTION(filew)->ok_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(filew));
    gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filew)->ok_button), 
                        "gpiv",
                        gpiv);
    gtk_signal_connect_object(GTK_OBJECT
			      (GTK_FILE_SELECTION(filew)->cancel_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(filew));
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(filew),
				    gpiv_var.fname_last);
    gtk_widget_show(filew);
    gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filew)), 
                        "gpiv",
                        gpiv);

}



void
file_saveas_ok_sel(GpivConsole *gpiv, 
                   GtkWidget *dialog
                   )
/* ----------------------------------------------------------------------------
 * PIV data file selection
 */
{
       gchar *fname_in, 
            *dirname,
            *fname_base, 
            *fname_ext;

        gchar *clist_buf_txt[MAX_BUFS][2], cl_int[2];

        fname_in = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
        fname_base = g_path_get_basename(fname_in);
        dirname = g_path_get_dirname(fname_in);
        strtok(fname_base, ".");
        fname_ext = g_strdup(g_strconcat(dirname, G_DIR_SEPARATOR_S,
                                         fname_base, NULL));

/*
 * Substituting fname_base and storing data of display_act
 */
        gtk_clist_remove(GTK_CLIST(gpiv->clist_buf), display_act->id);
        g_snprintf(cl_int, 2, "%d", display_act->id);
        g_snprintf(display_act->fname_base, GPIV_MAX_CHARS, "%s", fname_ext);
        clist_buf_txt[display_act->id][0] = (gchar *) cl_int;
        clist_buf_txt[display_act->id][1] = fname_base;
        gtk_clist_insert(GTK_CLIST(gpiv->clist_buf), display_act->id, 
                         clist_buf_txt[display_act->id]);
        
        g_snprintf(display_act->msg_display_default, GPIV_MAX_CHARS, "%s",
                   display_act->fname_base);
        gnome_appbar_set_default(GNOME_APPBAR(display_act->appbar),
                                 display_act->msg_display_default);
        exec_save(display_act);


        g_free(fname_in);
        g_free(dirname);
        g_free(fname_base);
        g_free(fname_ext);
}



void
save_all_data(GpivConsole * gpiv)
/*-----------------------------------------------------------------------------
 */
{
    gint row, ibuf;

    if (nbufs > 0) {
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                                 row); 
            ibuf = display_act->id;
            if (display[ibuf] != NULL
                && display_act->mwin != NULL
                && GTK_WIDGET_VISIBLE(GTK_WIDGET(display_act->mwin)) ) { 
                gdk_window_show(GTK_WIDGET(display_act->mwin)->window);
                gdk_window_raise(GTK_WIDGET(display_act->mwin)->window);
            }

            exec_save(display_act);

        }
    }
}



void 
on_print_activate(GtkMenuItem * menuitem, 
                    gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(menuitem), "gpiv");
    GtkDialog *gpiv_print_dialog = create_print_dialog(gpiv);
    gtk_widget_show(GTK_WIDGET (gpiv_print_dialog));
}



void 
on_execute_activate(GtkMenuItem * menuitem, 
                    gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(menuitem), "gpiv");
    gint row, ibuf;

    cancel_process = FALSE;

#ifndef DISABLE_DAC
    if (/* !cancel_process && */ gpiv_par.process_dac) {
        exec_rec_start(gpiv);
        if (dac_par.ttime.mode == GPIV_TIMER_MODE__DURATION) {
            gpiv->first_selected_row = 0;
            gpiv->last_selected_row = dac_par.ttime.cycles - 1;
        }
    }
#endif /* DISABLE_DAC */


    if (nbufs > 0) {
        exec_process = TRUE;
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                                 row); 
            ibuf = display_act->id;
            if (display[ibuf] != NULL
                && GTK_WIDGET_VISIBLE(GTK_WIDGET(display_act->mwin)) ) { 
                gdk_window_show(GTK_WIDGET(display_act->mwin)->window);
                gdk_window_raise(GTK_WIDGET(display_act->mwin)->window);
            }
            
            if (!cancel_process && gpiv_par.process_piv) {
                exec_piv(gpiv);
            }

            if (!cancel_process && gpiv_par.process_gradient) {
                exec_gradient();
            }

            if (!cancel_process && gpiv_par.process_resstats) {
                piv_valid_par.res_stats = 1;
                exec_errvec(gpiv->pivvalid);
            }

            if (!cancel_process && gpiv_par.process_errvec) {
                piv_valid_par.res_stats = 0;
                exec_errvec(gpiv->pivvalid);
            }

            if (!cancel_process && gpiv_par.process_peaklock) {
                exec_peaklock(gpiv->pivvalid);
            }

            if (!cancel_process && gpiv_par.process_scale) {
                exec_scale(gpiv->pivpost);
            }

            if (!cancel_process && gpiv_par.process_average) {
                exec_savg(gpiv->pivpost);
            }

            if (!cancel_process && gpiv_par.process_subtract) {
                exec_subavg(gpiv->pivpost);
            }

            if (!cancel_process && gpiv_par.process_vorstra) {
                exec_vorstra();
            }

        }

        exec_process = FALSE;
    }
}


void 
on_stop_activate(GtkMenuItem * menuitem, 
                    gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
/*     warning_gpiv(_("process has been stopped")); */
}


void
on_button_stop_press (GtkWidget * widget, 
                      gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    cancel_process = TRUE;
    message_gpiv(_("cancel_process = TRUE"));
}


void
on_button_stop_release (GtkWidget * widget, 
                        gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    cancel_process = FALSE;
    message_gpiv(_("cancel_process = FALSE"));
/* process has been stopped */
}


/* static gboolean  */
/* on_display_delete(GtkWidget *widget, */
/*                   GdkEvent  *event, */
/*                   gpointer   data) */
/* { */
/*     return FALSE; */
/* } */


void
on_close_activate(GtkMenuItem * menuitem, 
                    gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(menuitem), "gpiv");
    gchar *err_msg;
    gint row, ibuf;

/*     g_message("on_close_activate:: nbufs = %d, first=%d last = %d", */
/*               nbufs, gpiv->first_selected_row, gpiv->last_selected_row); */
#ifndef DISABLE_DAC
    if ((err_msg = gpiv_dac_free_camvar(&cam_var)) != NULL) {
        error_gpiv("from: on_close_activate\n from: gpiv_dac_free_dacvar\n %s", 
                  err_msg);
    }
#endif /* DISABLE_DAC */

    if (nbufs > 0) {
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf),
                                                 row); 
            ibuf = display_act->id;
            close_buffer(gpiv, display[ibuf], &ibuf);
        }       

/*
 * Cleaning up clist
 */
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data
                (GTK_CLIST(gpiv->clist_buf), gpiv->first_selected_row); 
            ibuf = display_act->id;
            if (display[ibuf] == NULL) {
              gtk_clist_remove(GTK_CLIST(gpiv->clist_buf), 
                                 gpiv->first_selected_row);
                nbufs--;
            } else {
                gpiv->first_selected_row++;
            }
       }

    }

    point_to_existbuffer(gpiv);
}



void 
on_exit_activate(GtkMenuItem * menuitem,
                 gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GtkDialog *gpiv_exit;
    gint ibuf;

#ifndef DISABLE_DAC
    exec_trigger_stop();
    exec_rec_stop();
/*     dc1394_stop_iso_transmission(cam_var.handle, cam_var.camera[0].node); */
/*     dc1394_release_camera(cam_var.handle, &cam_var.camera[0]);     */
    raw1394_destroy_handle(cam_var.handle);
#endif /* DISABLE_DAC */

    if (nbufs > 0) {
        for (ibuf = 0; ibuf < nbufs; ibuf++) {
            if (display[ibuf] != NULL) display_act = display[ibuf];

            if(!display_act->gpd.saved_piv
               || !display_act->gpd.saved_histo
               || !display_act->gpd.saved_vor
               || !display_act->gpd.saved_nstrain
               || !display_act->gpd.saved_sstrain) {
                gpiv_exit = create_exit_dialog ();
                gtk_widget_show (GTK_WIDGET(gpiv_exit));
            } else {
                free_all_bufmems(display_act);
                gtk_main_quit();
            }

        }
        
    } else {
        gtk_main_quit();
    }
}


void on_preferences_activate(GtkWidget * widget,
/*                              GtkMenuItem * menuitem,  */
                             gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    GtkDialog *preferences = NULL ; 

    preferences = create_preferences(gpiv);
}



void on_about_activate(GtkMenuItem * menuitem, 
                       gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GtkWidget *about;
    about = create_about();

}




void on_manual_activate(GtkMenuItem * menuitem, 
                        gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GError* error = NULL;

    if (!gnome_help_display("index.html", NULL, &error)) {
        g_warning("%s", error);
	g_error_free (error);
    }


}

/*
 * Gnome toolbar buttons
 */


void 
on_button_open_clicked(GtkButton * button, 
                       gpointer data)
/*-----------------------------------------------------------------------------
 * Uses gtk2
 */
{
    gchar *msg = "Opens PIV data";
 /*    GtkWidget *dialog; */
    GtkWidget *filew;
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(button), "gpiv");

/*     dialog = gtk_file_chooser_dialog_new ("Open File", */
/*
 * Disable console
 */
/*                                           gpiv->console, */
/*                                           NULL, */
/*                                           GTK_FILE_CHOOSER_ACTION_OPEN, */
/*                                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, */
/*                                           GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, */
/*                                           NULL); */
    
/*     if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) { */
/*         file_ok_sel(gpiv, dialog); */
/*     } */
    
/*     gtk_widget_destroy (dialog); */
/*     gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);     */

/* 
 * Create a new file selection widget
 * Connect the ok_button and cancel_button to file_ok_sel function
 * Set a default filename
 * set user data "gpiv" to filew
 */
    filew = gtk_file_selection_new(_("File selection"));
    gtk_signal_connect(GTK_OBJECT(filew), "destroy",
		       (GtkSignalFunc) destroy, &filew);
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filew)->ok_button),
		       "clicked", (GtkSignalFunc) file_ok_sel,
		       GTK_OBJECT(filew));
    gtk_signal_connect_object(GTK_OBJECT
			      (GTK_FILE_SELECTION(filew)->ok_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(filew));
    gtk_signal_connect_object(GTK_OBJECT
			      (GTK_FILE_SELECTION(filew)->cancel_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(filew));
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(filew), 
                                    /* DEFAULT_TEST_IMAGE */ 
                                    gpiv_var.fname_last);
    gtk_widget_show(filew);
    gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filew)), 
                        "gpiv",
                        gpiv);


    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_menubar_activate(GtkWidget * widget, 
                    gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    if (GTK_CHECK_MENU_ITEM(widget)->active) {
	 gtk_widget_show(gpiv->menubar);
    } else {
	 gtk_widget_hide(gpiv->menubar);
    }
}



void 
on_toolbuttons_activate(GtkWidget * widget, 
                        gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    if (GTK_CHECK_MENU_ITEM(widget)->active) {
	 gtk_widget_show(gpiv->toolbar1);
/*    gtk_widget_ref(settings_menu_gpiv_popup[1].widget); */
/* 	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM */
/* 				       (settings_menu_gpiv_popup[1].widget),  */
/*                                        TRUE); */
    } else {
	 gtk_widget_hide(gpiv->toolbar1);
/* 	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM */
/* 				       (settings_menu_gpiv_popup[1].widget),  */
/*                                        FALSE); */
    }

}



void 
on_gpivbuttons_activate(GtkWidget * widget, 
                        gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    if (GTK_CHECK_MENU_ITEM(widget)->active) {
	 gtk_widget_show(gpiv->handlebox1);

    } else {
	 gtk_widget_hide(gpiv->handlebox1);
    }
}



void 
on_tabulator_activate(GtkWidget * widget, 
                      gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    if (GTK_CHECK_MENU_ITEM(widget)->active) {
	 gtk_widget_show(gpiv->notebook);
    } else {
        gtk_widget_hide(gpiv->notebook);
    }

}



void 
on_tooltip_activate(GtkWidget * widget, 
                    gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    if (GTK_CHECK_MENU_ITEM(widget)->active) {
	 gtk_tooltips_enable(gpiv->tooltips );
    } else {
	 gtk_tooltips_disable(gpiv->tooltips );
    }


}



void
on_buffer_set_focus(GtkWindow * window,
		    GtkWidget * widget, 
                    gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
/* SEE: on_clist_buf_rowselect */
}



void 
gtk_window_destroy(GtkButton * button, 
                   gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{

}



void
on_appbar_display_user_response(GnomeAppBar * gnomeappbar,
			       gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{

}



void
on_button_open_enter(GtkContainer * container,
		     GtkDirectionType direction, 
                     gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(container), "gpiv");
    gchar *msg = _("Opens image/PIV data (and display)");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_save_enter(GtkContainer * container,
		     GtkDirectionType direction, 
                     gpointer user_data)
{/*-----------------------------------------------------------------------------
 */

    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(container), "gpiv");
    gchar *msg = _("Saves data");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_print_enter(GtkContainer * container,
		     GtkDirectionType direction, 
                     gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(container), "gpiv");
    gchar *msg = _("Prints selected buffer(s)");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_execute_enter(GtkContainer * container,
			GtkDirectionType direction, 
                        gpointer user_data)
{
/*-----------------------------------------------------------------------------
 */
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(container), "gpiv");
    gchar *msg = _("Executes all tickmarked processes");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_stop_enter(GtkContainer * container,
                     GtkDirectionType direction, 
                     gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(container), "gpiv");
    gchar *msg = _("Cancels any running processes");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_close_enter(GtkContainer * container,
                     GtkDirectionType direction, 
                     gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(container), "gpiv");
    gchar *msg = _("Close active buffer(s)");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void
on_button_exit_enter(GtkContainer * container,
		     GtkDirectionType direction, 
                     gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(container), "gpiv");
    gchar *msg = _("Exits GPIV");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



#ifndef DISABLE_DAC
void 
on_toolbar_checkbutton_dac(GtkWidget * widget, 
                           gpointer data)
/*-----------------------------------------------------------------------------
 * toolbar containing checkbutton for processes
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_dac = TRUE;
    } else {
	gpiv_par.process_dac = FALSE;
    }
}

#endif /* DISABLE_DAC */

void 
on_toolbar_checkbutton_piv(GtkWidget * widget, 
                           gpointer data)
/*-----------------------------------------------------------------------------
 * toolbar containing checkbutton for processes
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_piv = TRUE;
    } else {
	gpiv_par.process_piv = FALSE;
    }
}



void 
on_toolbar_checkbutton_gradient(GtkWidget * widget, 
                                gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_gradient = TRUE;
    } else {
	gpiv_par.process_gradient = FALSE;
    }
}



void 
on_toolbar_checkbutton_resstats(GtkWidget * widget, 
                                gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_resstats = TRUE;
        piv_valid_par.res_stats = 1;
    } else {
	gpiv_par.process_resstats = FALSE;
        piv_valid_par.res_stats = 0;
    }
}



void 
on_toolbar_checkbutton_errvec(GtkWidget * widget, 
                              gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_errvec = TRUE;
    } else {
	gpiv_par.process_errvec = FALSE;
    }
}



void 
on_toolbar_checkbutton_peaklck(GtkWidget * widget, 
                               gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_peaklock = TRUE;
    } else {
	gpiv_par.process_peaklock = FALSE;
    }
}



void 
on_toolbar_checkbutton_scale(GtkWidget * widget, 
                             gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_scale = TRUE;
    } else {
	gpiv_par.process_scale = FALSE;
    }
}



void 
on_toolbar_checkbutton_average(GtkWidget * widget, 
                               gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_average = TRUE;
    } else {
	gpiv_par.process_average = FALSE;
    }
}



void 
on_toolbar_checkbutton_subavg(GtkWidget * widget, 
                              gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_subtract = TRUE;
    } else {
	gpiv_par.process_subtract = FALSE;
    }
}



void 
on_toolbar_checkbutton_vorstra(GtkWidget * widget, 
                               gpointer data)
/*-----------------------------------------------------------------------------
 */
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
	gpiv_par.process_vorstra = TRUE;
    } else {
	gpiv_par.process_vorstra = FALSE;
    }
}




/* GpivData * gpd_active = &display_act->gpd; */

void 
on_button_quit_no_clicked(GtkButton * button, 
                          gpointer user_data)
/*-----------------------------------------------------------------------------
 * exit, message dialog callbacks
 */
{
/*     gnome_dialog_close(GTK_DIALOG(gpiv_exit)); */
/*
 * Gnome2:
 */
/*     gtk_widget_destroy(GTK_DIALOG(GTK_DIALOG(gpiv_exit))); */
}



void 
on_button_quit_gpiv_yes_clicked(GtkButton * button, 
                                gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    free_all_bufmems(display_act);
    gtk_main_quit();
}



/* void  */
/* on_button_message_clicked(GtkButton * button,  */
/*                           gpointer user_data) */
/* BUGFIX: obsolete function: on_button_message_clicked; clean up */
void
on_button_message_clicked(GtkDialog *dialog,
                            gint response,
                            gpointer data
                            )
/*-----------------------------------------------------------------------------
 */
{
    g_assert( response == GTK_RESPONSE_ACCEPT);

    switch (response) {
    case GTK_RESPONSE_ACCEPT:
/*         gnome_dialog_close(GNOME_DIALOG(gpiv_exit)); */
/*         gtk_widget_destroy(GTK_DIALOG(gpiv_exit)); */

    default:
        g_warning("on_message_clicked: should not arrive here");
        break;
    }
}



void
on_notebook_switch_page(GtkNotebook *notebook,
                        GtkNotebookPage *page,
                        gint page_num,
                        gpointer user_data)
/*-----------------------------------------------------------------------------
 */
{
    gpiv_var.tab_pos = page_num;

    gnome_config_push_prefix("/gpiv/RuntimeVariables/");
    gnome_config_set_int("tab_pos", gpiv_var.tab_pos);
    gnome_config_pop_prefix();
    gnome_config_sync();


}
