/******************************************************************
 *            gpe-expenses.c
 *
 *  Sun Nov 13 14:54:18 2005
 *  Copyright  2005  Neil Williams
 *  linux@codehelp.co.uk
 ******************************************************************/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _GNU_SOURCE
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <glib/gprintf.h>
#include <glib/gstdio.h>
#include <qof.h>
#include <gtk/gtk.h>
#include <gpe/init.h>
#include <gtk/gtkmain.h>
#include <gpe/pixmaps.h>
#include <gpe/pim-categories.h>
#include <gpe/errorbox.h>
#include <regex.h>
#include <popt.h>
#include <locale.h>
#include "qof-main.h"
#include "gpe-expenses.h"
#include "expenses-gtk.h"

/* used to print debug logs. */
static QofLogModule log_module = GPE_MOD_CLI;
#define EXPENSE_ICON PREFIX "/share/pixmaps/gpe-expenses.xpm"
#define GPE_EXPENSE_LOG "/tmp/gpe-expense.trace"
#define ARGUMENT_BAD_OPTION 17227
#define QOF_MOD_SQLITE "qof-sqlite-module"
#define SQLITE_DIR ".gpe/"
#define DEFAULT_FILE "expenses"
#define ACCESS_METHOD "sqlite"

#define ENUM_LIST_Q(_)  \
 	_(qof_op_noop, = 0) \
	_(qof_op_list,)     \
	_(qof_op_category,) \
	_(qof_op_time,) \
	_(qof_op_sql,)      \
	_(qof_op_sql_file,) \
	_(qof_op_write, )   \
	_(qof_op_explain,)  \
	_(qof_op_vers,)     \
	_(qof_op_compress,) \
	_(qof_op_debug,)    \
	_(qof_op_input, ) \
	_(qof_op_gui, )

	DEFINE_ENUM(qof_op_type, ENUM_LIST_Q)

GpeExpenseData*
gpe_expense_init (void)
{
	GpeExpenseData *context;

	qof_init();
	g_return_val_if_fail(ExpensesRegister (), NULL);
	context = g_new0(GpeExpenseData, 1);
	return context;
}

void
gpe_expense_close(GpeExpenseData *context)
{
	qof_main_free(&context->qof);
	qof_close();
	g_free(context);
}

void
gpe_expense_error (QofSession * session)
{
	if (qof_error_check (session))
		gpe_error_box (qof_error_get_message (session));
}

static struct gpe_icon my_icons[] = {
	{ "icon", EXPENSE_ICON, 0 },
  { NULL, NULL, NULL }
};

static void
gpe_gui_start(int argc, char *argv[], GpeExpenseData *context)
{
	g_return_if_fail(context);
	g_return_if_fail(gpe_application_init (&argc, &argv));
	g_return_if_fail(gpe_load_icons (my_icons));
	ENTER (" file=%s", context->qof.write_file);
	if(!context->qof.write_file)
	{
		gint test;
		gboolean gpe_home_exists;

		/* use a fixed file location. */
		test = 0;
		context->qof.write_file = g_strconcat (g_get_home_dir(), 
			"/", SQLITE_DIR, NULL);
		gpe_home_exists = g_file_test (context->qof.write_file, G_FILE_TEST_IS_DIR);
		if (!gpe_home_exists)
			test = g_mkdir (context->qof.write_file, 0700);
		g_free (context->qof.write_file);
		context->qof.write_file = g_strconcat (ACCESS_METHOD, 
			":", g_get_home_dir(), "/", SQLITE_DIR, DEFAULT_FILE, NULL);
		if (test)
			context->qof.write_file = g_strconcat (ACCESS_METHOD,
			":", g_get_tmp_dir(), "/", DEFAULT_FILE, NULL);
	}
	qof_session_begin(context->qof.input_session, 
		context->qof.write_file, TRUE, FALSE);
	gpe_expense_error (context->qof.input_session);
	qof_session_load(context->qof.input_session, NULL);
	context->book = qof_session_get_book(context->qof.input_session);
	gtk_set_locale ();
	gtk_init (&argc, &argv);

	open_expenses_window (context);

	gtk_main ();
	qof_session_save(context->qof.input_session, NULL);
	LEAVE (" ");
}

int
main (int argc, char *argv[])
{
	QOF_OP_VARS
	const gchar *help_header_text;
	GpeExpenseData *gpe_expense_context;
	gboolean debug_on;
	poptContext pc;
	gint optc;
	qof_op_type exp_command;

	struct poptOption options[] = {
		{ NULL, '\0', POPT_ARG_INCLUDE_TABLE, poptHelpOptionsI18N,
                        0, _("Help options:"), NULL },
		{"list", 'l', POPT_ARG_NONE, NULL, qof_op_list,
		 _("List all databases supported by the current QOF framework and exit."), NULL},
		{"explain", 0, POPT_ARG_NONE, NULL, qof_op_explain,
		 _("List the fields within the specified database and exit, requires -d."), NULL},
		{"input-file", 'i', POPT_ARG_STRING, &filename, qof_op_input,
		 _("Query the QSF XML data in <filename>"), _("filename")},
		{"date", 't', POPT_ARG_STRING, &date_time, qof_op_time,
		 _("Shorthand to only query objects that contain the specified date."), _("string")},
		{"sql", 's', POPT_ARG_STRING, &sql_query, qof_op_sql,
		 _("Specify a SQL query on the command line."), _("string")},
		{"sql-file", 'f', POPT_ARG_STRING, &sql_file, qof_op_sql_file,
		 _("Specify one or more SQL queries contained in a file."), _("filename")},
		{"write", 'w', POPT_ARG_STRING, &write_file, qof_op_write,
		 _("Write the results of any query to the file"), _("filename")},
		{"gui", 0, POPT_ARG_NONE, NULL, qof_op_gui,
		 _("Use the Gtk graphic interface"), NULL},
		{"debug", 0, POPT_ARG_NONE, NULL, qof_op_debug,
		 _("Print debugging information to a temporary file."), NULL},
		{"version", 0, POPT_ARG_NONE, NULL, qof_op_vers,
		 _("Display version information"), NULL},
		{"category", 'c', POPT_ARG_STRING, &category, qof_op_category,
		 _("Shorthand to only query objects that are set to the specified category."),
		 _("string")},
		POPT_TABLEEND
	};
	exp_command = qof_op_noop;
	debug_on = FALSE;
	QOF_OP_INIT
	database = g_strdup("gpe_expenses");

#ifdef ENABLE_NLS
	setlocale (LC_ALL, "");
	bindtextdomain (GETTEXT_PACKAGE, LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif
	help_header_text = _(
	/* Translators: please retain the line endings
	and punctuation. -i -l --gui or --explain are commands,
	options are as specified. Please retain 'or' in all cases,
	options and commands can only be combined in specific ways.
	*/
		"\n"
		"   Expenses applet for GPE using QOF - \n"
		"   the Query Object Framework.\n"
		"   Supports writing iPAQ data to SQLite.\n"
		"   SQL-type queries on the live data or SQLite file.\n"
		"   SQLite data can be imported into other QOF applications.\n\n"
		"   Use exactly one of -i -l --gui or --explain;\n"
		"   options are -c -t -w, -s or -f.\n\n");

	pc = poptGetContext (PACKAGE, argc, (const char **)argv, options, 0);

	poptSetOtherOptionHelp (pc, help_header_text);

	if (argc < 2)
	{
		poptPrintUsage (pc, stderr, 0);
		return EXIT_FAILURE;
	}
	gpe_expense_context = gpe_expense_init();
	g_return_val_if_fail (gpe_expense_context, 1);
	while ((optc = poptGetNextOpt (pc)) >= 0)
	{
		switch (optc)
		{
			/* commands - mutually exclusive */
			case qof_op_input:
			case qof_op_list:
			case qof_op_explain:
			case qof_op_gui:
			{
				if (qof_op_noop != exp_command)
				{
					fprintf (stderr, _("%s: ERROR: specify only one of"
						"-i, -l, --gui or --explain.\n"), PACKAGE);
					return 1;
				}
				exp_command = optc;
				break;
			}
			case qof_op_vers :
			{
				fprintf (stdout, "\n Copyright (c) 2005-2007 Neil Williams <linux@codehelp.co.uk>\n");
				fprintf (stdout, _(" For gpe-expenses support, join the QOF-devel mailing list at\n"));
				fprintf (stdout, " http://lists.sourceforge.net/mailman/listinfo/qof-devel\n");
				fprintf (stdout, _("\n This is gpe-expenses v%s\n"), VERSION);
				fprintf (stdout, _(" Expenses applet for GPE on iPAQ .\n"));
				/* Translators: Add or subtract dots to keep the translated lines aligned vertically */
				fprintf (stdout, _(" Build target.........: %s\n"), HOST_OS);
				/* Translators: Add or subtract dots to keep the translated lines aligned vertically */
				fprintf (stdout, _(" Build date...........: %s %s\n"), __DATE__, __TIME__);
				/* Translators: Add or subtract dots to keep the translated lines aligned vertically */
				fprintf (stdout, _(" --debug logs to......: %s\n\n"), GPE_EXPENSE_LOG);
				/* Translators: Add or subtract dots to keep the translated lines aligned vertically */
				fprintf (stdout, _(" Please use --help for more detailed options.\n\n"));
				return EXIT_SUCCESS;
			}
			/* optional modifiers - store to act on later. */
			case qof_op_category:
			{
				qof_mod_category (category, &gpe_expense_context->qof);
				break;
			}
			case qof_op_time:
			{
				qof_mod_time (date_time, &gpe_expense_context->qof);
				break;
			}
			case qof_op_sql:
			{
				qof_mod_sql (sql_query, &gpe_expense_context->qof);
				break;
			}
			case qof_op_sql_file:
			{
				qof_mod_sql_file (sql_file, &gpe_expense_context->qof);
				break;
			}
			case qof_op_write:
			{
				qof_mod_write (write_file, &gpe_expense_context->qof);
				break;
			}
			case qof_op_debug:
			{
				qof_log_init_filename(GPE_EXPENSE_LOG);
				qof_log_set_default(QOF_LOG_DETAIL);
				qof_log_set_level (GPE_MOD_CLI, QOF_LOG_DETAIL);
				qof_log_set_level (QOF_MAIN_CLI, QOF_LOG_DETAIL);
				qof_log_set_level (QOF_MOD_SQLITE, QOF_LOG_DETAIL);
				qof_log_set_level (GPE_MOD_GUI, QOF_LOG_DETAIL);
				debug_on = TRUE;
				break;
			}
			default:
			{
				fprintf (stderr, _("%s: ERROR: got option %d, arg %s\n"), PACKAGE,
					 optc, poptGetOptArg (pc));
				return EXIT_FAILURE;
			}
		}
	}
	if (optc < -1)
	{
		fprintf(stderr, "%s: %s %s\n\n", PACKAGE,
		poptBadOption(pc, POPT_BADOPTION_NOALIAS),
		poptStrerror(optc));
		poptPrintUsage(pc, stderr, 0);
		return EXIT_FAILURE;
	}
	if (gpe_expense_context->qof.error)
		return EXIT_FAILURE;
	/* If we get this far, we should have sensible options: start the work. */
	gpe_expense_context->qof.input_session = qof_session_new();
	switch (exp_command)
	{
		case qof_op_input:
		{
			gpe_expense_context->qof.filename = g_strdup(filename);
			/* despite the name, this is for any supported file */
			qof_cmd_xmlfile (&gpe_expense_context->qof);
			break;
		}
		case qof_op_list:
		{
			qof_cmd_list ();
			break;
		}
		case qof_op_gui :
		{
			gpe_gui_start(argc, argv, gpe_expense_context);
			break;
		}
		case qof_op_explain:
		{
			qof_mod_database (database, &gpe_expense_context->qof);
			if(!gpe_expense_context->qof.database) 
			{ 
				/* Translators: capitalise only the initial letter of error. */
				fprintf (stderr, _("%s: Error: please specify which "
					"database you would like explained.\n\n"), PACKAGE);
				break;
			}
			qof_cmd_explain(&gpe_expense_context->qof);
			break;
		}
		default:
		{
			/* should be impossible */
			break;
		}
	}
	poptFreeContext(pc);
	if(debug_on) { qof_log_shutdown(); }
	gpe_expense_close(gpe_expense_context);
	return EXIT_SUCCESS;
}
