# gozerbot/eventhandler.py
#
#

""" event handlers """

__copyright__ = 'this file is in the public domain'

from gozerbot.generic import handle_exception, rlog, lockdec
from gozerbot.thr import start_new_thread
import Queue, thread

handlerlock = thread.allocate_lock()
locked = lockdec(handlerlock)

class Eventhandler(object):

    """ events are handled in 11 queues with different priorities:
        queue0 is tried first queue10 last
    """

    def __init__(self):
        self.sortedlist = []
        self.queues = {}
        for i in range(11):
            self.queues[i] = Queue.Queue()
            self.sortedlist.append(i)
        self.sortedlist.sort()
        self.go = Queue.Queue()
        self.stopped = False
        self.running = False
        self.nooutput = False

    def start(self):
        """ start the eventhandler thread """
        self.stopped = False
        if not self.running:
            start_new_thread(self.handleloop, ())
            self.running = True

    def stop(self):
        """ stop the eventhandler thread """
        self.running = False
        self.stopped = True
        self.go.put('Yihaaa')

    def put(self, speed=10, *args):
        """ put item on the queue """
        self.queues[10-speed].put_nowait(args)
        self.go.put('go')

    def getready(self):
        """ check queues from available data """
        ready = []
        for i in self.sortedlist:
            if self.queues[i].qsize():
                ready.append(i)
        return ready

    def handle_one(self):
        """ do 1 loop over ready queues """
        ready = self.getready()
        for i in ready:
            self.dispatch(self.queues[i])

    def handleloop(self):
        """ thread that polls the queues for items to dispatch """
        rlog(0, 'eventhandler', 'starting handle thread')
        while not self.stopped:
            self.go.get()
            self.handle_one()
        rlog(10, 'eventhandler', 'stopping %s' % str(self))

    def dispatch(self, queue):
        """ dispatch functions from provided queue """
        try:
            todo = queue.get_nowait()
        except Queue.Empty:
            return
        try:
            (func, args, kwargs) = todo
            func(*args, **kwargs)
        except ValueError:
            try:
                (func, args) = todo
                func(*args)
            except ValueError:
                (func, ) = todo
                func()
        except:
            handle_exception()

class Irchandler(Eventhandler):

    """ handles ircevents """

    def dispatch(self, queue):
        """ dispatch ircevents from provided queue """
        try:
            (bot, ievent) = queue.get_nowait()
            bot.handle_ievent(ievent)
        except Queue.Empty:
            return
        except:
            handle_exception()

class Commandhandler(Eventhandler):

    """ handle bot commands """

    def dispatch(self, queue):
        """ dispatch bot commands from provided queue """
        try:
            (what, com, bot, ievent) = queue.get_nowait()
            rlog(10, 'eventhandler', '%s (%s) executing %s' % (ievent.nick, \
ievent.userhost, ievent.txt))
            what.dispatch(com, bot, ievent)
        except Queue.Empty:
            return
        except:
            handle_exception()

class Outputhandler(Eventhandler):

    """ handle bot output """

    def __init__(self, bot):
        Eventhandler.__init__(self)
        self.bot = bot
        self.lastout = 0
        
    def handle_one(self):
        """ do 1 loop over ready queues """
        if self.nooutput:
            self.go.put('later')
            return
        if self.bot.outputlock.locked():
            self.go.put('later')
            return
        ready = self.getready()
        for i in ready:
            self.dispatch(self.queues[i])

    def dispatch(self, queue):
        """ dispatch bot commands from provided queue """
        try:
            (bot, printto, what, how, who, fromm) = queue.get_nowait()
            bot.output(printto, what, how, who, fromm)
        except Queue.Empty:
            return
        except:
            handle_exception()

# handler to use in main prog
mainhandler = Eventhandler()
commandhandler = Commandhandler()
