/*
     This file is part of GNUnet
     (C) 2004, 2005, 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
     Boston, MA 02110-1301, USA.

*/

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_util_cron.h>
#include <GNUnet/gnunet_stats_lib.h>
#include <GNUnet/gnunet_getoption_lib.h>
#include <GNUnet/gnunet_protocols.h>
#include "functions.h"

#define FUNCTIONS_DEBUG NO

static StatPair *lastStatValues;

static unsigned int lsv_size;

static struct ClientServerConnection *sock;

static struct MUTEX *lock;

static long connectionGoal;

static long long banddown;

static long long bandup;

static struct GE_Context *ectx;

static struct GC_Configuration *cfg;

static struct CronManager *cron;

static int
getStatValue (long long *value,
              long long *lvalue,
              cron_t * dtime, const char *optName, int monotone)
{
  unsigned int i;

  *value = 0;
  if (lvalue != NULL)
    *lvalue = 0;
  for (i = 0; i < lsv_size; i++)
    {
      if (0 == strcmp (optName, lastStatValues[i].statName))
        {
          *value = lastStatValues[i].value;
          if (lvalue != NULL)
            *lvalue = lastStatValues[i].lvalue;
          if (dtime != NULL)
            *dtime = lastStatValues[i].delta;
          if ((monotone == YES) && (lvalue != NULL) && (*lvalue > *value))
            return SYSERR;      /* gnunetd restart? */
          return OK;
        }
    }
#if FUNCTIONS_DEBUG
  GE_LOG (ectx,
          GE_DEBUG | GE_DEVELOPER | GE_REQUEST,
          "Statistic not found: `%s'\n", optName);
#endif
  return SYSERR;
}

static void
updateConnectionGoal (void *unused)
{
  char *cmh;
  char *availableDown;
  char *availableUp;

  MUTEX_LOCK (lock);
  cmh = getConfigurationOptionValue (sock, "gnunetd", "connection-max-hosts");
  availableDown = getConfigurationOptionValue (sock,
                                               "LOAD", "MAXNETDOWNBPSTOTAL");
  availableUp = getConfigurationOptionValue (sock,
                                             "LOAD", "MAXNETUPBPSTOTAL");
  MUTEX_UNLOCK (lock);
  if (cmh == NULL)
    connectionGoal = 0;
  else
    connectionGoal = atol (cmh);
  if (availableDown == NULL)
    banddown = 0;
  else
    banddown = atol (availableDown);
  if (availableUp == NULL)
    bandup = 0;
  else
    bandup = atol (availableUp);

  FREENONNULL (cmh);
  FREENONNULL (availableDown);
  FREENONNULL (availableUp);
}

static int
getConnectedNodesStat (const void *closure, gfloat ** data)
{
  long long val;

  if (connectionGoal == 0)
    return SYSERR;
  if (OK != getStatValue (&val, NULL, NULL, "# of connected peers", NO))
    return SYSERR;
  data[0][0] = ((gfloat) val) / connectionGoal;
  return OK;
}

static int
getLoadStat (const void *closure, gfloat ** data)
{
  long long valc;
  long long vali;
  long long valu;
  long long vald;

  if (OK != getStatValue (&valc, NULL, NULL, "% of allowed cpu load", NO))
    return SYSERR;
  if (OK != getStatValue (&vali, NULL, NULL, "% of allowed io load", NO))
    return SYSERR;
  if (OK != getStatValue (&valu,
                          NULL, NULL, "% of allowed network load (up)", NO))
    return SYSERR;
  if (OK != getStatValue (&vald,
                          NULL, NULL, "% of allowed network load (down)", NO))
    return SYSERR;
  data[0][0] = (gfloat) valc / 100.0;
  data[0][1] = (gfloat) vali / 100.0;
  data[0][2] = (gfloat) valu / 100.0;
  data[0][3] = (gfloat) vald / 100.0;
  return OK;
}

static int
getQuotaStat (const void *closure, gfloat ** data)
{
  long long allowed;
  long long have;

  if (OK != getStatValue (&allowed,
                          NULL, NULL, "# bytes allowed in datastore", NO))
    return SYSERR;
  if (allowed == 0)
    return SYSERR;
  if (OK != getStatValue (&have, NULL, NULL, "# bytes in datastore", NO))
    return SYSERR;
  data[0][0] = ((gfloat) have) / allowed;
  return OK;
}

static int
getTrafficRecvStats (const void *closure, gfloat ** data)
{
  long long total;
  long long noise;
  long long content;
  long long queries;
  long long hellos;
  long long rlimit;
  long long ltotal;
  long long lnoise;
  long long lcontent;
  long long lqueries;
  long long lhellos;
  long long lrlimit;
  cron_t dtime;
  char *buffer;

  if (OK != getStatValue (&total, &ltotal, &dtime, "# bytes received", YES))
    return SYSERR;
  if (OK != getStatValue (&noise,
                          &lnoise, NULL, "# bytes of noise received", YES))
    return SYSERR;
  buffer = MALLOC (512);
  SNPRINTF (buffer, 512, "# bytes received of type %d", P2P_PROTO_gap_RESULT);
  if (OK != getStatValue (&content, &lcontent, NULL, buffer, YES))
    {
      content = 0;
      lcontent = 0;
    }
  SNPRINTF (buffer, 512, "# bytes received of type %d", p2p_PROTO_hello);
  if (OK != getStatValue (&hellos, &lhellos, NULL, buffer, YES))
    {
      hellos = 0;
      lhellos = 0;
    }
  SNPRINTF (buffer, 512, "# bytes received of type %d", P2P_PROTO_gap_QUERY);
  if (OK != getStatValue (&queries, &lqueries, NULL, buffer, YES))
    {
      queries = 0;
      lqueries = 0;
    }
  if (OK != getStatValue (&rlimit,
                          &lrlimit,
                          NULL, "# total bytes per second receive limit", NO))
    {
      rlimit = 0;
      lrlimit = 0;
    }
  FREE (buffer);
  if (banddown == 0)
    return SYSERR;

  total -= ltotal;
  noise -= lnoise;
  queries -= lqueries;
  content -= lcontent;
  hellos -= lhellos;
  if (banddown < 0)
    {
      data[0][0] = 0.0;
      data[0][1] = 0.0;
      data[0][2] = 0.0;
      data[0][3] = 0.0;
      data[0][4] = 0.0;
      data[0][5] = 0.0;
      return OK;
    }
  data[0][0] = ((gfloat) noise) / (banddown * dtime / cronSECONDS);     /* red */
  data[0][1] = ((gfloat) (content + noise)) / (banddown * dtime / cronSECONDS); /* green */
  data[0][2] = ((gfloat) (queries + content + noise)) / (banddown * dtime / cronSECONDS);       /* yellow */
  data[0][3] = ((gfloat) (queries + content + noise + hellos)) / (banddown * dtime / cronSECONDS);      /* blue */
  data[0][4] = ((gfloat) total) / (banddown * dtime / cronSECONDS);     /* gray */
  data[0][5] = (gfloat) rlimit / banddown;      /* magenta */
#if 0
  printf ("I: %f %f %f %f\n", data[0][0], data[0][1], data[0][2]);
#endif
  return OK;
}

static int
getTrafficSendStats (const void *closure, gfloat ** data)
{
  long long total;
  long long noise;
  long long content;
  long long queries;
  long long hellos;
  long long slimit;
  long long ltotal;
  long long lnoise;
  long long lcontent;
  long long lqueries;
  long long lhellos;
  long long lslimit;
  cron_t dtime;
  char *buffer;

  if (OK != getStatValue (&total,
                          &ltotal, &dtime, "# bytes transmitted", YES))
    return SYSERR;
  if (OK != getStatValue (&noise, &lnoise, NULL, "# bytes noise sent", YES))
    return SYSERR;
  buffer = MALLOC (512);
  SNPRINTF (buffer,
            512, "# bytes transmitted of type %d", P2P_PROTO_gap_RESULT);
  if (OK != getStatValue (&content, &lcontent, NULL, buffer, YES))
    {
      content = 0;
      lcontent = 0;
    }
  SNPRINTF (buffer,
            512, "# bytes transmitted of type %d", P2P_PROTO_gap_QUERY);
  if (OK != getStatValue (&queries, &lqueries, NULL, buffer, YES))
    {
      queries = 0;
      lqueries = 0;
    }
  SNPRINTF (buffer, 512, "# bytes transmitted of type %d", p2p_PROTO_hello);
  if (OK != getStatValue (&hellos, &lhellos, NULL, buffer, YES))
    {
      queries = 0;
      lqueries = 0;
    }
  if (OK != getStatValue (&slimit,
                          &lslimit,
                          NULL, "# total bytes per second send limit", NO))
    {
      slimit = 0;
      lslimit = 0;
    }
  FREE (buffer);
  if (bandup == 0)
    return SYSERR;
  total -= ltotal;
  noise -= lnoise;
  queries -= lqueries;
  content -= lcontent;
  hellos -= lhellos;
  if (bandup < 0)
    {
      data[0][0] = 0.0;
      data[0][1] = 0.0;
      data[0][2] = 0.0;
      data[0][3] = 0.0;
      data[0][4] = 0.0;
      data[0][5] = 0.0;
      return OK;
    }
  data[0][0] = ((gfloat) noise) / (bandup * dtime / cronSECONDS);       /* red */
  data[0][1] = ((gfloat) (noise + content)) / (bandup * dtime / cronSECONDS);   /* green */
  data[0][2] = ((gfloat) (noise + content + queries)) / (bandup * dtime / cronSECONDS); /* yellow */
  data[0][3] = ((gfloat) (noise + content + queries + hellos)) / (bandup * dtime / cronSECONDS);        /* blue */
  data[0][4] = ((gfloat) total) / (bandup * dtime / cronSECONDS);       /* grey */
  data[0][5] = ((gfloat) slimit) / bandup;      /* magenta */
#if 0
  printf ("O: %f %f %f %f\n", data[0][0], data[0][1], data[0][2], data[0][3]);
#endif
  return OK;
}


static int
getEffectivenessStats (const void *closure, gfloat ** data)
{
  static cron_t last;
  static double lastdata;
  static double lastavg;
  long long total;
  long long success;
  long long local;
  long long ltotal;
  long long lsuccess;
  long long llocal;
  cron_t now;

  now = get_time ();
  if (now < last + 2 * cronMINUTES)
    {
      data[0][0] = lastdata;
      data[0][1] = lastavg;
      return OK;
    }
  last = now;
  if (OK != getStatValue (&total,
                          &ltotal,
                          NULL,
                          "# gap requests forwarded (counting each peer)",
                          YES))
    return SYSERR;
  if (OK != getStatValue (&success,
                          &lsuccess,
                          NULL, "# gap routing successes (total)", YES))
    return SYSERR;
  if (OK != getStatValue (&local,
                          &llocal,
                          NULL,
                          "# gap requests processed: local result", YES))
    return SYSERR;
  total -= ltotal;
  data[0][0] = 0.0;
  if (ltotal + total > 0)
    {
      data[0][1] = lastavg =
        1.0 * (success + lsuccess - local - llocal) / (total + ltotal);
    }
  else
    {
      data[0][1] = 0.0;
      return OK;
    }
  if (total == 0)
    return OK;
  success -= lsuccess;
  local -= llocal;
  if (success <= local)
    return OK;
  success -= local;
  lsuccess -= llocal;
  data[0][0] = lastdata = 1.0 * success / total;
  return OK;
}

static int
statsProcessor (const char *optName, unsigned long long value, void *data)
{
  /**
   * Keep track of last match (or, more precisely, position
   * of next expected match) since 99.99% of the time we
   * go over the same stats in the same order and thus
   * this will predict correctly).
   */
  static unsigned int last;
  cron_t *delta = data;
  int j;
  int found;

  if (last >= lsv_size)
    last = 0;
  j = last;
  found = -1;
  if ((j < lsv_size) && (0 == strcmp (optName, lastStatValues[j].statName)))
    found = j;
  if (found == -1)
    {
      for (j = 0; j < lsv_size; j++)
        {
          if (0 == strcmp (optName, lastStatValues[j].statName))
            {
              found = j;
              break;
            }
        }
    }
  if (found == -1)
    {
      found = lsv_size;
      GROW (lastStatValues, lsv_size, lsv_size + 1);
      lastStatValues[found].statName = STRDUP (optName);
    }
  lastStatValues[found].lvalue = lastStatValues[found].value;
  lastStatValues[found].value = value;
  lastStatValues[found].delta = *delta;
  last = found + 1;
  return OK;
}

/**
 * Cron-job that updates all stat values.
 */
static void
updateStatValues (void *unused)
{
  static cron_t lastUpdate;
  cron_t now;
  cron_t delta;

  now = get_time ();
  delta = now - lastUpdate;
  MUTEX_LOCK (lock);
  if (OK == STATS_getStatistics (ectx, sock, &statsProcessor, &delta))
    lastUpdate = now;
  MUTEX_UNLOCK (lock);
}


StatEntry stats[] = {
  {
   gettext_noop ("Connectivity"),
   gettext_noop ("# connected nodes (100% = connection table size)"),
   &getConnectedNodesStat,
   NULL,
   1,
   NO,
   }
  ,
  {
   gettext_noop ("System load"),
   gettext_noop
   ("CPU load (red), IO load (green), Network upload (yellow), Network download (blue)"),
   &getLoadStat,
   NULL,
   4,
   NO,
   }
  ,
  {
   gettext_noop ("Datastore capacity"),
   gettext_noop ("Data in datastore (in percent of allowed quota)"),
   &getQuotaStat,
   NULL,
   1,
   NO,
   }
  ,
  {
   gettext_noop ("Inbound Traffic"),
   gettext_noop
   ("Noise (red), Content (green), Queries (yellow), Hellos (blue), other (gray), limit (magenta)"),
   &getTrafficRecvStats,
   NULL,
   6,
   5,
   }
  ,
  {
   gettext_noop ("Outbound Traffic"),
   gettext_noop
   ("Noise (red), Content (green), Queries (yellow), Hellos (blue), other (gray), limit (magenta)"),
   &getTrafficSendStats,
   NULL,
   6,
   5,
   }
  ,
  {
   gettext_noop ("Routing Effectiveness"),
   gettext_noop
   ("Current (red) and average (green) effectiveness (100% = perfect)"),
   &getEffectivenessStats,
   NULL,
   2,
   NO,
   }
  ,
  {
   NULL,
   NULL,
   NULL,
   NULL,
   0,
   NO,
   }
  ,
};

static unsigned long long UPDATE_INTERVAL;

void
init_functions (struct GE_Context *e, struct GC_Configuration *c)
{
  ectx = e;
  cfg = c;
  GC_get_configuration_value_number (cfg,
                                     "GNUNET-GTK",
                                     "STATS-INTERVAL",
                                     1,
                                     99 * cronYEARS,
                                     30 * cronSECONDS, &UPDATE_INTERVAL);
  sock = client_connection_create (ectx, cfg);
  lock = MUTEX_CREATE (NO);
  cron = gnunet_gtk_get_cron ();
  cron_add_job (cron,
                &updateStatValues, UPDATE_INTERVAL, UPDATE_INTERVAL, NULL);
  cron_add_job (cron,
                &updateConnectionGoal,
                5 * cronMINUTES, 5 * cronMINUTES, NULL);
}

void
done_functions ()
{
  int i;

  cron_del_job (cron, &updateConnectionGoal, 5 * cronMINUTES, NULL);
  cron_del_job (cron, &updateStatValues, UPDATE_INTERVAL, NULL);
  MUTEX_DESTROY (lock);
  connection_destroy (sock);
  for (i = 0; i < lsv_size; i++)
    FREE (lastStatValues[i].statName);
  GROW (lastStatValues, lsv_size, 0);
  sock = NULL;
}


/* end of functions.c */
