/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescanpreviewarea.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>
#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#define GNOME_SCAN_PREVIEW_AREA_ERROR			(g_type_qname (GNOME_TYPE_SCAN_PREVIEW_AREA))
#define	GNOME_SCAN_PREVIEW_AREA_PARENT_CLASS(klass)	(GTK_WIDGET_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)				(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_PREVIEW_AREA, GnomeScanPreviewAreaPrivate))
#define	ANCHOR_SIDE					10
#define debug_rect(r)					g_debug ("%s %i×%i+%i+%i", __FUNCTION__, r->width, r->height, r->x, r->y)
#define debug_area(a)					g_debug ("%s %f×%f+%f+%f", __FUNCTION__, a->width, a->height, a->x, a->y)

typedef struct _GnomeScanPreviewAreaPrivate		GnomeScanPreviewAreaPrivate;

struct _GnomeScanPreviewAreaPrivate {
  gboolean		dispose_has_run;

  GtkTooltips		*tooltips;

  gboolean		button_pressed;
  GdkPixbuf		*image;
  /* The rotation degree to apply to the current preview */
  GdkPixbufRotation	rotate;
  /* the final rotation to apply */
  GdkPixbufRotation	rotation;

  cairo_t		*cr;
  cairo_surface_t	*buffer;
  gboolean		buffer_uptodate;

  /* origine of selection */
  GdkPoint		*orig;
  /* region of interest */
  GdkRectangle		*roi;
  gboolean		custom;
  /* current pointer coord */
  GdkPoint		*coord;
  /* current over flyed anchor */
  GtkAnchorType		anchor;
};

enum {
  PROP_0,
  PROP_CONTEXT
};

enum {
  N_SIGNALS
};

static guint signals[N_SIGNALS];



/********************************
 * 	     CALLBACKS		*
 ********************************/

void				gspa_scanner_selected				(GnomeScanContext *context,
										 GnomeScanner *scanner,
										 GtkWidget *area);

void				gspa_preview_received				(GnomeScanContext *context,
										 GnomeScanResult *info,
										 GtkWidget *area);

gboolean			gspa_exposed					(GtkWidget *da,
										 GdkEventExpose *event,
										 GtkWidget *area);

gboolean			gspa_button_pressed				(GtkWidget *widget,
										 GdkEventButton *event,
										 GtkWidget *area);

gboolean			gspa_mouse_moved				(GtkWidget *widget,
										 GdkEventMotion *event,
										 GtkWidget *area);

gboolean			gspa_button_released				(GtkWidget *widget,
										 GdkEventButton *event,
										 GtkWidget *area);

gboolean			gspa_scrolled					(GtkWidget *widget,
										 GdkEventScroll *event,
										 GtkWidget *area);

void				gspa_rotate_counter_clockwise_clicked		(GnomeScanPreviewArea *pa,
										 gpointer data);

void				gspa_rotate_clockwise_clicked			(GnomeScanPreviewArea *pa,
										 gpointer data);

void				gspa_context_changed				(GnomeScanContext *context,
										 gchar* option_name,
										 GnomeScanPreviewArea *gspa);

/********************************
 * 	      OTHER		*
 ********************************/

void				gspa_deprecate_buffer				(GnomeScanPreviewArea *pa);

void				gspa_draw_buffer				(GnomeScanPreviewArea *pa);

GdkRectangle*			gspa_compute_anchor_rect			(GtkAnchorType anchor,
										 GnomeScanPreviewArea *pa);

void				gspa_draw_anchor_path				(GtkAnchorType anchor,
										 GnomeScanPreviewArea *pa);

void				gspa_in_anchor					(GtkAnchorType anchor,
										 GnomeScanPreviewArea *pa);

void				gspa_anchor_over				(GnomeScanPreviewArea *pa);

void				gspa_set_cursor					(GtkAnchorType anchor,
										 GnomeScanPreviewArea *pa);

void				gspa_send_area					(GnomeScanPreviewArea *pa);

void				gspa_apply_transformations			(GnomeScanPreviewArea *pa);

void				gspa_gdk_rectangle_rotate_simple		(GdkRectangle *rect,
										 GdkPixbufRotation rotation,
										 gint width,
										 gint height);

/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scan_preview_area_dispose			(GObject *obj);

void				gnome_scan_preview_area_set_property 		(GObject *obj,
										 guint property_id,
										 const GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_preview_area_get_property 		(GObject *obj,
										 guint property_id,
										 GValue *value,
										 GParamSpec *pspec);

G_DEFINE_TYPE (GnomeScanPreviewArea, gnome_scan_preview_area, GTK_TYPE_SCROLLED_WINDOW);

void
gnome_scan_preview_area_class_init (GnomeScanPreviewAreaClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property	= gnome_scan_preview_area_set_property;
  gobject_class->get_property	= gnome_scan_preview_area_get_property;
  gobject_class->dispose	= gnome_scan_preview_area_dispose;

  g_type_class_add_private (gobject_class, sizeof (GnomeScanPreviewArea));

  klass->rotate_counter_clockwise_clicked	= gspa_rotate_counter_clockwise_clicked;
  klass->rotate_clockwise_clicked		= gspa_rotate_clockwise_clicked;

  /* Properties */
  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"The GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));
}

void
gnome_scan_preview_area_init (GnomeScanPreviewArea *widget)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (widget);

  widget->context = NULL;
  widget->drawing_area = NULL;

  priv->dispose_has_run = FALSE;
  priv->tooltips	= NULL;
  priv->button_pressed	= FALSE;
  priv->buffer		= NULL;
  priv->buffer_uptodate	= FALSE;
  priv->orig		= g_new0 (GdkPoint, 1);
  priv->roi		= g_new0 (GdkRectangle, 1);
  priv->custom		= FALSE;
  priv->coord		= g_new0 (GdkPoint, 1);
  priv->rotate		= GDK_PIXBUF_ROTATE_NONE;
  priv->rotation	= GDK_PIXBUF_ROTATE_NONE;
}

void
gnome_scan_preview_area_dispose (GObject *obj)
{
  GnomeScanPreviewArea *widget = GNOME_SCAN_PREVIEW_AREA (obj);
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (widget);
  GnomeScanPreviewAreaClass *klass = GNOME_SCAN_PREVIEW_AREA_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref devices */
  g_object_unref (widget->context);
  g_object_unref (widget->drawing_area);
  g_object_unref (priv->tooltips);
  g_free (priv->orig);
  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_PREVIEW_AREA_PARENT_CLASS (b_klass)->dispose (obj); */
}

void
gnome_scan_preview_area_set_property (GObject *obj,
				      guint property_id,
				      const GValue *value,
				      GParamSpec *pspec)
{
  GnomeScanPreviewArea *area = GNOME_SCAN_PREVIEW_AREA (obj);
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (area);

  switch (property_id) {
  case PROP_CONTEXT:
    area->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id,pspec);
    break;
  }
}

void
gnome_scan_preview_area_get_property (GObject *obj,
				      guint property_id,
				      GValue *value,
				      GParamSpec *pspec)
{
  GnomeScanPreviewArea *area = GNOME_SCAN_PREVIEW_AREA (obj);
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (area);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, area->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,property_id,pspec);
    break;
  }
}



/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_preview_area_new:
 * @context: a #GnomeScanContext
 * 
 * Create a new #GnomeScanPreviewArea and connect it to @context.
 * 
 * Return value: a new #GnomeScanPreviewArea
 */
GtkWidget*
gnome_scan_preview_area_new (GnomeScanContext *context)
{
  GtkWidget *area, *da, *eventbox;
  GdkWindow *window;
  GnomeScanPreviewArea *gspa;
  GnomeScanPreviewAreaPrivate *priv;

  gspa = g_object_new (GNOME_TYPE_SCAN_PREVIEW_AREA,
		       "context", context,
		       NULL);
  priv = GET_PRIVATE (gspa);

  da = gspa->drawing_area = gtk_drawing_area_new ();
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (gspa), da);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (gspa), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_set_size_request (GTK_WIDGET (gspa), 250, 300);

  gtk_widget_set_events (da,
			 GDK_EXPOSURE_MASK |
			 GDK_BUTTON_PRESS_MASK |
			 GDK_BUTTON_RELEASE_MASK |
			 GDK_POINTER_MOTION_MASK |
			 GDK_POINTER_MOTION_HINT_MASK |
			 GDK_SCROLL_MASK);

/*   priv->tooltips = gtk_tooltips_new (); */
/*   gtk_tooltips_set_tip (GTK_TOOLTIPS (priv->tooltips), GTK_WIDGET (gspa), */
/* 			_("Preview area."), */
/* 			_("Use the preview to select the scan area and the rotation to get the right image.")); */

  g_signal_connect (context,
		    "scanner-selected",
		    G_CALLBACK (gspa_scanner_selected),
		    gspa);

  g_signal_connect (context,
		    "preview-received",
		    G_CALLBACK (gspa_preview_received),
		    gspa);

  g_signal_connect (da,
		    "expose-event",
		    G_CALLBACK (gspa_exposed),
		    gspa);

  g_signal_connect (da,
		    "button-press-event",
		    G_CALLBACK (gspa_button_pressed),
		    gspa);

  g_signal_connect (da,
		    "motion-notify-event",
		    G_CALLBACK (gspa_mouse_moved),
		    gspa);

  g_signal_connect (da,
		    "button-release-event",
		    G_CALLBACK (gspa_button_released),
		    gspa);

  g_signal_connect (da,
		    "scroll-event",
		    G_CALLBACK (gspa_scrolled),
		    gspa);

  g_signal_connect (context,
		    "changed",
		    G_CALLBACK (gspa_context_changed),
		    gspa);

  return GTK_WIDGET (gspa);
}


/**
 * gnome_scan_preview_area_refresh:
 * @area: a #GnomeScanPreviewArea
 * 
 * Query the #GnomeScanContext to acquire a fresh scan review.
 *
 * Deprecated: application must directly ask the #GnomeScanContext for
 * a preview.
 */
void
gnome_scan_preview_area_refresh (GnomeScanPreviewArea *area)
{
  gnome_scan_context_acquire_preview (area->context);
}

/**
 * gnome_scan_preview_area_button_refresh:
 * @area: 
 * 
 * 
 * 
 * Returns: 
 **/
GtkWidget*
gnome_scan_preview_area_button_refresh (GnomeScanPreviewArea *area)
{
  GtkWidget *button;

  button = gtk_button_new_from_stock (GTK_STOCK_REFRESH);
  g_signal_connect_swapped (button,
			    "clicked",
			    (GCallback) gnome_scan_preview_area_refresh,
			    area);

  return button;
}

/**
 * gnome_scan_preview_area_button_rotate_cw:
 * @area: 
 * 
 * 
 * 
 * Returns: 
 **/
GtkWidget*
gnome_scan_preview_area_button_rotate_cw (GnomeScanPreviewArea *area)
{
  GtkWidget *button;
  button = gtk_button_new_from_stock (GS_STOCK_ROTATE_CLOCKWISE);
  g_signal_connect_swapped (button,
			    "clicked",
			    (GCallback) GNOME_SCAN_PREVIEW_AREA_GET_CLASS (area)->rotate_clockwise_clicked,
			    area);
  return button;
}

/**
 * gnome_scan_preview_area_button_rotate_ccw:
 * @area: 
 * 
 * 
 * 
 * Returns: 
 **/
GtkWidget*
gnome_scan_preview_area_button_rotate_ccw (GnomeScanPreviewArea *area)
{
  GtkWidget *button;
  button = gtk_button_new_from_stock (GS_STOCK_ROTATE_COUNTER_CLOCKWISE);
  g_signal_connect_swapped (button,
			    "clicked",
			    (GCallback) GNOME_SCAN_PREVIEW_AREA_GET_CLASS (area)->rotate_counter_clockwise_clicked,
			    area);
  return button;
}

/********************************
 * 	     CALLBACKS		*
 ********************************/

/* create an empty preview with the right geometry. */
void
gspa_scanner_selected (GnomeScanContext *context,
		       GnomeScanner *scanner,
		       GtkWidget *area)
{
  GnomeScanPreviewArea *pa;
  GnomeScanGeometry *dim;
  GnomeScanArea *a;
  GnomeScanResult *result;
  GdkPixmap *pixmap;
  GdkPixbuf *icon;
  gint width, height, size;
  cairo_t *cr;

  pa = GNOME_SCAN_PREVIEW_AREA (area);

  /*
   * Here we have too possibility : either we auto acquire a preview
   * (Should be done only if selected source is flatbed); either we
   * create a pixbuf with the device size.
   */

  /* simulate acquisition of a 50dpi preview */
  result = g_new0 (GnomeScanResult, 1);
  result->resolution = 50.;
  result->source = gnome_scan_context_get_source (context);

  /* create an empty pixbuf at the correct geometry */
  dim = gnome_scanner_get_geometry (scanner);
  width = gnome_scan_pixel_from_mm (dim->width, 50);
  height = gnome_scan_pixel_from_mm (dim->height, 50);
  pixmap = gdk_pixmap_new (NULL, width, height, 24);

  /* Draw a big (and nice) logo */
  cr = gdk_cairo_create (GDK_DRAWABLE (pixmap));
  cairo_set_source_rgb (cr, 1., 1., 1.);
  cairo_paint (cr);

  /* icon take 70% of the picture */	
  size = MIN (width, height) * .6;
  if (icon = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
				       "document",
				       size,
				       GTK_ICON_LOOKUP_FORCE_SVG,
				       NULL)) {
    /* we put the file as close as possible to the border. Often,
       icons have a large empty border around image */
    gdk_cairo_set_source_pixbuf (cr, icon, - (size * .1), - (size * .05));
  }
  else {
    cairo_set_source_rgb (cr, .8, .8, .8);
  }

  cairo_paint (cr);
  cairo_destroy (cr);

  /* create pixbuf */
  result->image  = gdk_pixbuf_get_from_drawable (NULL,
						 GDK_DRAWABLE (pixmap),
						 gdk_screen_get_default_colormap (gdk_screen_get_default ()),	
						 0, 0,
						 0, 0,
						 width,
						 height);

  /* simulate the acquisition of the pixbuf. Not very clean. */
  gspa_preview_received (context, result, area);
}

/* store pixbuf and query draw */
void
gspa_preview_received (GnomeScanContext *context,
		       GnomeScanResult *result,
		       GtkWidget *area)
{ 
  GnomeScanPreviewArea *pa = GNOME_SCAN_PREVIEW_AREA (area);
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (area);
  GdkRectangle *rect;

  /* drop old preview */
  if (priv->image) {
    g_object_unref (priv->image);
  }
  /* ref new one. */
  priv->image = result->image;
  g_object_ref (priv->image);
  /* preview are always in 50x50 dpi */
  pa->resolution = result->resolution;

  /*
   * We received a 0° rotated image.
   * We have a priv->rotation rotated ROI.
   * We rotate back the ROI
   * And ask for a rotation (roi + image) in priv->rotate
   */
  /* get width and height of pixbuf */
  rect = g_new0 (GdkRectangle, 1);
  rect->width = gdk_pixbuf_get_width (priv->image);
  rect->height = gdk_pixbuf_get_height (priv->image);

  /* rotate image size to the same rotation as the ROI. */
  gspa_gdk_rectangle_rotate_simple (rect,
				    priv->rotation,
				    rect->width,
				    rect->height);
  /* rotate back ROI using rotated image size */
  gspa_gdk_rectangle_rotate_simple (priv->roi,
				    360 - priv->rotation,
				    rect->width,
				    rect->height);
  g_free (rect);

  /* Reinit the rotation */
  priv->rotate = priv->rotation;
  priv->rotation = 0;

  /* Ok, rotate to new image and new ROI, and much more ... */
  gspa_apply_transformations (pa);
}

void
gspa_context_changed (GnomeScanContext *context,
		      gchar* option_name,
		      GnomeScanPreviewArea *gspa)
{
  GnomeScanPreviewAreaPrivate *priv;
  GnomeScanArea *area;
  GdkRectangle *rect;

  if (!g_ascii_strcasecmp (option_name, "area")) {
    priv = GET_PRIVATE (gspa);
    area = gnome_scan_context_get_area (context);
    priv->custom = area->custom;
    priv->roi->x = gnome_scan_pixel_from_mm (area->x, gspa->resolution);
    priv->roi->y = gnome_scan_pixel_from_mm (area->y, gspa->resolution);
    priv->roi->width = gnome_scan_pixel_from_mm (area->width, gspa->resolution);
    priv->roi->height = gnome_scan_pixel_from_mm (area->height, gspa->resolution);

    gspa_gdk_rectangle_rotate_simple (priv->roi, priv->rotation,
				      gdk_pixbuf_get_width (priv->image),
				      gdk_pixbuf_get_height (priv->image));
    gspa_deprecate_buffer (gspa);
  }
}

/* paint buffer in GdkWindow */
gboolean
gspa_exposed (GtkWidget *da,
	      GdkEventExpose *event,
	      GtkWidget *area)
{
  GnomeScanPreviewArea *gspa = GNOME_SCAN_PREVIEW_AREA (area);
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (area);
  gint lw, lh;
  cairo_t *cr;

  if (!priv->buffer_uptodate) {
    gspa_draw_buffer (gspa);
  }

  cr = gdk_cairo_create (gspa->drawing_area->window);
  cairo_set_source_surface (cr, priv->buffer, 0, 0);
  cairo_paint (cr);
  cairo_destroy (cr);

  return FALSE;
}

/*
 * Start ROI selection and confine pointer into area.
 */
gboolean
gspa_button_pressed (GtkWidget *widget,
		     GdkEventButton *event,
		     GtkWidget *area)
{
  GnomeScanPreviewArea *pa = GNOME_SCAN_PREVIEW_AREA (area);
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);

  if (priv->custom || priv->anchor == GTK_ANCHOR_CENTER) {
    priv->button_pressed = TRUE;

    priv->orig->x = event->x;
    priv->orig->y = event->y;

    gdk_pointer_grab (pa->drawing_area->window,
		      TRUE,
		      GDK_BUTTON_RELEASE_MASK |
		      GDK_BUTTON1_MOTION_MASK |
		      GDK_POINTER_MOTION_MASK |
		      GDK_POINTER_MOTION_HINT_MASK,
		      pa->drawing_area->window,
		      NULL,
		      event->time);
  }

  return FALSE;
}

#define	set_if_included(var,val,min,max)	var = ((min) <= (val) && (val) <= (max)) ? (val) : (var)

gboolean
gspa_mouse_moved (GtkWidget *widget,
		  GdkEventMotion *event,
		  GtkWidget *area)
{
  GnomeScanPreviewArea *pa = GNOME_SCAN_PREVIEW_AREA (area);
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);

  if (priv->button_pressed) {
    /*
     * If button is pressed, then compute the new ROI selected from
     * priv->orig to current position.
     */
    switch (priv->anchor) {
    case GTK_ANCHOR_NORTH_WEST:
      priv->roi->width += priv->roi->x - event->x;
      priv->roi->height+= priv->roi->y - event->y;
      priv->roi->x = event->x;
      priv->roi->y = event->y;
      break;
    case GTK_ANCHOR_NORTH:
      priv->roi->height+= priv->roi->y - event->y;
      priv->roi->y = event->y;
      break;
    case GTK_ANCHOR_NORTH_EAST:
      priv->roi->width  = event->x - priv->roi->x;
      priv->roi->height+= priv->roi->y - event->y;
      priv->roi->y = event->y;
      break;
    case GTK_ANCHOR_EAST:
      priv->roi->width  = event->x - priv->roi->x;
      break;
    case GTK_ANCHOR_SOUTH_EAST:
      priv->roi->width  = event->x - priv->roi->x;
      priv->roi->height = event->y - priv->roi->y ;
      break;
    case GTK_ANCHOR_SOUTH:
      priv->roi->height = event->y - priv->roi->y ;
      break;
    case GTK_ANCHOR_SOUTH_WEST:
      priv->roi->width += priv->roi->x - event->x;
      priv->roi->height = event->y - priv->roi->y ;
      priv->roi->x = event->x;
      break;
    case GTK_ANCHOR_WEST:
      priv->roi->width += priv->roi->x - event->x;
      priv->roi->x = event->x;
      break;
    case GTK_ANCHOR_CENTER:
      set_if_included (priv->roi->x, event->x - priv->roi->width / 2, 0, gdk_pixbuf_get_width (priv->image) - priv->roi->width);
      set_if_included (priv->roi->y, event->y - priv->roi->height / 2, 0, gdk_pixbuf_get_height (priv->image) - priv->roi->height);
      break;
    default:
      if (ABS (event->x - priv->orig->x) > 20 && ABS (event->y - priv->orig->y) > 20) {
	priv->roi->x = MIN (event->x, priv->orig->x);
	priv->roi->y = MIN (event->y, priv->orig->y);
	priv->roi->width = MAX (event->x, priv->orig->x) - priv->roi->x;
	priv->roi->height = MAX (event->y, priv->orig->y) - priv->roi->y;
      }
      break;
    }

    gspa_deprecate_buffer (pa);

    /* TODO: auto scroll on edge */

    while (gtk_events_pending ())
      gtk_main_iteration ();

  }
  else {
    /*
     * If no button is pressed, then search if the user move over an
     * anchor.
     */
    priv->coord->x = event->x;
    priv->coord->y = event->y;
    gspa_anchor_over (pa);
    gspa_set_cursor (priv->anchor, pa);
  }

  gdk_window_get_pointer (widget->window, NULL, NULL, NULL);
  return FALSE;
}

/*
 * Stop the selection of the ROI and sent the selected ROI to the
 * context.
 */
gboolean
gspa_button_released (GtkWidget *widget,
		      GdkEventButton *event,
		      GtkWidget *area)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (area);

  priv->button_pressed = FALSE;
  gdk_pointer_ungrab (event->time);
  gspa_send_area (GNOME_SCAN_PREVIEW_AREA (area));

  return FALSE;
}

gboolean
gspa_scrolled (GtkWidget *widget,
	       GdkEventScroll *event,
	       GtkWidget *area)
{
  GtkAdjustment *adj;
  gint factor;
  gdouble value;

  /* Determine the adj to update and the update factor */
  switch (event->direction) {
  case GDK_SCROLL_UP:
  case GDK_SCROLL_DOWN:
    adj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (area));
    factor = event->direction == GDK_SCROLL_UP ? -1 : 1;
    break;
  case GDK_SCROLL_LEFT:
  case GDK_SCROLL_RIGHT:
    adj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (area));
    /* RTL/LTR bug ? */
    factor = event->direction == GDK_SCROLL_LEFT ? -1 : 1;
    break;
  }

  value = adj->value + (adj->step_increment * factor);

  /* round value to lower and upper if close */
  if (adj->upper - adj->page_size - adj->step_increment < value) {
    value = adj->upper - adj->page_size;
  }
  if (value < adj->lower + adj->step_increment) {
    value = adj->lower;
  }

  /* set value if in range */
  if (adj->lower <= value
      && value <= adj->upper - adj->page_size) {
    gtk_adjustment_set_value (adj,
			      value);
  }
}

void
gspa_rotate_counter_clockwise_clicked (GnomeScanPreviewArea *pa,
				       gpointer data)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);
  priv->rotate = GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE;
  gspa_apply_transformations (pa);
}

void
gspa_rotate_clockwise_clicked (GnomeScanPreviewArea *pa,
			       gpointer data)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);
  priv->rotate = GDK_PIXBUF_ROTATE_CLOCKWISE;
  gspa_apply_transformations (pa);
}


/********************************
 * 	      OTHER		*
 ********************************/

/*
 * ask for buffer update and queue widget for drawing.
 */
void
gspa_deprecate_buffer (GnomeScanPreviewArea *pa)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);
  priv->buffer_uptodate = FALSE;
  gtk_widget_queue_draw (pa->drawing_area);
}


/* 
 * Draw the pixbuf priv->image inside priv->buffer
 * cairo_surface_t. Also draw faded white rectangles outside
 * priv->roi, black line around priv->roi and anchors.
 */
void
gspa_draw_buffer (GnomeScanPreviewArea *pa)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);
  GdkRectangle *a;
  cairo_t *cr;
  static gdouble dashes[] = {5., 5.};
  static gint n_dashes = G_N_ELEMENTS (dashes);
  gint width, height;

  /* If there is no preview, no need to draw anything. */
  if (!priv->image) {
    return;
  }

  width = gdk_pixbuf_get_width (priv->image);
  height = gdk_pixbuf_get_height (priv->image);

  /* destroy any existing buffer. */
  if (priv->buffer) {
    cairo_surface_destroy (priv->buffer);
  }

  /* Duplicate ROI in order to modify. */
  a = g_memdup (priv->roi, sizeof (GnomeScanArea));
  /* Don't draw over the ROI */
  a->x--;
  a->y--;
  a->width+= 2;
  a->height+= 2;

  /* create a buffer with the preview size */
  cr = gdk_cairo_create (pa->drawing_area->window);
  priv->buffer = cairo_surface_create_similar (cairo_get_target (cr),
					       CAIRO_CONTENT_COLOR,
					       width,
					       height);
  cairo_destroy (cr);

  /* paint preview to buffer */
  priv->cr = cr = cairo_create (priv->buffer);

  cairo_set_fill_rule (cr, CAIRO_FILL_RULE_EVEN_ODD);
  gdk_cairo_set_source_pixbuf (cr, priv->image, 0, 0);
  cairo_paint (cr);

  /* Clip the region outside the ROI and outside anchor */
  cairo_new_path (cr);
  cairo_rectangle (cr, 0, 0, width, height);
  gdk_cairo_rectangle (cr, a);
  cairo_clip_preserve (cr);
  if (priv->custom) {
    gs_enum_foreach (GTK_TYPE_ANCHOR_TYPE, (GSEFunc) gspa_draw_anchor_path, pa);
  }
  else {
    gspa_draw_anchor_path (GTK_ANCHOR_CENTER, pa);
  }
  cairo_close_path (cr);
  cairo_clip (cr);

  /* paint faded white */
  cairo_set_source_rgba (cr, 1., 1., 1., .7);
  cairo_paint (cr);

  /* draw anchors */
  cairo_reset_clip (cr);
  cairo_new_path (cr);
  if (priv->custom) {
    gs_enum_foreach (GTK_TYPE_ANCHOR_TYPE, (GSEFunc) gspa_draw_anchor_path, pa);
  }
  else {
    gspa_draw_anchor_path (GTK_ANCHOR_CENTER, pa);
  }

  /* fill a 50% white */
  cairo_set_source_rgba (cr, 1., 1., 1., .5);
  cairo_fill_preserve (cr);
  /* line with 100% black */
  cairo_set_source_rgba (cr, 0., 0., 0., 1.);
  cairo_set_line_width (cr, 2);
  cairo_stroke_preserve (cr);

  cairo_rectangle (cr, 0, 0, width, height);
  cairo_clip (cr);

  /* draw black lines around ROI */
  cairo_new_path (cr);
  gdk_cairo_rectangle (cr, a);
  cairo_set_source_rgba (cr, 0., 0., 0., 1.);
  cairo_set_line_width (cr, 2);
  cairo_set_dash (cr, dashes, n_dashes, 0.);
  cairo_stroke (cr);

  /* Free all */
  cairo_destroy (cr);
  g_free (a);
  priv->buffer_uptodate = TRUE;
  gtk_widget_queue_draw (pa->drawing_area);
}

/*
 * Create a new GdkRectangle containing the drawing coordonate of an anchor.
 */
GdkRectangle*
gspa_compute_anchor_rect (GtkAnchorType anchor,
			  GnomeScanPreviewArea *pa)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);
  GdkRectangle *roi, *rect;

  /* s = square side */
  gdouble x, y, s = ANCHOR_SIDE;

  roi = g_memdup (priv->roi, sizeof (GdkRectangle));
  roi->x--;
  roi->y--;
  roi->width+=2;
  roi->height+=2;

  switch (anchor) {
  case GTK_ANCHOR_NORTH_WEST:
    x = roi->x;
    y = roi->y;
    break;
  case GTK_ANCHOR_NORTH:
    x =  roi->x + roi->width / 2;
    y =  roi->y;
    break;
  case GTK_ANCHOR_NORTH_EAST:
    x = roi->x + roi->width;
    y = roi->y;
    break;
  case GTK_ANCHOR_EAST:
    x = roi->x + roi->width;
    y = roi->y + roi->height / 2;
    break;
  case GTK_ANCHOR_SOUTH_EAST:
    x = roi->x + roi->width;
    y = roi->y + roi->height;
    break;
  case GTK_ANCHOR_SOUTH:
    x = roi->x + roi->width / 2;
    y = roi->y + roi->height;
    break;
  case GTK_ANCHOR_SOUTH_WEST:
    x = roi->x;
    y = roi->y + roi->height;
    break;
  case GTK_ANCHOR_WEST:
    x = roi->x;
    y = roi->y + roi->height / 2;
    break;
  case GTK_ANCHOR_CENTER:
  default:
    x = roi->x + roi->width / 2;
    y = roi->y + roi->height / 2;
    break;
  }

  rect = g_new0 (GdkRectangle, 1);
  rect->x = x - s/2;
  rect->y = y - s/2;
  rect->width = s;
  rect->height = s;

  return rect;
}

/* Draw an anchor */
void
gspa_draw_anchor_path (GtkAnchorType anchor,
		       GnomeScanPreviewArea *pa)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);
  GdkRectangle *rect;
  cairo_t *cr;

  rect = gspa_compute_anchor_rect (anchor, pa);
  cr = priv->cr;

  gdk_cairo_rectangle (cr, rect);
  g_free (rect);
}

/*
 * Test if the current coord is in @anchor. If TRUE, the anchor is
 * stored.
 */
void
gspa_in_anchor (GtkAnchorType anchor,
		GnomeScanPreviewArea *pa)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);
  GdkPoint *c = priv->coord;
  GdkRectangle *r = gspa_compute_anchor_rect (anchor, pa);

  /* test if anchor hasn't been set in order to avoir long test. */
  if (priv->anchor == -1) {
    if (r->x <= c->x && c->x <= r->x + r->width &&
	r->y <= c->y && c->y <= r->y + r->height) {
      GET_PRIVATE (pa)->anchor = anchor;
    }
  }
}

/*
 * Set default anchor to -1 and search the current anchor.
 */
void
gspa_anchor_over (GnomeScanPreviewArea *pa)
{
  GET_PRIVATE (pa)->anchor = -1;
  if (GET_PRIVATE (pa)->custom) {
    gs_enum_foreach (GTK_TYPE_ANCHOR_TYPE, (GSEFunc) gspa_in_anchor, pa);
  }
  else {
    gspa_in_anchor (GTK_ANCHOR_CENTER, pa);
  }
}

/*
 * Set Window cursor considering current anchor.
 */
void
gspa_set_cursor (GtkAnchorType anchor,
		 GnomeScanPreviewArea *pa)
{
  GdkCursor *cursor;

  switch (anchor) {
  case GTK_ANCHOR_NORTH_WEST:
    cursor = gdk_cursor_new (GDK_TOP_LEFT_CORNER);
    break;
  case GTK_ANCHOR_NORTH:
    cursor = gdk_cursor_new (GDK_TOP_SIDE);
    break;
  case GTK_ANCHOR_NORTH_EAST:
    cursor = gdk_cursor_new (GDK_TOP_RIGHT_CORNER);
    break;
  case GTK_ANCHOR_EAST:
    cursor = gdk_cursor_new (GDK_RIGHT_SIDE);
    break;
  case GTK_ANCHOR_SOUTH_EAST:
    cursor = gdk_cursor_new (GDK_BOTTOM_RIGHT_CORNER);
    break;
  case GTK_ANCHOR_SOUTH:
    cursor = gdk_cursor_new (GDK_BOTTOM_SIDE);
    break;
  case GTK_ANCHOR_SOUTH_WEST:
    cursor = gdk_cursor_new (GDK_BOTTOM_LEFT_CORNER);
    break;
  case GTK_ANCHOR_WEST:
    cursor = gdk_cursor_new (GDK_LEFT_SIDE);
    break;
  case GTK_ANCHOR_CENTER:
    cursor = gdk_cursor_new (GDK_FLEUR);
    break;
  default:
    cursor = NULL;
    break;
  }

  gdk_window_set_cursor (pa->drawing_area->window,
			 cursor);
}

/*
 *
 */
void
gspa_send_area (GnomeScanPreviewArea *pa)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);
  GdkRectangle *roi;
  GnomeScanArea *area = g_new0 (GnomeScanArea, 1);

  roi = g_memdup (priv->roi, sizeof (GdkRectangle));

  /* apply back the ROI rotation */
  gspa_gdk_rectangle_rotate_simple (roi,
				    360 - priv->rotation,
				    gdk_pixbuf_get_width (priv->image),
				    gdk_pixbuf_get_height (priv->image));
  /*   debug_rect (roi); */

  area->custom = priv->custom;
  area->x = gnome_scan_mm_from_pixel (roi->x, pa->resolution);
  area->y = gnome_scan_mm_from_pixel (roi->y, pa->resolution);
  area->width = gnome_scan_mm_from_pixel (roi->width, pa->resolution);
  area->height = gnome_scan_mm_from_pixel (roi->height, pa->resolution);

  /*   debug_area (area); */

  gnome_scan_context_set_area (pa->context, area);
}

#define wrap_rotation(val)	((val) + 360) % 360
void
gspa_apply_transformations (GnomeScanPreviewArea *pa)
{
  GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (pa);
  gint width, height;
  gint tmp;
  GdkPixbuf *image;

  width = gdk_pixbuf_get_width (priv->image);
  height = gdk_pixbuf_get_height (priv->image);

  /* rotation */
  if (priv->rotate) {
    image = gdk_pixbuf_rotate_simple (priv->image, priv->rotate);
    /* free the original */
    g_object_unref (priv->image);
    /* use the copy */
    priv->image = image;
    /* store the final rotation */
    priv->rotation = wrap_rotation (priv->rotation + priv->rotate);
    /* update the desired final rotation */
    gnome_scan_context_set_rotation (pa->context, priv->rotation);

    /* rotate the ROI */
    gspa_gdk_rectangle_rotate_simple (priv->roi, priv->rotate,
				      width, height);

    /* switch width and height */
    tmp = width;
    width = height;
    height = tmp;

    /* do not rotate again ! */
    priv->rotate = 0;
  }

  gtk_widget_set_size_request (pa->drawing_area, width, height);

  /* Create auto selection */
#define set_if_not_included(var,val,min,max)	var = !((min) < var && var < (max)) ? (val) : var

  /* crop selection if oversized or extend if 0 */
  set_if_not_included(priv->roi->width, width - priv->roi->x, 0, width - priv->roi->x); 
  set_if_not_included(priv->roi->height, height - priv->roi->y, 0, height - priv->roi->y);

  gspa_deprecate_buffer (pa);
  gspa_send_area (pa);
}

/*
 * Rotate @rect inside a @width×@height image at @rotation
 * degree. @rect is modified.
 */
void
gspa_gdk_rectangle_rotate_simple (GdkRectangle *rect,
				  GdkPixbufRotation rotation,
				  gint width,
				  gint height)
{
  gint tmp;

  rotation = wrap_rotation (rotation);

  if (!rotation) {
    return;
  }

  /* rotate ROI */
  switch (rotation) {
  case GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE:
    tmp = width;
    width = height;
    height = tmp;

    tmp = rect->width;
    rect->width = rect->height;
    rect->height = tmp;

    tmp = rect->y;
    rect->y = height - rect->height - rect->x; /* old x */
    rect->x = tmp;
    break;
  case GDK_PIXBUF_ROTATE_UPSIDEDOWN:
    rect->x = width - rect->width - rect->x;
    rect->y = height - rect->height - rect->y;
    break;
  case GDK_PIXBUF_ROTATE_CLOCKWISE:
    tmp = width;
    width = height;
    height = tmp;

    tmp = rect->width;
    rect->width = rect->height;
    rect->height = tmp;

    tmp = rect->x;
    rect->x = width - rect->width - rect->y; /* old y */
    rect->y = tmp;
    break;
  }
}
