/* GnomeScan - Gnome Scanning Infrastructure
 *
 * gnome-scanbackend.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/*
 * The huge work in the SANE GnomeScanBackend is to triage options. In
 * SANE, "options" means scan option, device option, option groups and
 * buttons in the same structure.
 *
 * GnomeScanBackend is not really an object. It still has some
 * procedural design. But it allow a real object to handle scan :
 * GnomeScanContext.
 */

#include <string.h>
#define	SANE_I18N(text)	_(text)
#include <sane/sane.h>
#include <sane/saneopts.h>
#include "gnomescan.h"
#include "gnomescan-intl.h"

#define GNOME_SCAN_BACKEND_ERROR		(gnome_scan_backend_error_quark ())
#define	GNOME_SCAN_BACKEND_PARENT_CLASS(klass)	(G_OBJECT_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)			(G_TYPE_INSTANCE_GET_PRIVATE((obj), GNOME_TYPE_SCAN_BACKEND, GnomeScanBackendPrivate))

typedef struct _GnomeScanBackendPrivate		GnomeScanBackendPrivate;
typedef struct _GnomeScannerId			GnomeScannerId;
typedef struct _GnomeScannerOptionId		GnomeScannerOptionId;
typedef GValue* (*GSBOptionValueFunc)		(GnomeScannerId* sid,
						 GnomeScannerOptionId* oid,
						 GnomeScanContextInfo* info);

struct _GnomeScanBackendPrivate {
  /* opened sids */
  GSList 			*sids;
};

/* PRIVATE TYPES */
struct _GnomeScannerId {
  const SANE_Device	*descriptor;
  /*
   * We store an handle per device in order to keep valid option
   * descriptor. This will allow to monitor each devices in dbus
   * demons.
   */
  SANE_Handle		handle;
  gboolean		opened;
  /* device options are publics */
  GHashTable*		options;
  /* scan options are hidden */
  GHashTable*		hidden;
  GSList*		buttons;
  /* sources flags correspondance table */
  GHashTable		*sources;
  GHashTable		*colorspaces;
};

struct _GnomeScannerOptionId {
  /*
   * Since we store an handle per device, option id and descriptor
   * remain valid while the backend is instanciate. We can store this
   * data. No need to re-get option descriptor since we don't re-open
   * devices.
   */
  gint				id;
  const SANE_Option_Descriptor*	descriptor;
};



/********************************
 * 	    SANE WRAP		*
 ********************************/

/* Scanner functions */
GnomeScanner*			gsb_scanner_new				(GnomeScanBackend *backend,
									 GnomeScannerId *sid,
									 GError **error);

void				gsb_scanner_destroy			(GnomeScannerId *sid);

GnomeScannerType		gsb_scanner_get_type			(GnomeScannerId *sid);
GnomeScanGeometry*		gsb_scanner_get_geometry		(GnomeScannerId *sid);
GValueArray*			gsb_scanner_get_sources			(GnomeScannerId *sid);
GValueArray*			gsb_scanner_get_colorspaces		(GnomeScannerId *sid);
GValueArray*			gsb_scanner_get_depths			(GnomeScannerId *sid);

GHashTable*			gsb_scanner_get_options			(GnomeScannerId *sid,
									 GError **error);

gboolean			gsb_scanner_steal_button		(gchar *key,
									 GnomeScannerOption *option,
									 GnomeScannerId *sid);

void				gsb_scanner_reload_options		(GnomeScannerId *sid);

/* scanner sources function */
void				gsb_scanner_source_parse		(GValue *value,
									 GnomeScannerId *sid);

void				gsb_scanner_source_add			(GnomeScannerSource *value,
									 const gchar *source,
									 GValueArray *array);

/* scan colorspace function */
void				gsb_scanner_colorspace_parse		(GValue *value,
									 GnomeScannerId *sid);

void				gsb_scanner_colorspace_add		(GnomeScanColorspace *value,
									 const gchar *colorspace,
									 GValueArray *array);

void				gsb_scanner_depth_add			(GValue *value,
									 GValueArray *array);

void                            gsb_scanner_resolution_add              (GValue *value,
                                                                         GValueArray *array);

/* (real) Options functions (either scan or device options) */
GValue*				gsb_scanner_option_get_value		(GnomeScannerId *sid,
									 GnomeScannerOptionId *oid,
									 GError **error);

void				gsb_scanner_option_set_value		(GnomeScannerId *sid,
									 GnomeScannerOptionId *oid,
									 GValue *value,
									 GError **error);

GType				gsb_scanner_option_get_type		(GnomeScannerOptionId *oid);
GnomeScanUnit			gsb_scanner_option_get_unit		(GnomeScannerOptionId *oid);
GnomeScannerOptionConstraint*	gsb_scanner_option_get_constraint	(GnomeScannerOptionId *oid);
GnomeScannerOptionFlags		gsb_scanner_option_get_flags		(GnomeScannerOptionId *oid);

void				gsb_scanner_option_reload		(gconstpointer key,
									 GnomeScannerOption *option,
									 GnomeScannerId *sid);

GnomeScanForecast*		gsb_scan_forecast			(GnomeScanContextInfo *info,
									 GError **error);

/* apply functions */

void				gsb_scan_apply_option			(GnomeScanContextInfo *info,
									 GnomeScannerId *sid,
									 const gchar* oname,
									 GSBOptionValueFunc value_func,
									 GError **error);


GValue*				gsb_scan_apply_option_value_resolution	(GnomeScannerId *sid,
									 GnomeScannerOptionId *oid,
									 GnomeScanContextInfo *info);

GValue*				gsb_scan_apply_option_value_area	(GnomeScannerId *sid,
									 GnomeScannerOptionId *oid,
									 GnomeScanContextInfo *info);

GValue*				gsb_scan_apply_option_value_preview	(GnomeScannerId *sid,
									 GnomeScannerOptionId *oid,
									 GnomeScanContextInfo *info);

GValue*				gsb_scan_apply_option_value_source	(GnomeScannerId *sid,
									 GnomeScannerOptionId *oid,
									 GnomeScanContextInfo *info);

GValue*				gsb_scan_apply_option_value_colorspace	(GnomeScannerId *sid,
									 GnomeScannerOptionId *oid,
									 GnomeScanContextInfo *info);

GValue*				gsb_scan_apply_option_value_depth	(GnomeScannerId *sid,
									 GnomeScannerOptionId *oid,
									 GnomeScanContextInfo *info);

GValue*				gsb_scan_apply_option_value_test_picture(GnomeScannerId *sid,
									 GnomeScannerOptionId *oid,
									 GnomeScanContextInfo *info);

void				gsb_scan_apply_context			(GnomeScanContextInfo *info,
									 GError **error);

/* Acquisition functions */

GnomeScanResult*		gsb_acquire_one				(GnomeScanBackend *backend,
									 GnomeScannerId *sid,
									 GnomeScanContextInfo *info,
									 GError **error);

void				gsb_convert_data_to_rgb			(SANE_Frame format,
									 gint depth,
									 guchar *data,
									 guchar *pixels,
									 gint offset,
									 gint length);


/* misc */
GError*				gsb_get_error				(SANE_Status status);

/********************************
 * 	      GOBJECT		*
 ********************************/


enum {
  DATA_RECEIVED,
  IMAGE_ACQUIRED,
  N_SIGNALS
};

GQuark				gnome_scan_backend_error_quark			(void);

static guint signals[N_SIGNALS];

G_DEFINE_TYPE (GnomeScanBackend, gnome_scan_backend, G_TYPE_OBJECT)

     void
gnome_scan_backend_class_init (GnomeScanBackendClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (gobject_class, sizeof (GnomeScanBackendPrivate));

  /* SIGNAL
   *
   * We avoid signals in backend since we might have several
   * backend. Most signals are in GnomeScanContext.
   */

  /**
   * GnomeScanBackend::data-received:
   * @backend: the emitting #GnomeScanBackend
   * @data_received: the amount of value received
   *
   * Emitted when the backend received data from the scanner during an
   * acquisition.
   **/
  signals[DATA_RECEIVED] = g_signal_new ("data-received",
					 GNOME_TYPE_SCAN_BACKEND,
					 G_SIGNAL_RUN_FIRST,
					 G_STRUCT_OFFSET (GnomeScanBackendClass, data_received),
					 NULL,
					 NULL,
					 g_cclosure_marshal_VOID__INT,
					 G_TYPE_NONE,
					 1,
					 G_TYPE_INT);

  /**
   * GnomeScanBackend::image-acquired:
   * @backend: the emitting #GnomeScanBackend
   * @result: the scan result containing the acquired image
   *
   * Emitted when the backend successfully acquired an image.
   **/
  signals[IMAGE_ACQUIRED] = g_signal_new ("image-acquired",
					  GNOME_TYPE_SCAN_BACKEND,
					  G_SIGNAL_RUN_FIRST,
					  G_STRUCT_OFFSET (GnomeScanBackendClass, image_acquired),
					  NULL,
					  NULL,
					  g_cclosure_marshal_VOID__BOXED,
					  G_TYPE_NONE,
					  1,
					  GNOME_TYPE_SCAN_RESULT);
}

void
gnome_scan_backend_init (GnomeScanBackend *backend)
{
  GnomeScanBackendPrivate *priv = GET_PRIVATE (backend);
}

GQuark
gnome_scan_backend_error_quark (void)
{
  return g_quark_from_static_string ("gnome-scan-backend-error-quark");
}



/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_backend_new:
 * @error:	A #GError
 * 
 * Instanciate a new backend. This function is also responsible to
 * init the third party backend (e.g. SANE).
 * 
 * Return value: a new #GnomeScanBackend
 **/
GnomeScanBackend*
gnome_scan_backend_new (GError **error)
{
  SANE_Status status;
  gint version;

  g_debug ("%s %s", PACKAGE, VERSION);

  /* ONLY SUPPORT SANE 1 */
  if (SANE_CURRENT_MAJOR != 1) {
    *error = g_error_new (GNOME_SCAN_BACKEND_ERROR,
			  GNOME_SCAN_BACKEND_ERROR_UNSUPPORTED_SANE_VERSION,
			  /* Translators, %s is the version of libgnomescan. */
			  _("GnomeScanBackend %s support only SANE version 1.X"), VERSION);
    return NULL;
  }

  /* TODO: authorization handling */
  g_debug ("sane_init(%p, NULL)", &version);
  status = sane_init (&version, NULL);
  g_debug ("SANE Version = %i.%i.%i", SANE_VERSION_MAJOR (version), SANE_VERSION_MINOR (version), SANE_VERSION_BUILD (version));

  /* HANDLE SANE ERROR ON INIT */
  if (*error = gsb_get_error (status)) {
    g_critical ("Unable to init SANE : %s", (*error)->message);
    return;
  }

  return g_object_new (GNOME_TYPE_SCAN_BACKEND, NULL);
}

/**
 * gnome_scan_backend_destroy:
 * @backend: a #GnomeScanBackend
 * 
 * Destroy a backend and free SANE resources (i.e. SANE).
 **/
void
gnome_scan_backend_destroy (GnomeScanBackend *backend)
{
  g_return_if_fail (GNOME_IS_SCAN_BACKEND (backend));
  GnomeScanBackendPrivate *priv = GET_PRIVATE (backend);

  g_debug ("Closing devices");
  g_slist_foreach (priv->sids, (GFunc) gsb_scanner_destroy, NULL);
  g_slist_free (priv->sids);

  g_debug ("sane_exit ()");
  sane_exit ();
  g_object_unref (backend);
}

/**
 * gnome_scan_backend_probe_scanners:
 * @backend:	a #GnomeScanBackend
 * @error:	a #GError
 * 
 * Do the long device probe task and return a #GSList of
 * #GnomeScanner.
 *
 * Return value: the scanners list.
 **/
GSList*
gnome_scan_backend_probe_scanners (GnomeScanBackend *backend,
				   GError **error)
{
  g_return_val_if_fail (GNOME_IS_SCAN_BACKEND (backend), NULL);
  SANE_Status status;
  GnomeScannerId *sid;
  GnomeScanner *scanner;
  GSList *scanners;
  gint i;

#if	ENABLE_PROBE
  const SANE_Device **devices;
  g_debug ("sane_get_devices (%p, FALSE)", &devices);
  status = sane_get_devices (&devices, FALSE);
#else
  g_debug ("Fake device probe. Using 'test' SANE device.");
  SANE_Device **devices;
  devices = g_new0(SANE_Device*, 2);
  devices[0] = g_new0(SANE_Device, 1);
  devices[0]->name = "test";
  devices[0]->vendor = "Gnome Scan";
  devices[0]->model = "Test device";
  devices[0]->type = "virtual device";
  status = SANE_STATUS_GOOD;
#endif

  scanners = NULL;
  if (*error = gsb_get_error (status)) {
    g_debug ("%s", (*error)->message);
  }
  else {
    for (i = 0; devices[i] && !(*error); i++) {
      sid = g_new0 (GnomeScannerId, 1);
      sid->descriptor = devices[i];
      scanner = gsb_scanner_new (backend, sid, error);
      if (!*error) {
	scanners = g_slist_append (scanners, scanner);
      }
    }
  }

  return scanners;
}

/**
 * gnome_scan_backend_forecast:
 * @backend: a #GnomeScanBackend
 * @info: a scan context
 * @error: a #GError
 * 
 * Compute acquisition forecast.
 * 
 * Returns: The forecast or NULL if error.
 **/
GnomeScanForecast*
gnome_scan_backend_forecast (GnomeScanBackend *backend,
			     GnomeScanContextInfo *info,
			     GError **error)
{
  g_return_val_if_fail (GNOME_IS_SCAN_BACKEND (backend), NULL);
  GnomeScanForecast *forecast = NULL;

  gsb_scan_apply_context (info, error);
  if (!(*error)) {
    forecast = gsb_scan_forecast (info, error);
  }
  return forecast;
}

/**
 * gnome_scan_backend_acquire:
 * @backend: a #GnomeScanBackend
 * @info: a #GnomeScanContextInfo
 * @error: a #GError or NULL
 * 
 * Do the acquisition work and return the picture in a #GdkPixbuf. The
 * #GnomeScanBackend::"data-received" signal is emitted each time the
 * device sent data. You should use
 * gnome_scan_context_start_acquisition().
 * 
 **/
void
gnome_scan_backend_acquire (GnomeScanBackend *backend,
			    GnomeScanContextInfo *info,
			    GError **error)
{
  g_return_if_fail (GNOME_IS_SCAN_BACKEND (backend));
  g_return_if_fail (GNOME_IS_SCANNER (info->scanner));

  SANE_Status status;
  GnomeScannerId *sid;
  GnomeScanResult *result;

  gsb_scan_apply_context (info, error);
  if (*error) {
    g_debug ("Failed to apply options !");
  }
  else {
    sid = gnome_scanner_get_id (info->scanner);
    do {
      result = gsb_acquire_one (backend, sid, info, error);
      if (result) {
	g_signal_emit_by_name (backend, "image-acquired", result);
	g_boxed_free (GNOME_TYPE_SCAN_RESULT, result);
      }
    } while (!info->preview &&
	     info->source == GNOME_SCANNER_SOURCE_AUTOMATIC_DOCUMENT_FEEDER
	     && result);
  }
}

/**
 * gnome_scan_backend_stop:
 * @backend: a #GnomeScanBackend
 * 
 * Cancel acquisition (preview or final). You should use
 * gnome_scan_context_stop_acquisition() instead.
 **/
void
gnome_scan_backend_stop (GnomeScanBackend *backend,
			 GnomeScanner *scanner)
{
  g_return_if_fail (GNOME_IS_SCAN_BACKEND (backend));
  GnomeScannerId *sid = gnome_scanner_get_id (scanner);
  g_debug ("sane_cancel ()");
  sane_cancel (sid->handle);
}

/********************************
 * 	    SANE WRAP		*
 ********************************/

/* SCANNER FUNCTIONS */

/*
 * Create a GnomeScanner from a SANE_Device and more. This is the true
 * gnome_scanner_new() function.
 */
GnomeScanner*
gsb_scanner_new (GnomeScanBackend *backend,
		 GnomeScannerId *sid,
		 GError **error)
{
  GnomeScanBackendPrivate *priv = GET_PRIVATE (backend);
  SANE_Status status;
  GnomeScanner *scanner;
  gchar *vendor,  *product;
  GnomeScannerType type;
  GnomeScannerOption *option;
  GValueArray *sources, *colorspaces, *depths;
  GnomeScanGeometry *geometry;
  GnomeScannerOptionConstraint *constraint;
  GnomeScanRange *resolution_range;
  GValueArray *resolution_enum;

  /*
   * Now we go further sane, we probe all options and distribute them
   * in various properties
   */
  status = sane_open (sid->descriptor->name, &(sid->handle));
  if (*error = gsb_get_error (status)) {
    scanner = NULL;
  }
  else {
    sid->opened = TRUE;
    priv->sids = g_slist_prepend (priv->sids, sid);

    vendor = g_strdup (sid->descriptor->vendor);
    product = g_strdup (sid->descriptor->model);
    type = gsb_scanner_get_type (sid);

    g_debug ("Creating %s Scanner %s %s (%s)", gs_enum_get_nick (GNOME_TYPE_SCANNER_TYPE, type), vendor, product, sid->descriptor->name);
    sid->options = gsb_scanner_get_options (sid, error);
    if (*error) {
      g_debug ("Failed to get %s options !", sid->descriptor->name);
    }
    else {
      /*
       * Move scan option to hidden options since we want to provide
       * only device option in "options" property. Most of this options are "Well-known options in SANE".
       */
      sid->hidden = g_hash_table_new (g_str_hash, g_str_equal);
#define gsb_scanner_hide_option(b,s,o)	option = g_hash_table_lookup (s->options, o); if (g_hash_table_steal (s->options, o)) g_hash_table_insert (s->hidden, o, option)
      gsb_scanner_hide_option (backend, sid, "resolution");
      gsb_scanner_hide_option (backend, sid, "preview");
      gsb_scanner_hide_option (backend, sid, "tl-x");
      gsb_scanner_hide_option (backend, sid, "tl-y");
      gsb_scanner_hide_option (backend, sid, "br-x");
      gsb_scanner_hide_option (backend, sid, "br-y");
      gsb_scanner_hide_option (backend, sid, "depth");
      gsb_scanner_hide_option (backend, sid, "mode");
      gsb_scanner_hide_option (backend, sid, "source");
      /* gamma, shadow, highlight, batch, compression */
#undef	gsb_scanner_hide_option

      /* search for buttons. */
      g_hash_table_foreach_steal (sid->options, (GHRFunc) gsb_scanner_steal_button, sid);

      option = g_hash_table_lookup (sid->hidden, "resolution");
      if (option) {
        constraint = gnome_scanner_option_get_constraint (option);

      	if (gnome_scanner_option_get_flags(option) & GNOME_SCANNER_OPTION_RANGE) {
	  resolution_range = constraint->range;
        }
	else {
	resolution_range = NULL;
	}

        if (gnome_scanner_option_get_flags(option) & GNOME_SCANNER_OPTION_ENUMERATION) {
	  resolution_enum = g_value_array_new (g_slist_length (constraint->enumeration));
	  g_slist_foreach (constraint->enumeration, (GFunc) gsb_scanner_resolution_add, resolution_enum);
        }
	else {
	resolution_enum = NULL;
	}
      }

      sources 		= gsb_scanner_get_sources (sid);
      colorspaces	= gsb_scanner_get_colorspaces (sid);
      depths		= gsb_scanner_get_depths (sid);
      geometry  	= gsb_scanner_get_geometry (sid);
      scanner = g_object_new (GNOME_TYPE_SCANNER,
			      "id", sid,
			      "vendor", vendor,
			      "product", product,
			      "type", type,
			      "geometry", geometry,
			      "sources", sources,
			      "colorspaces", colorspaces,
			      "depths", depths,
			      "options", sid->options,
			      "resolution-range", resolution_range,
			      "resolution-enum", resolution_enum,
			      NULL);
    }
  }

  return scanner;
}

/* close an opened devices */
void
gsb_scanner_destroy (GnomeScannerId *sid)
{
  if (sid->opened == TRUE) {
    g_debug ("Destroying %s", sid->descriptor->name);
    sane_close (sid->handle);
    sid->opened = FALSE;
  }
}

/* translate sane device type string into GnomeScannerType enum */
GnomeScannerType
gsb_scanner_get_type (GnomeScannerId *sid)
{
  const gchar *type = sid->descriptor->type;
  GnomeScannerType result;

  if (g_strrstr (type, "flatbed scanner")) {
    result = GNOME_SCANNER_FLATBED;
  }
  else if (gs_strcmp (type, "multi-function peripheral") == 0) {
    result = GNOME_SCANNER_MULTI_FUNCTION;
  }
  else if (gs_strcmp (type, "handheld scanner") == 0) {
    result = GNOME_SCANNER_HANDHELD;
  }
  else if (gs_strcmp (type, "sheetfed scanner") == 0) {
    result = GNOME_SCANNER_SHEETFED;
  }
  else {
    result = GNOME_SCANNER_UNKNOWN;
  }

  return result;
}


/*
 * Returns all options in a GHashTable with name as key.
 */
GHashTable*
gsb_scanner_get_options (GnomeScannerId *sid,
			 GError **error)
{
  GHashTable *options;
  GnomeScannerOptionId *oid;
  GnomeScannerOption *option;
  GValue *value;
  gint i, count;

  /* first n=0 option is option count + group count + button count. */
  oid = g_new0 (GnomeScannerOptionId, 1);
  oid->descriptor = sane_get_option_descriptor (sid->handle, oid->id);
  value = gsb_scanner_option_get_value (sid, oid, error);

  if (*error) {
    g_debug ("Unable to get options count !");
  }
  else {
    count = g_value_get_int (value);
    g_free (value);

    options = g_hash_table_new (g_str_hash, g_str_equal);

    /* Get other options */
    for (i = 1; i < count; i++) {
      oid = g_new0 (GnomeScannerOptionId, 1);
      oid->id = i;
      oid->descriptor = sane_get_option_descriptor (sid->handle, oid->id);

      switch (oid->descriptor->type) {
      case SANE_TYPE_BUTTON:
	/* would be nice to have a button handling. but buttons are
	   often declared as option having "button" in the name :( */
      case SANE_TYPE_GROUP:
	/* Also, would be nice to have group handling. But since we have
	   a complicated options triaging work, that's going to be very
	   tricky to triage groups. Luckily, groups have options of the
	   same kind : buttons or scan or device. */
	break;
      case SANE_TYPE_BOOL:
      case SANE_TYPE_INT:
      case SANE_TYPE_FIXED:
      case SANE_TYPE_STRING:
	g_debug ("Creating option %s(%i) '%s'", g_strdup (oid->descriptor->name), oid->id, g_strdup (oid->descriptor->title));
	value = gsb_scanner_option_get_value (sid, oid, error);
	if (!(*error)) {
	  option = g_object_new (GNOME_TYPE_SCANNER_OPTION,
				 "id", oid,
				 "name", g_strdup (oid->descriptor->name),
				 "title", g_strdup (oid->descriptor->title),
				 "desc", g_strdup (oid->descriptor->desc),
				 "value", value,
				 "type", G_VALUE_TYPE (value),
				 "unit", gsb_scanner_option_get_unit (oid),
				 "constraint", gsb_scanner_option_get_constraint (oid),
				 "flags", gsb_scanner_option_get_flags (oid),
				 NULL);

	  g_hash_table_insert (options, g_strdup (gnome_scanner_option_get_name (option)), option);
	}
	else {
	  g_debug ("Failed to create option %s '%s'", g_strdup (oid->descriptor->name), g_strdup (oid->descriptor->title));
	}
	break;
      }
    }
  }

  return options;
}

/*
 * Get the scanner geometry from scan area range.
 */
GnomeScanGeometry*
gsb_scanner_get_geometry (GnomeScannerId *sid)
{
  GnomeScanGeometry *geometry;
  GnomeScannerOption *option;
  GnomeScannerOptionConstraint *constraint;

  geometry = g_new0 (GnomeScanGeometry, 1);

  option = g_hash_table_lookup (sid->hidden, "br-x");
  constraint = gnome_scanner_option_get_constraint (option);
  geometry->width = constraint->range->upper;

  option = g_hash_table_lookup (sid->hidden, "br-y");
  constraint = gnome_scanner_option_get_constraint (option);
  geometry->height = constraint->range->upper;

  return geometry;
}


/*
 * Parse sources from source option enum constraint. Since backend use
 * different string to determine available sources, we have to store
 * those value in a per device hash table in order to retrieve the
 * correct value using enum values as index. SANE 2 add a
 * naming convention that help that. :)
 */
GValueArray*
gsb_scanner_get_sources (GnomeScannerId *sid)
{
  GValueArray* result;
  GnomeScannerOption *option;
  GnomeScannerOptionConstraint *constraint;

  option = g_hash_table_lookup (sid->hidden, "source");
  if (option) {
    sid->sources = g_hash_table_new (g_int_hash, g_int_equal);
    constraint = gnome_scanner_option_get_constraint (option);
    g_slist_foreach (constraint->enumeration, (GFunc) gsb_scanner_source_parse, sid);

    result = g_value_array_new (g_hash_table_size (sid->sources));
    g_hash_table_foreach (sid->sources, (GHFunc) gsb_scanner_source_add, result);
  }
  else {
    g_debug ("Emulating option 'source'");
    result = g_value_array_new (1);
    g_value_array_append (result, gs_value_new_enum (GNOME_TYPE_SCANNER_SOURCE, GNOME_SCANNER_SOURCE_FLATBED));
  }

  return result;
}

/**
 * Parse mode in return each mode in an array of colorspace.
 */
GValueArray*
gsb_scanner_get_colorspaces (GnomeScannerId *sid)
{
  GValueArray* result;
  GnomeScannerOption *option;
  GnomeScannerOptionConstraint *constraint;

  option = g_hash_table_lookup (sid->hidden, "mode");
  if (option) {
    sid->colorspaces = g_hash_table_new (g_int_hash, g_int_equal);
    constraint = gnome_scanner_option_get_constraint (option);
    g_slist_foreach (constraint->enumeration, (GFunc) gsb_scanner_colorspace_parse, sid);

    result = g_value_array_new (g_hash_table_size (sid->colorspaces));
    g_hash_table_foreach (sid->colorspaces, (GHFunc) gsb_scanner_colorspace_add, result);
  }
  else {
    g_debug ("Emulating option 'mode' allowing only RGB.");
    result = g_value_array_new (1);
    g_value_array_append (result, gs_value_new_enum (GNOME_TYPE_SCAN_COLORSPACE, GNOME_SCAN_COLORSPACE_RGB));
  }

  return result;
}

GValueArray*
gsb_scanner_get_depths (GnomeScannerId *sid)
{
  GValueArray *result;
  GnomeScannerOption *option;
  GnomeScannerOptionConstraint *constraint;


  option = g_hash_table_lookup (sid->hidden, "depth");
  if (option) {
    constraint = gnome_scanner_option_get_constraint (option);
    result = g_value_array_new (g_slist_length (constraint->enumeration));
    g_slist_foreach (constraint->enumeration, (GFunc) gsb_scanner_depth_add, result);
  }
  else {
    g_debug ("Emulating option 'depth'");
    result = g_value_array_new (1);
    g_value_array_append (result, gs_value_new_int (8));
  }

  return result;
}

/*
 * In SANE 1, buttons are option with button in the name. We separate
 * button from other options.
 */
gboolean
gsb_scanner_steal_button (gchar *key,
			  GnomeScannerOption *option,
			  GnomeScannerId *sid)
{
  if (g_strrstr (gnome_scanner_option_get_name (option), "button")) {
    sid->buttons = g_slist_append (sid->buttons, option);
    return TRUE;
  }
  return FALSE;
}

void
gsb_scanner_reload_options (GnomeScannerId *sid)
{
  g_hash_table_foreach (sid->options, (GHFunc) gsb_scanner_option_reload, sid);
  g_hash_table_foreach (sid->hidden, (GHFunc) gsb_scanner_option_reload, sid);
}


/* SCANNER SOURCES FUNCTIONS */

/*
 * Parse a source and keep string <=> flag correspondance in a
 * sid->sources hash table.
 */
void
gsb_scanner_source_parse (GValue *value,
			  GnomeScannerId *sid)
{
  const gchar *str = g_value_get_string (value);
  gint *source = g_new0 (gint, 1);
  *source = GNOME_SCANNER_SOURCE_NONE;

  /* all tests :(
   *
   * Note that this is incomplete. Have to read all backend code in
   * order to know which string they use.
   *
   * Tested only with test, plustek, hpoj and hpaio backend.
   */
  if (gs_strcmp (str, "Flatbed") == 0) {
    *source = GNOME_SCANNER_SOURCE_FLATBED;
  }
  else if (gs_strcmp (str, "Automatic Document Feeder") == 0) {
    *source = GNOME_SCANNER_SOURCE_AUTOMATIC_DOCUMENT_FEEDER;
  }
  else if (gs_strcmp (str, "ADF") == 0) {
    *source = GNOME_SCANNER_SOURCE_AUTOMATIC_DOCUMENT_FEEDER;
  }
  else if (gs_strcmp (str, "Auto") == 0) {
    *source = GNOME_SCANNER_SOURCE_AUTOMATIC;
  }
  else if (gs_strcmp (str, "Normal") == 0) {
    *source = GNOME_SCANNER_SOURCE_FLATBED;
  }

  if (*source) {
    g_hash_table_insert (sid->sources, source, g_strdup (str));
  }
}

/*
 * Add sources to GValueArray
 */
void
gsb_scanner_source_add (GnomeScannerSource *value,
			const gchar *source,
			GValueArray *array)
{
  g_value_array_append (array, gs_value_new_enum (GNOME_TYPE_SCANNER_SOURCE, *value));
}

void
gsb_scanner_depth_add (GValue *value,
		       GValueArray *array)
{
  g_value_array_append (array, gs_value_new_int (g_value_get_int (value)));
}

void
gsb_scanner_resolution_add (GValue *value,
                       GValueArray *array)
{
  GValue *vdouble = gs_value_new_double (0.);
  g_value_transform (value, vdouble);
  g_value_array_append (array, vdouble);
}


/* SCAN COLORSPACES FUNCTIONS */

/*
 * Parse a colorspace and keep string <=> flag correspondance in a
 * sid->colorspaces hash table.
 */
void
gsb_scanner_colorspace_parse (GValue *value,
			      GnomeScannerId *sid)
{
  const gchar *str = g_value_get_string (value);
  gint *colorspace = g_new0 (gint, 1);

  /* all tests :(
   *
   * Note that this is incomplete. Have to read all backend code in
   * order to know which string they use.
   *
   * Tested only with plustek, hpoj and hpaio backend.
   */
  if (gs_strcmp (str, "Gray") == 0) {
    *colorspace = GNOME_SCAN_COLORSPACE_GRAY;
  }
  else if (gs_strcmp (str, "Lineart") == 0) {
    *colorspace = GNOME_SCAN_COLORSPACE_LINEART;
  }
  else if (gs_strcmp (str, "Halftone") == 0) {
    *colorspace = GNOME_SCAN_COLORSPACE_HALFTONE;
  }
  else {
    *colorspace = GNOME_SCAN_COLORSPACE_RGB;
  }

  g_hash_table_insert (sid->colorspaces, colorspace, g_strdup (str));
}

/*
 * Add colorspaces to GValueArray
 */
void
gsb_scanner_colorspace_add (GnomeScanColorspace *value,
			    const gchar *colorspace,
			    GValueArray *array)
{
  g_value_array_append (array, gs_value_new_enum (GNOME_TYPE_SCAN_COLORSPACE, *value));
}

/* SCANNER OPTION FUNCTIONS */


/*
 * Translate SANE Option value to GValue.
 */
GValue*
gsb_scanner_option_get_value (GnomeScannerId *sid,
			      GnomeScannerOptionId *oid,
			      GError **error)
{
  SANE_Status status;
  GValue *value;
  GType type;
  gpointer data;

  value = g_new0 (GValue, 1);
  type = gsb_scanner_option_get_type (oid);
  g_value_init (value, type);

  /* I really doubt this is the right way to allocate data, especially for int, bool and fixed :( */
  data = g_malloc0 (oid->descriptor->size);

  /*   g_debug ("%s:%i sane_control_option (%s, %s)", __FUNCTION__, __LINE__, sid->descriptor->model, oid->descriptor->name); */
  status = sane_control_option (sid->handle, oid->id, SANE_ACTION_GET_VALUE, data, NULL);

  switch (type) {
  case G_TYPE_BOOLEAN:
    g_value_set_boolean (value, *((gboolean*) data));
    break;
  case G_TYPE_INT:
    g_value_set_int (value, *((gint*) data));
    break;
  case G_TYPE_DOUBLE:
    g_value_set_double (value, SANE_UNFIX (*((gint*) data)));
    break;
  case G_TYPE_STRING:
    g_value_set_string (value, data);
    break;
  default:
    break;
  }
  g_debug (" value\t= %s", g_strdup_value_contents (value));

  return value;
}

/*
 * If value is NULL, then the option is set the backend is ask to set
 * the value automatically.
 */
void
gsb_scanner_option_set_value (GnomeScannerId *sid,
			      GnomeScannerOptionId *oid,
			      GValue *value,
			      GError **error)
{
  SANE_Status status;
  gpointer data;
  gboolean boolean;
  gint info, action, integer;

  if (oid->descriptor->cap & SANE_CAP_INACTIVE) {
    g_debug ("Not setting inactive option %s %s", sid->descriptor->name, oid->descriptor->name);
    return;
  }

  if (value) {
    action = SANE_ACTION_SET_VALUE;

    switch (G_VALUE_TYPE (value)) {
    case G_TYPE_INT:
      integer = g_value_get_int (value);
      data = &integer;
      break;
    case G_TYPE_DOUBLE:
      integer = SANE_FIX (g_value_get_double (value));
      data = &integer;
      break;
    case G_TYPE_BOOLEAN:
      boolean = g_value_get_boolean (value);
      data = &boolean;
      break;
    case G_TYPE_STRING:
      data = g_strdup (g_value_get_string (value));
      break;
    default:
      break;
    }
  }
  else if (oid->descriptor->cap & SANE_CAP_HARD_SELECT) {
    g_debug ("Auto set option value for %s %s", sid->descriptor->name, oid->descriptor->name);
    data = NULL;
    action = SANE_ACTION_SET_AUTO;
  }
  else {
    g_debug ("Not setting option %s %s !", sid->descriptor->name, oid->descriptor->name);
    *error = g_error_new (GNOME_SCAN_BACKEND_ERROR,
			  GNOME_SCAN_BACKEND_ERROR_FAILED,
			  "Option %s of device %s is not auto settable, you must provide a non-NULL value.",
			  sid->descriptor->name, oid->descriptor->name);
    return;
  }

  g_debug ("Setting option %s %s = %s", sid->descriptor->name, oid->descriptor->name, g_strdup_value_contents (value));
  status = sane_control_option (sid->handle, oid->id, action, data, &info);
  *error = gsb_get_error (status);

  if (info & SANE_INFO_INEXACT || info & SANE_INFO_RELOAD_OPTIONS) {
    gsb_scanner_reload_options (sid);
  }
}



/*
 * Convert SANE_TYPE_*
 */
GType
gsb_scanner_option_get_type (GnomeScannerOptionId *oid)
{
  GType type;

  /* workaround: bug in epson which declare 0th option as boolean. */
  if (oid->id == 0) {
    type = G_TYPE_INT;
  }
  else {

    switch (oid->descriptor->type) {
    case SANE_TYPE_BOOL:
      type = G_TYPE_BOOLEAN;
      break;
    case SANE_TYPE_INT:
      type = G_TYPE_INT;
      break;
    case SANE_TYPE_FIXED:
      type = G_TYPE_DOUBLE;
      break;
    case SANE_TYPE_STRING:
      type = G_TYPE_STRING;
      break;
    default:
      type = G_TYPE_NONE;
      break;
    }
  }

  g_debug (" type\t= %s", g_type_name (type));

  return type;
}

/*
 * Convert SANE_Unit to GnomeScanUnit.
 */
GnomeScanUnit
gsb_scanner_option_get_unit (GnomeScannerOptionId *oid)
{
  GnomeScanUnit unit;

  /* This switch may be completly useless since SANE_UNIT_$1 == GNOME_SCAN_UNIT_$1. */
  switch (oid->descriptor->type) {
  case SANE_UNIT_PIXEL:
    unit = GNOME_SCAN_UNIT_PIXEL;
    break;
  case SANE_UNIT_BIT:
    unit = GNOME_SCAN_UNIT_BIT;
    break;
  case SANE_UNIT_MM:
    unit = GNOME_SCAN_UNIT_MM;
    break;
  case SANE_UNIT_DPI:
    unit = GNOME_SCAN_UNIT_DPI;
    break;
  case SANE_UNIT_PERCENT:
    unit = GNOME_SCAN_UNIT_PERCENT;
    break;
  case SANE_UNIT_MICROSECOND:
    unit = GNOME_SCAN_UNIT_MICROSECOND;
    break;
  case SANE_UNIT_NONE:
  default:
    unit = GNOME_SCAN_UNIT_NONE;
    break;
  }

  g_debug (" unit\t= %s", gs_enum_get_nick (GNOME_TYPE_SCAN_UNIT, unit));
  return unit;
}

/*
 * Convert SANE constraint to GnomeScannerOptionConstraint.
 */
GnomeScannerOptionConstraint*
gsb_scanner_option_get_constraint (GnomeScannerOptionId *oid)
{
  GnomeScannerOptionConstraint *constraint;
  GValue *value;
  gchararray string;
  gint i, count;

  constraint = g_new0 (GnomeScannerOptionConstraint, 1);

  switch (oid->descriptor->constraint_type) {
  case SANE_CONSTRAINT_RANGE:
    /* values belong to {x = lower + k * step; k € N; x < upper} */
    constraint->range = g_new0 (GnomeScanRange, 1);
    switch (oid->descriptor->type) {
    case SANE_TYPE_INT:
      constraint->range->lower = (gdouble) oid->descriptor->constraint.range->min;
      constraint->range->upper = (gdouble) oid->descriptor->constraint.range->max;
      constraint->range->step = (gdouble) oid->descriptor->constraint.range->quant;
      break;
    case SANE_TYPE_FIXED:
      constraint->range->lower = SANE_UNFIX (oid->descriptor->constraint.range->min);
      constraint->range->upper = SANE_UNFIX (oid->descriptor->constraint.range->max);
      constraint->range->step = SANE_UNFIX (oid->descriptor->constraint.range->quant);
      break;
    default:
      break;
    }
    g_debug (" range\t= [%f ; %f] step = %f", constraint->range->lower, constraint->range->upper, constraint->range->step);
    break;
    /* We use GValue so we treat int, fixed or string the same way. */
  case SANE_CONSTRAINT_WORD_LIST:
    constraint->enumeration = NULL;
    count = oid->descriptor->constraint.word_list[0];
    string = " enum = {";
    for (i = 0; i < count; i++) {
      switch (oid->descriptor->type) {
      case SANE_TYPE_INT:
	value = gs_value_new_int ((gint) oid->descriptor->constraint.word_list[i+1]);
	string = g_strdup_printf ("%s %i", string, g_value_get_int (value));
	break;
      case SANE_TYPE_FIXED:
	value = gs_value_new_double ((gdouble) SANE_UNFIX (oid->descriptor->constraint.word_list[i+1]));
	string = g_strdup_printf ("%s %f", string, g_value_get_double (value));
	break;
      default:
	break;
      }
      constraint->enumeration = g_slist_append (constraint->enumeration, value);
    }
    string = g_strdup_printf ("%s}", string);
    g_debug (string);
    break;
  case SANE_CONSTRAINT_STRING_LIST:
    constraint->enumeration = NULL;
    string = " enum\t= {";
    for (i = 0; oid->descriptor->constraint.string_list[i]; i++) {
      value = gs_value_new_string (oid->descriptor->constraint.string_list[i]);
      string = g_strdup_printf ("%s %s", string, g_value_get_string (value));
      constraint->enumeration = g_slist_append (constraint->enumeration, value);
    }
    string = g_strdup_printf ("%s }", string);
    g_debug (string);
    break;
  default:
    break;
  }

  return constraint;
}

/*
 * Gather some SANE informations into flags.
 */
GnomeScannerOptionFlags
gsb_scanner_option_get_flags (GnomeScannerOptionId *oid)
{
  GnomeScannerOptionFlags flags = 0;
  gchar *string = " flags\t= {";

  if (!SANE_OPTION_IS_SETTABLE (oid->descriptor->cap)) {
    flags|= GNOME_SCANNER_OPTION_READONLY;
    string = g_strdup_printf ("%s readonly", string);
  }

  if (!SANE_OPTION_IS_ACTIVE (oid->descriptor->cap)) {
    flags|= GNOME_SCANNER_OPTION_INACTIVE;
    string = g_strdup_printf ("%s inactive", string);
  }

  if (oid->descriptor->cap & SANE_CAP_ADVANCED) {
    flags|= GNOME_SCANNER_OPTION_ADVANCED;
    string = g_strdup_printf ("%s advanced", string);
  }

  if (oid->descriptor->cap & SANE_CAP_AUTOMATIC) {
    flags|= GNOME_SCANNER_OPTION_AUTOMATIC;
    string = g_strdup_printf ("%s automatic", string);
  }

  switch (oid->descriptor->constraint_type) {
  case SANE_CONSTRAINT_NONE:
    break;
  case SANE_CONSTRAINT_RANGE:
    flags|= GNOME_SCANNER_OPTION_RANGE;
    string = g_strdup_printf ("%s range", string);
    break;
  case SANE_CONSTRAINT_WORD_LIST:
  case SANE_CONSTRAINT_STRING_LIST:
    flags|= GNOME_SCANNER_OPTION_ENUMERATION;
    string = g_strdup_printf ("%s enumeration", string);
    break;
  }

  g_debug ("%s }", string);

  return flags;
}

void
gsb_scanner_option_reload (gconstpointer key,
			   GnomeScannerOption *option,
			   GnomeScannerId *sid)
{
  GnomeScannerOptionId *oid = gnome_scanner_option_get_id (option);
  GValue *value;
  GnomeScannerOptionFlags flags;
  GError *error = NULL;

  oid->descriptor = sane_get_option_descriptor (sid->handle, oid->id);

  g_debug ("Reloading option %s", key);
  /* Only the backend should write flags. (That's not very elegant, but). */
  flags = gsb_scanner_option_get_flags (oid);
  option->flags = flags;

  value = gsb_scanner_option_get_value (sid, oid, &error);
  gnome_scanner_option_set_value (option, value);
}



/* ACQUISITION FUNCTIONS */


/* APPLY OPTION FUNCTIONS */

/* Convenient function to apply an option if exists. */
void
gsb_scan_apply_option (GnomeScanContextInfo *info,
		       GnomeScannerId *sid,
		       const gchar* oname,
		       GSBOptionValueFunc value_func,
		       GError **error)
{
  GnomeScannerOption *option;
  GnomeScannerOptionId *oid;
  GValue *value = NULL;

  option = g_hash_table_lookup (sid->hidden, oname);

  if (option) {
    oid = gnome_scanner_option_get_id (option);
    value = (*value_func) (sid, oid, info);
    gsb_scanner_option_set_value (sid, oid, value, error);
    g_free (value);
  }
  else {
    g_debug ("Option %s is unknown for device %s. Skipping.", oname, sid->descriptor->name);
  }
}

void
gsb_scan_apply_test_option (GnomeScannerId *sid,
			    const gchar* oname,
			    GValue *value,
			    GError **error)
{
  GnomeScannerOption *option;
  GnomeScannerOptionId *oid;

  option = g_hash_table_lookup (sid->options, oname);

  if (option) {
    oid = gnome_scanner_option_get_id (option);
    gsb_scanner_option_set_value (sid, oid, value, error);
  }
  else {
    g_debug ("Option %s is unknown for device %s. Skipping.", oname, sid->descriptor->name);
  }
}

/* Return a GValue settable to SANE option value */
GValue*
gsb_scan_apply_option_value_resolution (GnomeScannerId *sid,
					GnomeScannerOptionId *oid,
					GnomeScanContextInfo *info)
{
  GValue *value;
  GnomeScannerOption *option = g_hash_table_lookup (sid->hidden, oid->descriptor->name);
  GType type = gnome_scanner_option_get_value_type (option);

  switch (type) {
  case G_TYPE_INT:
    value = gs_value_new_int ((gint) info->resolution);
    break;
  case G_TYPE_DOUBLE:
    value = gs_value_new_double (info->resolution);
    break;
  default:
    g_debug ("Option resolution of type %s not supported.", g_type_name (type));
    break;
  }

  return value;
}

GValue*
gsb_scan_apply_option_value_area (GnomeScannerId *sid,
				  GnomeScannerOptionId *oid,
				  GnomeScanContextInfo *info)
{
  /* TODO: handle int {tl,br}-{x,y} */
  /* TODO: handle pixel {tl,br}-{x,y} */
  GValue *value;
  const gchar* oname = oid->descriptor->name;

  value = g_new0 (GValue, 1);
  g_value_init (value, G_TYPE_DOUBLE);

  switch (oname[0]) {
  case 't':
    switch (oname[3]) {
    case 'x':
      g_value_set_double (value, info->area->x);
      break;
    case 'y':
      g_value_set_double (value, info->area->y);
      break;
    }
    break;
  case 'b':
    switch (oname[3]) {
    case 'x':
      g_value_set_double (value, info->area->x + info->area->width);
      break;
    case 'y':
      g_value_set_double (value, info->area->y + info->area->height);
      break;
    }
    break;
  }

  return value;
}

GValue*
gsb_scan_apply_option_value_preview (GnomeScannerId *sid,
				     GnomeScannerOptionId *oid,
				     GnomeScanContextInfo *info)
{
  return gs_value_new_boolean (info->preview);
}

GValue*
gsb_scan_apply_option_value_source (GnomeScannerId *sid,
				    GnomeScannerOptionId *oid,
				    GnomeScanContextInfo *info)
{
  return gs_value_new_string (g_hash_table_lookup (sid->sources, g_memdup (&(info->source), sizeof (GnomeScannerSource))));
}

GValue*
gsb_scan_apply_option_value_colorspace	(GnomeScannerId *sid,
					 GnomeScannerOptionId *oid,
					 GnomeScanContextInfo *info)
{
  gchar* value = g_hash_table_lookup (sid->colorspaces, g_memdup (&(info->colorspace), sizeof (GnomeScanColorspace)));
  return value ? gs_value_new_string (value) : NULL;
}

GValue*
gsb_scan_apply_option_value_depth	(GnomeScannerId *sid,
					 GnomeScannerOptionId *oid,
					 GnomeScanContextInfo *info)
{
  return gs_value_new_int (info->depth);
}

/*
 * Apply scan option to sane device.
 */
void
gsb_scan_apply_context (GnomeScanContextInfo *info,
			GError **error)
{
  GnomeScannerOption *option;
  GnomeScannerId *sid;
  GnomeScannerOptionId *oid;
  GValue *value;

  sid = gnome_scanner_get_id (info->scanner);
  g_debug ("Applying scan options to %s", sid->descriptor->name);

  gsb_scan_apply_option (info, sid, "resolution", (GSBOptionValueFunc) gsb_scan_apply_option_value_resolution, error);
  gsb_scan_apply_option (info, sid, "tl-x", (GSBOptionValueFunc) gsb_scan_apply_option_value_area, error);
  gsb_scan_apply_option (info, sid, "tl-y", (GSBOptionValueFunc) gsb_scan_apply_option_value_area, error);
  gsb_scan_apply_option (info, sid, "br-x", (GSBOptionValueFunc) gsb_scan_apply_option_value_area, error);
  gsb_scan_apply_option (info, sid, "br-y", (GSBOptionValueFunc) gsb_scan_apply_option_value_area, error);
  gsb_scan_apply_option (info, sid, "preview", (GSBOptionValueFunc) gsb_scan_apply_option_value_preview, error);
  gsb_scan_apply_option (info, sid, "source", (GSBOptionValueFunc) gsb_scan_apply_option_value_source, error);
  gsb_scan_apply_option (info, sid, "mode", (GSBOptionValueFunc) gsb_scan_apply_option_value_colorspace, error);
  gsb_scan_apply_option (info, sid, "depth", (GSBOptionValueFunc) gsb_scan_apply_option_value_depth, error);
#if	!ENABLE_PROBE
  gsb_scan_apply_test_option (sid, "test-picture", gs_value_new_string ("Color pattern"), error);
  gsb_scan_apply_test_option (sid, "three-pass", gs_value_new_boolean (TRUE), error);
  gsb_scan_apply_test_option (sid, "three-pass-order", gs_value_new_string ("GBR"), error);
#endif
}

/*
 * Compute forecast
 */
GnomeScanForecast*
gsb_scan_forecast (GnomeScanContextInfo *info,
		   GError **error)
{
  SANE_Status status;
  SANE_Parameters parameters;
  GnomeScanForecast *forecast;
  GnomeScannerId *sid;

  sid = gnome_scanner_get_id (info->scanner);
  status = sane_get_parameters (sid->handle, &parameters);
  if (!(*error = gsb_get_error (status))) {
    forecast = g_new0 (GnomeScanForecast, 1);
    forecast->geometry.width = (gdouble) parameters.pixels_per_line;
    forecast->geometry.height = (gdouble) parameters.lines;
    forecast->size = parameters.lines * parameters.bytes_per_line * (parameters.format > SANE_FRAME_RGB ? 3 : 1);
    forecast->depth = parameters.depth;
    forecast->rowstride = parameters.bytes_per_line;
    g_debug ("Forecast %fx%f image of %iB", forecast->geometry.width, forecast->geometry.height, forecast->size);
  }

  return forecast;
}


/* ACQUISITION */

/* acquire one image */
GnomeScanResult*
gsb_acquire_one (GnomeScanBackend *backend,
		 GnomeScannerId *sid,
		 GnomeScanContextInfo *info,
		 GError **error)
{
  SANE_Status status;
  SANE_Parameters parameters;
  GnomeScannerOption *option;
  GValue *value;
  GnomeScanForecast *fc;
  GnomeScanResult* result;
  guchar *buffer = NULL, *pixels = NULL;
  gint frame_count, count, len;
  gsize size, chunk_size;
  gboolean first_frame = TRUE;

  result = NULL;
  count = 0;

  do {				/* Loop each frame */
    status =  sane_start (sid->handle);
    if (*error = gsb_get_error (status)) {
      /* we should handle properly INVAL and other status like that */
      g_debug ("Failed to start scan from %s: %s", sid->descriptor->name, (*error)->message);
    }
    else {

      /* We must forcast after te first sane_start ... */
      if (first_frame) {
	fc = gsb_scan_forecast (info, error);
	if (!*error) {
	  size = (fc->geometry.width * fc->geometry.height * 3 * 8) / 8;
	  g_debug ("Final pixbuf size is %iB", size);
	  pixels = g_try_malloc0 (size);
	  chunk_size = 32768;
	  buffer = g_malloc0 (chunk_size);
	}
	else {
	  return NULL;
	}
	first_frame = FALSE;
      }	
      sane_get_parameters (sid->handle, &parameters);
      frame_count = 0;

      do {			/* Loop each chunk */
	/* Real frame acquisition. */
	status = sane_read (sid->handle, buffer, chunk_size, &len);

	if (status == SANE_STATUS_GOOD) {
	  gsb_convert_data_to_rgb (parameters.format, fc->depth, buffer, pixels, frame_count, len);
	  frame_count+= len;
	  count+= len;
	  g_debug ("Received %iB; (%i of %iB = %f%%)", len, count, size, 100. * (gdouble) count / (gdouble) size);
	  g_signal_emit_by_name (backend, "data-received", count);
	}
      } while (status == SANE_STATUS_GOOD && count <= fc->size);

      if (status != SANE_STATUS_CANCELLED) {
	/* other errors are jammed, no docs, cover open, etc. */
	*error = gsb_get_error (status);
      }
    }
  } while (!parameters.last_frame
	   && parameters.format != SANE_FRAME_RGB && parameters.format != SANE_FRAME_GRAY);

  if (status == SANE_STATUS_EOF || status == SANE_STATUS_GOOD) {
    /* Store result */
    result = g_new0 (GnomeScanResult, 1);

    g_debug ("Retrieving final resolution");
    /* we retrieve the actual resolution */
      
    option = g_hash_table_lookup (sid->hidden, "resolution");
    if (option) {
      value = gsb_scanner_option_get_value (sid, gnome_scanner_option_get_id (option), error);
      switch (G_VALUE_TYPE (value)) {
      case G_TYPE_INT:
        result->resolution = (gdouble) g_value_get_int (value);
	break;
      case G_TYPE_DOUBLE:
        result->resolution = g_value_get_double (value);
	break;
      }
    }
    /* or use the asked one. */
    else {
	result->resolution = info->resolution;
    }

  g_debug ("Final resolution is %f",
	   result->resolution);

  result->preview = info->preview;
  result->image = gdk_pixbuf_new_from_data (pixels, GDK_COLORSPACE_RGB,
					    FALSE, /* has_alpha */
					    8, /* bit_per_sample */
					    fc->geometry.width, fc->geometry.height,
					    fc->geometry.width * 3, /* rowstride */
					    (GdkPixbufDestroyNotify) g_free, NULL);
  }
    
  if (buffer) {
    g_free (buffer);

    /*
     * Some backend NEED to cancel an acquisition even if terminated
     * (e.g. hpaio).
     */
    sane_cancel (sid->handle);
  }

  return result;
}

void
gsb_convert_data_to_rgb (SANE_Frame format,
			 gint depth,
			 guchar *data,
			 guchar *pixels,
			 gint offset,
			 gint length)
{
  int i;

  switch (format) {
  case SANE_FRAME_RGB:
    memcpy (pixels+offset, data, length);
    break;
  case SANE_FRAME_GRAY:
    offset*=3;
    for (i = 0 ; i < length ; i++) {
      *(pixels+offset+(3*i)+0) = data[i]; /* RED */
      *(pixels+offset+(3*i)+1) = data[i]; /* GREEN */
      *(pixels+offset+(3*i)+2) = data[i]; /* BLUE */
    }
    break;
  case SANE_FRAME_RED:
  case SANE_FRAME_GREEN:
  case SANE_FRAME_BLUE:
    offset*=3;
    for (i = 0; i < length ; i++) {
      /* format = 2, 3 or 4. So format-2 = 0, 1 or 2 */
      *(pixels+offset+(3*i)+(format-2)) = data[i];
    }
    break;
  default:
    g_debug ("Frame format (%i) is not supported", format);
    break;
  }
}

/* MISCELLANEOUS FUNCTIONS */

/*
 * Return an error or NULL considering a SANE status.
 */
GError*
gsb_get_error (SANE_Status status)
{
  const gchar *string;
  GError *error = NULL;

  /* translation ? */

  if (status != SANE_STATUS_GOOD) {
    string = sane_strstatus (status);
    error = g_error_new_literal (GNOME_SCAN_BACKEND_ERROR,
				 GNOME_SCAN_BACKEND_ERROR_SANE_ERROR,
				 string);
    g_debug (string);
  }

  return error;
}
