/* $Id: arch_arp.c,v 1.14 2012-02-10 14:40:15 vrsieh Exp $ 
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef INCLUDE

/* Hack to get ntohl etc working */
#define _LINUX_BYTEORDER_GENERIC_H	1

#include "config.h"

#if defined(OPENBSD) || defined(DARWIN)
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in_systm.h>
#include <net/if.h>
#include <termios.h>
/*
 * IEEE 802.3 Ethernet magic constants.  The frame sizes omit the preamble
 * and FCS/CRC (frame check sequence).
 * Taken from Linux /usr/include/linux/if_ether.h
 */

#define ETH_ALEN        6	/* Octets in one ethernet addr   */
#define ETH_HLEN        14	/* Total octets in header.       */
#define ETH_FRAME_LEN   1514	/* Max. octets in frame sans FCS */

#else
#include <net/ethernet.h>
#endif

#include <net/if_arp.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UTIL_H
#include <util.h>
#endif

#ifndef ETHERTYPE_IPV6
#define ETHERTYPE_IPV6 0x86dd
#endif

#endif /* INCLUDE */

#ifdef STATE

#define ARP_CACHE_HASH 16

struct {
	struct arp_host {
		unsigned char mac[ETH_ALEN];
		uint32_t ip;
		struct arp_host *next;
	} *arp_cache_static[ARP_CACHE_HASH],
	  *arp_cache_dynamic[ARP_CACHE_HASH];
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

/** remember ip/mac accociation */
static int
NAME_(_add_mac)(
	struct cpssp *cpssp,
	struct arp_host **table,
	uint32_t ip,
	const unsigned char *mac
)
{
	struct arp_host *h;

	for (h = table[ip % ARP_CACHE_HASH]; h; h = h->next) {
		if (h->ip != ip)
			continue;

		/* already there, refresh it */
		memcpy(h->mac, mac, ETH_ALEN);
		return 0;
	}
	/* create new entry */
	h = (struct arp_host*) shm_alloc(sizeof(struct arp_host));
	h->ip = ip;
	memcpy(h->mac, mac, ETH_ALEN);
	h->next = table[ip % ARP_CACHE_HASH];
	table[ip % ARP_CACHE_HASH] = h;
	return 0;
}

/** lookup mac address by ip */
static int
NAME_(_ip_to_mac)(
	struct cpssp *cpssp,
	struct arp_host **table,
	uint32_t ip,
	unsigned char *mac
)
{
	struct arp_host *h;

	for (h = table[ip % ARP_CACHE_HASH]; h; h = h->next) {
		if (h->ip != ip)
			continue;

		memcpy(mac, h->mac, ETH_ALEN);
		return 0;
	}

	errno = ENOENT;
	return -1;
}

static int
NAME_(ip_to_mac)(struct cpssp *cpssp, uint32_t ip, unsigned char *mac)
{
	/* lookup mac address, use default gw if we can't find one */
	return NAME_(_ip_to_mac)(cpssp, cpssp->NAME.arp_cache_static,
			ip, mac)
	    && NAME_(_ip_to_mac)(cpssp, cpssp->NAME.arp_cache_dynamic,
			ip, mac);
}

static int
NAME_(resolve)(struct cpssp *cpssp, uint32_t ip, unsigned char *mac)
{
	int ret;

	ret = NAME_(_ip_to_mac)(cpssp, cpssp->NAME.arp_cache_static,
			ip, mac);
	return ret;
}

/** handle arp request */
static void
NAME_(recv)(struct cpssp *cpssp, const unsigned char *_buf, int len)
{
	char buf[1024];
	struct ether_header *ethhdr = (struct ether_header*) buf;
	struct ether_arp *arphdr = (struct ether_arp*)
				(buf + sizeof(struct ether_header));
	uint32_t ip;
	uint32_t tmp;

	memcpy(buf, _buf, len);

#if 0
	/* check */
	errno = EINVAL; /* just in case we fail */
	if( ntohs(arphdr->arp_op) != ARPOP_REQUEST ) return -1;
	if( ntohs(arphdr->arp_hrd) != ARPHRD_ETHER ) return -1;
	//if( arphdr->arp_pro != XXX ) return -1;
	if( arphdr->arp_hln != 6 ) return -1;
	if( arphdr->arp_pln != 4 ) return -1;
#endif

	memcpy(&tmp, &arphdr->arp_tpa, sizeof(tmp));
	ip = ntohl(tmp);

	/* transform request -> reply */
	memcpy(arphdr->arp_tha, arphdr->arp_sha,
			arphdr->arp_hln+arphdr->arp_pln);
	tmp = htonl(ip);
	memcpy(&arphdr->arp_spa, &tmp, sizeof(arphdr->arp_spa));
	arphdr->arp_op = htons(ARPOP_REPLY);
	if (NAME_(resolve)(cpssp, ip, arphdr->arp_sha) != 0) {
		return;
	}

	memcpy(ethhdr->ether_dhost, ethhdr->ether_shost, ETH_ALEN);
	memcpy(ethhdr->ether_shost, arphdr->arp_sha, ETH_ALEN);

	/* send the reply for this request */
	errno = 0;
	NAME_(to_eth)(cpssp, buf, len);
}

/** cache source addr of incoming packets */
static int
NAME_(recv_eth)(struct cpssp *cpssp, const unsigned char *buf, int len)
{
	const struct ether_header *ethhdr = (const struct ether_header *) buf;
	const struct ip *iphdr = (const struct ip*) (buf + sizeof(struct ether_header));

	if(ntohs(ethhdr->ether_type) != ETHERTYPE_IP) return 0;

	return NAME_(_add_mac)(cpssp, cpssp->NAME.arp_cache_dynamic,
			ntohl(iphdr->ip_src.s_addr), ethhdr->ether_shost);
}

static void
NAME_(from_route_ip)(
	struct cpssp *cpssp,
	const unsigned char *ip_buf,
	unsigned int ip_len
)
{
	char buf[2048];
	unsigned int len;
	struct ether_header *ethhdr = (struct ether_header*) buf;
	struct ip *iphdr = (struct ip*) (buf + sizeof(struct ether_header));

	assert(sizeof(struct ether_header) + ip_len < sizeof(buf));

	memcpy(iphdr, ip_buf, ip_len);
	len = sizeof(struct ether_header) + ip_len;

	/*
	 * Initialize eth frame.
	 */
	/* Destination MAC */
	if (iphdr->ip_dst.s_addr == 0xffffffff) {
		/* Broadcast */
		ethhdr->ether_dhost[0] = 0xff;
		ethhdr->ether_dhost[1] = 0xff;
		ethhdr->ether_dhost[2] = 0xff;
		ethhdr->ether_dhost[3] = 0xff;
		ethhdr->ether_dhost[4] = 0xff;
		ethhdr->ether_dhost[5] = 0xff;
	} else {
		NAME_(ip_to_mac)(cpssp, ntohl(iphdr->ip_dst.s_addr),
				ethhdr->ether_dhost);
	}

	/* Source MAC */
	if (NAME_(ip_to_mac)(cpssp, ntohl(iphdr->ip_src.s_addr),
				ethhdr->ether_shost) < 0) {
		/* last try: 0.0.0.0 */
		NAME_(ip_to_mac)(cpssp, 0x00000000, ethhdr->ether_shost);
	}
	ethhdr->ether_type = htons(ETHERTYPE_IP);

	NAME_(to_eth)(cpssp, buf, len);
}

static void
NAME_(from_eth)(struct cpssp *cpssp, const unsigned char *buf, unsigned int len)
{
	const struct ether_header *hdr = (const struct ether_header*) buf;

	/* parse eth header */
	NAME_(recv_eth)(cpssp, buf, len);

	/* delegate according to protocol */
	switch (ntohs(hdr->ether_type)) {
	case ETHERTYPE_ARP:
		NAME_(recv)(cpssp, buf, len);
		break;
	case ETHERTYPE_IP:
		route_ip_from_arp(cpssp, buf + ETH_HLEN, len - ETH_HLEN);
		break;
	case ETHERTYPE_IPV6:
		/* not supported yet */
		break;
	default:
		fprintf(stderr, "arp_from_eth: unknown ether_type=0x%x\n",
				ntohs(hdr->ether_type));
		break;
	}
}

/* methods to configure the arp list */

static void
NAME_(create)(struct cpssp *cpssp, const char *mac, uint32_t ip)
{
	unsigned int i;

	for (i = 0; i < sizeof(cpssp->NAME.arp_cache_static) / sizeof(cpssp->NAME.arp_cache_static[0]); i++) {
		cpssp->NAME.arp_cache_static[i] = NULL;
	}
	for (i = 0; i < sizeof(cpssp->NAME.arp_cache_dynamic) / sizeof(cpssp->NAME.arp_cache_dynamic[0]); i++) {
		cpssp->NAME.arp_cache_dynamic[i] = NULL;
	}

	NAME_(_add_mac)(cpssp, cpssp->NAME.arp_cache_static, ip, mac);
	NAME_(_add_mac)(cpssp, cpssp->NAME.arp_cache_static, 0x00000000, mac);
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
