/* $Id: cim_telephone.c,v 1.12 2009-01-27 17:06:40 potyra Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <inttypes.h>
#include <arpa/inet.h>

#include "cim_telephone.h"
#include "bridge.h"

/* packet structure, that gets transfered via cim. */
struct sig_telephone_packet {
	/** type of message */
	enum sig_telephone_protocol protocol;
	/** data or number */
	union {
		/** number to dial */
		uint32_t number;
		/** data */
		uint8_t c;
	} data;
};

static void
cim_telephone_recv_data(void *_css, uint8_t data)
{
	struct cim_telephone *css = (struct cim_telephone *) _css;
	static struct sig_telephone_packet p;

	p.protocol = SIG_TELE_DATA;
	p.data.c = data;

	cim_send(&css->bridge, &p, sizeof(p));
}

static void
cim_telephone_recv_dial(void *_css, uint32_t number)
{
	struct cim_telephone *css = (struct cim_telephone *) _css;
	static struct sig_telephone_packet p;

	p.protocol = SIG_TELE_DIAL;
	p.data.number = htonl(number);

	cim_send(&css->bridge, &p, sizeof(p));
}

static void
cim_telephone_recv_ctrl(void *_css, enum sig_telephone_protocol proto)
{
	struct cim_telephone *css = (struct cim_telephone *) _css;
	static struct sig_telephone_packet p;

	p.protocol = proto;

	cim_send(&css->bridge, &p, sizeof(p));
}

static void
cim_telephone_interrupt(void *_css, void *_buf, unsigned int bufsize)
{
	struct cim_telephone *css = (struct cim_telephone *) _css;
	struct sig_telephone_packet *buf = 
		(struct sig_telephone_packet *) _buf;

	assert(bufsize = sizeof(*buf));

	switch(buf->protocol) {
	case SIG_TELE_DATA:
		sig_telephone_send_data(css->sig_telephone, css,
				buf->data.c);
		break;
	case SIG_TELE_DIAL:
		sig_telephone_dial(css->sig_telephone, css,
				ntohl(buf->data.number));
		break;
	default:
		sig_telephone_send_ctrl(css->sig_telephone, css, 
				buf->protocol);
		break;
	}
}

void
cim_telephone_init(struct cim_telephone *css, 
		   struct sig_telephone *sig_telephone)
{
	static const struct sig_telephone_funcs funcs = {
		.recv_data = cim_telephone_recv_data,
		.recv_ctrl = cim_telephone_recv_ctrl,
		.recv_dial = cim_telephone_recv_dial
	};

	cim_connect(&css->bridge, cim_telephone_interrupt, css);

	css->sig_telephone = sig_telephone;
	sig_telephone_connect(sig_telephone, css, &funcs);
}

void
cim_telephone_create(struct cim_telephone *css)
{
	cim_create(&css->bridge);
}

void
cim_telephone_destroy(struct cim_telephone *css)
{
	cim_destroy(&css->bridge);
}

